import React, { useEffect, useMemo, useRef, useState, } from 'react';
import { ExpressionType } from './utils/Expression';
import { convertConditionToExpression, getExpressionCondition, sanitizeExpression, validateExpression, } from './utils/utils';
import { GroupBuilder } from './GroupBuilder';
import { BLANK_MARKER } from './utils/Config';
import { useDebouncedValue } from '../../hooks';
import { ConditionBuilderVariableType, } from './ConditionBuilderVariable';
export const ConditionBuilder = (props) => {
    const { onConditionChange, onSanitizedConditionChanged, condition: conditionProp, variables, onDelete, } = props;
    const firstTime = useRef(true);
    const actualVariables = useMemo(() => {
        return variables
            .filter((it) => it.variableType !== ConditionBuilderVariableType.IMAGE &&
            it.variableType !== ConditionBuilderVariableType.VIDEO)
            .map((it) => {
            return Object.assign(Object.assign({}, it), { variableName: '$' + it.variableName });
        });
    }, [variables]);
    const [actualExpression, setExpression] = useState(() => {
        return convertConditionToExpression(conditionProp, actualVariables);
    });
    const expression = useDebouncedValue(actualExpression, 500);
    const condition = useMemo(() => {
        return getExpressionCondition(expression);
    }, [expression]);
    useEffect(() => {
        if (conditionProp !== condition && !firstTime.current) {
            const exp = convertConditionToExpression(conditionProp, actualVariables);
            setExpression(exp);
        }
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [conditionProp]);
    const isExpressionValid = useMemo(() => {
        return !condition.includes(BLANK_MARKER) && validateExpression(expression);
    }, [condition, expression]);
    useEffect(() => {
        if (conditionProp !== condition) {
            onConditionChange(condition, isExpressionValid);
        }
        if (onSanitizedConditionChanged) {
            const conditionHTML = sanitizeExpression(expression, actualVariables);
            onSanitizedConditionChanged(conditionHTML);
        }
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [condition]);
    useEffect(() => {
        onConditionChange(condition, isExpressionValid);
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [isExpressionValid]);
    useEffect(() => {
        firstTime.current = false;
    }, []);
    function onExpressionChange(expression) {
        if (!expression || expression.type !== ExpressionType.GROUP) {
            return;
        }
        setExpression(expression);
    }
    return (React.createElement("div", { className: 'flex flex-col gap-4 flex-grow' },
        variables.length === 0 && (React.createElement("span", null, "No variables available to build conditional")),
        variables.length > 0 && (React.createElement(GroupBuilder, { expression: actualExpression, onChange: onExpressionChange, isDeleteEnabled: onDelete ? true : false, deleteGroup: onDelete, variables: actualVariables }))));
};
export { VariableType } from './utils/Config';
export { convertConditionToHTML } from './utils/utils';
//# sourceMappingURL=ConditionBuilder.js.map