import { getLiteralStringValue } from './utils';
import { ExpressionType } from './Expression';
import { Operator } from './Config';
import { CONTEXT_VARIABLE_REGEX } from '../../../constants';
import { getLiteralRulePart, getOperatorRulePart, getVariableRulePart, } from './rulePartUtils';
import { ConditionBuilderVariableType, } from '../ConditionBuilderVariable';
const VAR_REGEX_SRC = CONTEXT_VARIABLE_REGEX.source;
function getVariable(variables, variableName) {
    return variables.find((it) => it.variableName === variableName);
}
function getRule(variables, variableName, operator, config, literal) {
    const variable = getVariable(variables, variableName);
    if (!variable)
        return null;
    return {
        type: ExpressionType.RULE,
        parts: [
            getVariableRulePart(variableName),
            getOperatorRulePart(operator, variable),
            getLiteralRulePart(config, variable, literal),
        ],
    };
}
export const OperatorConfigMap = {
    [Operator.BELONGS]: {
        getRule(condition, variables) {
            const regexp = new RegExp(`^${VAR_REGEX_SRC} % \\[ #[\\-_0-9a-zA-Z @.]+# (, #[\\-_0-9a-zA-Z @.]+#)* ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' % ');
                const variableName = split[0];
                const operator = Operator.BELONGS;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'is in',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `${variableName} % ${getLiteralStringValue(rhs)}`;
        },
        supportedTypes: [ConditionBuilderVariableType.CATEGORY],
        getLiteralType(variableType) {
            if (variableType === ConditionBuilderVariableType.CATEGORY) {
                return ConditionBuilderVariableType.LIST_STRING;
            }
            return ConditionBuilderVariableType.STRING;
        },
    },
    [Operator.NOT_BELONGS]: {
        getRule(condition, variables) {
            const notMatch = /^! \( .+ \)$/;
            const isNot = condition.match(notMatch);
            if (!isNot)
                return null;
            condition = condition.slice(4, condition.length - 2);
            const regexp = new RegExp(`^${VAR_REGEX_SRC} % \\[ #[\\-_0-9a-zA-Z @.]+# (, #[\\-_0-9a-zA-Z @.]+#)* ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' % ');
                const variableName = split[0];
                const operator = Operator.NOT_BELONGS;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'is not in',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `! ( ${variableName} % ${getLiteralStringValue(rhs)} )`;
        },
        supportedTypes: [ConditionBuilderVariableType.CATEGORY],
        getLiteralType(variableType) {
            if (variableType === ConditionBuilderVariableType.CATEGORY) {
                return ConditionBuilderVariableType.LIST_STRING;
            }
            return ConditionBuilderVariableType.STRING;
        },
    },
    [Operator.BELONGS_RANGE]: {
        getRule(condition, variables) {
            const regexp = new RegExp(`^${VAR_REGEX_SRC} % \\[ \\d+(\\.\\d+)? - \\d+(\\.\\d+)? ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' % ');
                const variableName = split[0];
                const operator = Operator.BELONGS_RANGE;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'is in range',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `${variableName} % ${getLiteralStringValue(rhs)}`;
        },
        supportedTypes: [
            ConditionBuilderVariableType.INTEGER,
            ConditionBuilderVariableType.REAL,
            ConditionBuilderVariableType.DATE,
            ConditionBuilderVariableType.DATE_TIME,
            ConditionBuilderVariableType.TIME,
        ],
        getLiteralType(variableType) {
            if (variableType === ConditionBuilderVariableType.TIME ||
                variableType === ConditionBuilderVariableType.DATE_TIME ||
                variableType === ConditionBuilderVariableType.DATE) {
                return 'DATE_RANGE';
            }
            return 'RANGE';
        },
    },
    [Operator.NOT_BELONGS_RANGE]: {
        getRule(condition, variables) {
            const notMatch = /^! \( .+ \)$/;
            const isNot = condition.match(notMatch);
            if (!isNot)
                return null;
            condition = condition.slice(4, condition.length - 2);
            const regexp = new RegExp(`^${VAR_REGEX_SRC} % \\[ \\d+(\\.\\d+)? - \\d+(\\.\\d+)? ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' % ');
                const variableName = split[0];
                const operator = Operator.NOT_BELONGS_RANGE;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'is not in range',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `! ( ${variableName} % ${getLiteralStringValue(rhs)} )`;
        },
        supportedTypes: [
            ConditionBuilderVariableType.INTEGER,
            ConditionBuilderVariableType.REAL,
            ConditionBuilderVariableType.DATE,
            ConditionBuilderVariableType.DATE_TIME,
            ConditionBuilderVariableType.TIME,
        ],
        getLiteralType(variableType) {
            if (variableType === ConditionBuilderVariableType.TIME ||
                variableType === ConditionBuilderVariableType.DATE_TIME ||
                variableType === ConditionBuilderVariableType.DATE) {
                return 'DATE_RANGE';
            }
            return 'RANGE';
        },
    },
    [Operator.EQUALS]: {
        getRule(condition, variables) {
            const regexp = new RegExp(`^${VAR_REGEX_SRC}  %  \\[ .+ ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split('  %  ');
                const variableName = split[0];
                const operator = Operator.EQUALS;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: '=',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `${variableName}  %  [ ${getLiteralStringValue(rhs)} ]`;
        },
        supportedTypes: [
            ConditionBuilderVariableType.STRING,
            ConditionBuilderVariableType.BOOLEAN,
            ConditionBuilderVariableType.CATEGORY,
            ConditionBuilderVariableType.INTEGER,
            ConditionBuilderVariableType.REAL,
            ConditionBuilderVariableType.DATE,
            ConditionBuilderVariableType.TIME,
            ConditionBuilderVariableType.DATE_TIME,
            ConditionBuilderVariableType.IMAGE,
            ConditionBuilderVariableType.VIDEO,
        ],
        getLiteralType(variableType) {
            return variableType;
        },
    },
    [Operator.NOT_EQUALS]: {
        getRule(condition, variables) {
            const notMatch = /^! \( .+ \)$/;
            const isNot = condition.match(notMatch);
            if (!isNot)
                return null;
            condition = condition.slice(4, condition.length - 2);
            const regexp = new RegExp(`^${VAR_REGEX_SRC}  %  \\[ .+ ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split('  %  ');
                const variableName = split[0];
                const operator = Operator.NOT_EQUALS;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: '!=',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `! ( ${variableName}  %  [ ${getLiteralStringValue(rhs)} ] )`;
        },
        supportedTypes: [
            ConditionBuilderVariableType.STRING,
            ConditionBuilderVariableType.BOOLEAN,
            ConditionBuilderVariableType.CATEGORY,
            ConditionBuilderVariableType.INTEGER,
            ConditionBuilderVariableType.REAL,
            ConditionBuilderVariableType.DATE,
            ConditionBuilderVariableType.TIME,
            ConditionBuilderVariableType.DATE_TIME,
            ConditionBuilderVariableType.IMAGE,
            ConditionBuilderVariableType.VIDEO,
        ],
        getLiteralType(variableType) {
            return variableType;
        },
    },
    [Operator.STARTSWITH]: {
        getRule(condition, variables) {
            const regexp = new RegExp(`^${VAR_REGEX_SRC} \\|% #[\\-_0-9a-zA-Z @.]+#`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' |% ');
                const variableName = split[0];
                const operator = Operator.STARTSWITH;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'starts with',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `${variableName} |% ${getLiteralStringValue(rhs)}`;
        },
        supportedTypes: [ConditionBuilderVariableType.STRING],
        getLiteralType() {
            return ConditionBuilderVariableType.STRING;
        },
    },
    [Operator.NOT_STARTSWITH]: {
        getRule(condition, variables) {
            const notMatch = /^! \( .+ \)$/;
            const isNot = condition.match(notMatch);
            if (!isNot)
                return null;
            condition = condition.slice(4, condition.length - 2);
            const regexp = new RegExp(`^${VAR_REGEX_SRC} \\|% #[\\-_0-9a-zA-Z @.]+#`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' |% ');
                const variableName = split[0];
                const operator = Operator.NOT_STARTSWITH;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'does not start with',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `! ( ${variableName} |% ${getLiteralStringValue(rhs)} )`;
        },
        supportedTypes: [ConditionBuilderVariableType.STRING],
        getLiteralType() {
            return ConditionBuilderVariableType.STRING;
        },
    },
    [Operator.ENDSWITH]: {
        getRule(condition, variables) {
            const regexp = new RegExp(`^${VAR_REGEX_SRC} %\\| #[\\-_0-9a-zA-Z @.]+#`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' %| ');
                const variableName = split[0];
                const operator = Operator.ENDSWITH;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'ends with',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `${variableName} %| ${getLiteralStringValue(rhs)}`;
        },
        supportedTypes: [ConditionBuilderVariableType.STRING],
        getLiteralType() {
            return ConditionBuilderVariableType.STRING;
        },
    },
    [Operator.NOT_ENDSWITH]: {
        getRule(condition, variables) {
            const notMatch = /^! \( .+ \)$/;
            const isNot = condition.match(notMatch);
            if (!isNot)
                return null;
            condition = condition.slice(4, condition.length - 2);
            const regexp = new RegExp(`^${VAR_REGEX_SRC} %\\| #[\\-_0-9a-zA-Z @.]+#`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' %| ');
                const variableName = split[0];
                const operator = Operator.NOT_ENDSWITH;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'does not end with',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `! ( ${variableName} %| ${getLiteralStringValue(rhs)} )`;
        },
        supportedTypes: [ConditionBuilderVariableType.STRING],
        getLiteralType() {
            return ConditionBuilderVariableType.STRING;
        },
    },
    [Operator.EXACTMATCH]: {
        getRule(condition, variables) {
            const regexp = new RegExp(`^${VAR_REGEX_SRC} == \\[ .+ ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' == ');
                const variableName = split[0];
                const operator = Operator.EXACTMATCH;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'is',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `${variableName} == ${getLiteralStringValue(rhs)}`;
        },
        supportedTypes: [
            ConditionBuilderVariableType.LIST_STRING,
            ConditionBuilderVariableType.LIST_INTEGER,
            ConditionBuilderVariableType.LIST_REAL,
            ConditionBuilderVariableType.CATEGORY_MULTIPLE,
        ],
        getLiteralType(variableType) {
            if (variableType === ConditionBuilderVariableType.CATEGORY_MULTIPLE) {
                return ConditionBuilderVariableType.LIST_STRING;
            }
            return variableType;
        },
    },
    [Operator.NOT_EXACTMATCH]: {
        getRule(condition, variables) {
            const notMatch = /^! \( .+ \)$/;
            const isNot = condition.match(notMatch);
            if (!isNot)
                return null;
            condition = condition.slice(4, condition.length - 2);
            const regexp = new RegExp(`^${VAR_REGEX_SRC} == \\[ .+ ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' == ');
                const variableName = split[0];
                const operator = Operator.NOT_EXACTMATCH;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'is not',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `! ( ${variableName} == ${getLiteralStringValue(rhs)} )`;
        },
        supportedTypes: [
            ConditionBuilderVariableType.LIST_STRING,
            ConditionBuilderVariableType.LIST_INTEGER,
            ConditionBuilderVariableType.LIST_REAL,
            ConditionBuilderVariableType.CATEGORY_MULTIPLE,
        ],
        getLiteralType(variableType) {
            if (variableType === ConditionBuilderVariableType.CATEGORY_MULTIPLE) {
                return ConditionBuilderVariableType.LIST_STRING;
            }
            return variableType;
        },
    },
    [Operator.ALLMATCH]: {
        getRule(condition, variables) {
            const regexp = new RegExp(`^${VAR_REGEX_SRC} \\|> \\[ .+ ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' |> ');
                const variableName = split[0];
                const operator = Operator.ALLMATCH;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'are all in range',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `${variableName} |> ${getLiteralStringValue(rhs)}`;
        },
        supportedTypes: [
            ConditionBuilderVariableType.LIST_INTEGER,
            ConditionBuilderVariableType.LIST_REAL,
        ],
        getLiteralType() {
            return 'RANGE';
        },
    },
    [Operator.ANYMATCH]: {
        getRule(condition, variables) {
            const regexp = new RegExp(`^${VAR_REGEX_SRC} <\\| \\[ .+ ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' <| ');
                const variableName = split[0];
                const operator = Operator.ANYMATCH;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'has some in range',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `${variableName} <| ${getLiteralStringValue(rhs)}`;
        },
        supportedTypes: [
            ConditionBuilderVariableType.LIST_INTEGER,
            ConditionBuilderVariableType.LIST_REAL,
        ],
        getLiteralType() {
            return 'RANGE';
        },
    },
    [Operator.LESSTHAN]: {
        getRule(condition, variables) {
            const regexp = new RegExp(`^${VAR_REGEX_SRC} % \\[ \\* - .+ ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' % ');
                const variableName = split[0];
                const operator = Operator.LESSTHAN;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: '<',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `${variableName} % [ * - ${getLiteralStringValue(rhs)} ]`;
        },
        supportedTypes: [
            ConditionBuilderVariableType.INTEGER,
            ConditionBuilderVariableType.REAL,
            ConditionBuilderVariableType.DATE_TIME,
            ConditionBuilderVariableType.DATE,
            ConditionBuilderVariableType.TIME,
        ],
        getLiteralType(variableType) {
            return variableType;
        },
    },
    [Operator.GREATERTHAN_EQUALS]: {
        getRule(condition, variables) {
            const notMatch = /^! \( .+ \)$/;
            const isNot = condition.match(notMatch);
            if (!isNot)
                return null;
            condition = condition.slice(4, condition.length - 2);
            const regexp = new RegExp(`^${VAR_REGEX_SRC} % \\[ \\* - .+ ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' % ');
                const variableName = split[0];
                const operator = Operator.GREATERTHAN_EQUALS;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: '>=',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `! ( ${variableName} % [ * - ${getLiteralStringValue(rhs)} ] )`;
        },
        supportedTypes: [
            ConditionBuilderVariableType.INTEGER,
            ConditionBuilderVariableType.REAL,
            ConditionBuilderVariableType.DATE_TIME,
            ConditionBuilderVariableType.DATE,
            ConditionBuilderVariableType.TIME,
        ],
        getLiteralType(variableType) {
            return variableType;
        },
    },
    [Operator.GREATERTHAN]: {
        getRule(condition, variables) {
            const regexp = new RegExp(`^${VAR_REGEX_SRC} % \\[ .+ - \\* ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' % ');
                const variableName = split[0];
                const operator = Operator.LESSTHAN;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: '>',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `${variableName} % [ ${getLiteralStringValue(rhs)} - * ]`;
        },
        supportedTypes: [
            ConditionBuilderVariableType.INTEGER,
            ConditionBuilderVariableType.REAL,
            ConditionBuilderVariableType.DATE_TIME,
            ConditionBuilderVariableType.DATE,
            ConditionBuilderVariableType.TIME,
        ],
        getLiteralType(variableType) {
            return variableType;
        },
    },
    [Operator.LESSTHAN_EQUALS]: {
        getRule(condition, variables) {
            const notMatch = /^! \( .+ \)$/;
            const isNot = condition.match(notMatch);
            if (!isNot)
                return null;
            condition = condition.slice(4, condition.length - 2);
            const regexp = new RegExp(`^${VAR_REGEX_SRC} % \\[ .+ - \\* ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' % ');
                const variableName = split[0];
                const operator = Operator.LESSTHAN_EQUALS;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: '<=',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `! ( ${variableName} % [ ${getLiteralStringValue(rhs)} - * ] )`;
        },
        supportedTypes: [
            ConditionBuilderVariableType.INTEGER,
            ConditionBuilderVariableType.REAL,
            ConditionBuilderVariableType.DATE_TIME,
            ConditionBuilderVariableType.DATE,
            ConditionBuilderVariableType.TIME,
        ],
        getLiteralType(variableType) {
            return variableType;
        },
    },
    [Operator.CONTAINS_STRING]: {
        getRule(condition, variables) {
            const regexp = new RegExp(`^#[\\-_0-9a-zA-Z @.]+# % ${VAR_REGEX_SRC}$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' % ');
                const variableName = split[1];
                const operator = Operator.CONTAINS_STRING;
                const literal = split[0];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'contains',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `${getLiteralStringValue(rhs)} % ${variableName}`;
        },
        supportedTypes: [ConditionBuilderVariableType.STRING],
        getLiteralType(variableType) {
            return variableType;
        },
    },
    [Operator.NOT_CONTAINS_STRING]: {
        getRule(condition, variables) {
            const notMatch = /^! \( .+ \)$/;
            const isNot = condition.match(notMatch);
            if (!isNot)
                return null;
            condition = condition.slice(4, condition.length - 2);
            const regexp = new RegExp(`^#[\\-_0-9a-zA-Z @.]+# % ${VAR_REGEX_SRC}$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' % ');
                const variableName = split[1];
                const operator = Operator.NOT_CONTAINS_STRING;
                const literal = split[0];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'does not contain',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `${getLiteralStringValue(rhs)} % ${variableName}`;
        },
        supportedTypes: [ConditionBuilderVariableType.STRING],
        getLiteralType(variableType) {
            return variableType;
        },
    },
    [Operator.CONTAINS_ANY_OF]: {
        getRule(condition, variables) {
            const regexp = new RegExp(`${VAR_REGEX_SRC} <% \\[ .+ ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' <% ');
                const variableName = split[0];
                const operator = Operator.CONTAINS_ANY_OF;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'contains any of',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `${variableName} <% ${getLiteralStringValue(rhs)}`;
        },
        supportedTypes: [
            ConditionBuilderVariableType.LIST_INTEGER,
            ConditionBuilderVariableType.LIST_STRING,
            ConditionBuilderVariableType.LIST_REAL,
            ConditionBuilderVariableType.CATEGORY_MULTIPLE,
        ],
        getLiteralType(variableType) {
            if (variableType === ConditionBuilderVariableType.CATEGORY_MULTIPLE) {
                return ConditionBuilderVariableType.LIST_STRING;
            }
            return variableType;
        },
    },
    [Operator.NOT_CONTAINS_ANY_OF]: {
        getRule(condition, variables) {
            const notMatch = /^! \( .+ \)$/;
            const isNot = condition.match(notMatch);
            if (!isNot)
                return null;
            condition = condition.slice(4, condition.length - 2);
            const regexp = new RegExp(`${VAR_REGEX_SRC} <% \\[ .+ ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' <% ');
                const variableName = split[0];
                const operator = Operator.NOT_CONTAINS_ANY_OF;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'does not contain any of',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `! ( ${variableName} <% ${getLiteralStringValue(rhs)} )`;
        },
        supportedTypes: [
            ConditionBuilderVariableType.LIST_INTEGER,
            ConditionBuilderVariableType.LIST_STRING,
            ConditionBuilderVariableType.LIST_REAL,
            ConditionBuilderVariableType.CATEGORY_MULTIPLE,
        ],
        getLiteralType(variableType) {
            if (variableType === ConditionBuilderVariableType.CATEGORY_MULTIPLE) {
                return ConditionBuilderVariableType.LIST_STRING;
            }
            return variableType;
        },
    },
    [Operator.CONTAINS_ALL_OF]: {
        getRule(condition, variables) {
            const regexp = new RegExp(`${VAR_REGEX_SRC} %> \\[ .+ ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' %> ');
                const variableName = split[0];
                const operator = Operator.CONTAINS_ALL_OF;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'contains all of',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `${variableName} %> ${getLiteralStringValue(rhs)}`;
        },
        supportedTypes: [
            ConditionBuilderVariableType.LIST_INTEGER,
            ConditionBuilderVariableType.LIST_STRING,
            ConditionBuilderVariableType.LIST_REAL,
            ConditionBuilderVariableType.CATEGORY_MULTIPLE,
        ],
        getLiteralType(variableType) {
            if (variableType === ConditionBuilderVariableType.CATEGORY_MULTIPLE) {
                return ConditionBuilderVariableType.LIST_STRING;
            }
            return variableType;
        },
    },
    [Operator.NOT_CONTAINS_ALL_OF]: {
        getRule(condition, variables) {
            const notMatch = /^! \( .+ \)$/;
            const isNot = condition.match(notMatch);
            if (!isNot)
                return null;
            condition = condition.slice(4, condition.length - 2);
            const regexp = new RegExp(`${VAR_REGEX_SRC} %> \\[ .+ ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' %> ');
                const variableName = split[0];
                const operator = Operator.NOT_CONTAINS_ALL_OF;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'does not contain all of',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `! ( ${variableName} %> ${getLiteralStringValue(rhs)} )`;
        },
        supportedTypes: [
            ConditionBuilderVariableType.LIST_INTEGER,
            ConditionBuilderVariableType.LIST_STRING,
            ConditionBuilderVariableType.LIST_REAL,
            ConditionBuilderVariableType.CATEGORY_MULTIPLE,
        ],
        getLiteralType(variableType) {
            if (variableType === ConditionBuilderVariableType.CATEGORY_MULTIPLE) {
                return ConditionBuilderVariableType.LIST_STRING;
            }
            return variableType;
        },
    },
    [Operator.EXACTEQUAL]: {
        getRule(condition, variables) {
            const regexp = new RegExp(`^${VAR_REGEX_SRC} === \\[ .+ ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' === ');
                const variableName = split[0];
                const operator = Operator.EXACTEQUAL;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'is',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `${variableName} === ${getLiteralStringValue(rhs)}`;
        },
        supportedTypes: [ConditionBuilderVariableType.CATEGORY_RANKING],
        getLiteralType(variableType) {
            return variableType;
        },
    },
    [Operator.NOT_EXACTEQUAL]: {
        getRule(condition, variables) {
            const notMatch = /^! \( .+ \)$/;
            const isNot = condition.match(notMatch);
            if (!isNot)
                return null;
            condition = condition.slice(4, condition.length - 2);
            const regexp = new RegExp(`^${VAR_REGEX_SRC} === \\[ .+ ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' === ');
                const variableName = split[0];
                const operator = Operator.NOT_EXACTEQUAL;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'is not',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `! ( ${variableName} === ${getLiteralStringValue(rhs)} )`;
        },
        supportedTypes: [ConditionBuilderVariableType.CATEGORY_RANKING],
        getLiteralType(variableType) {
            return variableType;
        },
    },
    [Operator.RANKING_ORDER]: {
        getRule(condition, variables) {
            const regexp = new RegExp(`^#\\d+# < #\\d+# % ${VAR_REGEX_SRC}`);
            const matches = condition.match(regexp);
            if (matches) {
                let split = condition.split(' < ');
                split = [split[0], ...split[1].split(' % ')];
                const variableName = split[2];
                const operator = Operator.RANKING_ORDER;
                const literal = `${split[0]}:${split[1]}`;
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'has order',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            const [option1, option2] = getLiteralStringValue(rhs).split(':');
            return `${option1} < ${option2} % ${variableName}`;
        },
        supportedTypes: [ConditionBuilderVariableType.CATEGORY_RANKING],
        getLiteralType() {
            return 'RANKING_ORDER';
        },
    },
    [Operator.NOT_RANKING_ORDER]: {
        getRule(condition, variables) {
            const notMatch = /^! \( .+ \)$/;
            const isNot = condition.match(notMatch);
            if (!isNot)
                return null;
            condition = condition.slice(4, condition.length - 2);
            const regexp = new RegExp(`^#\\d+# < #\\d+# % ${VAR_REGEX_SRC}`);
            const matches = condition.match(regexp);
            if (matches) {
                let split = condition.split(' < ');
                split = [split[0], ...split[1].split(' % ')];
                const variableName = split[2];
                const operator = Operator.NOT_RANKING_ORDER;
                const literal = `${split[0]}:${split[1]}`;
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: "doesn't have order",
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            const [option1, option2] = getLiteralStringValue(rhs).split(':');
            return `! ( ${option1} < ${option2} % ${variableName} )`;
        },
        supportedTypes: [ConditionBuilderVariableType.CATEGORY_RANKING],
        getLiteralType() {
            return 'RANKING_ORDER';
        },
    },
    [Operator.RANKING_POSITION]: {
        getRule(condition, variables) {
            const regexp = new RegExp(`^${VAR_REGEX_SRC} <|> #\\d# \\[ #[\\-_0-9a-zA-Z @.]+# (, #[\\-_0-9a-zA-Z @.]+#)* ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' <|> ');
                const variableName = split[0];
                const literal = split[1];
                const operator = Operator.RANKING_POSITION;
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'has',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            const stringValue = getLiteralStringValue(rhs);
            return `${variableName} <|> ${stringValue}`;
        },
        supportedTypes: [ConditionBuilderVariableType.CATEGORY_RANKING],
        getLiteralType() {
            return 'RANKING_POSITION';
        },
    },
    [Operator.NOT_RANKING_POSITION]: {
        getRule(condition, variables) {
            const notMatch = /^! \( .+ \)$/;
            const isNot = condition.match(notMatch);
            if (!isNot)
                return null;
            condition = condition.slice(4, condition.length - 2);
            const regexp = new RegExp(`^${VAR_REGEX_SRC} <|> #\\d# \\[ #[\\-_0-9a-zA-Z @.]+# (, #[\\-_0-9a-zA-Z @.]+#)* ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' <|> ');
                const variableName = split[0];
                const literal = split[1];
                const operator = Operator.NOT_RANKING_POSITION;
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: "doesn't have",
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            const stringValue = getLiteralStringValue(rhs);
            return `${variableName} <|> ${stringValue}`;
        },
        supportedTypes: [ConditionBuilderVariableType.CATEGORY_RANKING],
        getLiteralType() {
            return 'RANKING_POSITION';
        },
    },
};
//# sourceMappingURL=OperatorConfigMap.js.map