import { useDrag, useDrop } from 'react-dnd';
import React, { useMemo, useState } from 'react';
import { DragHint } from '../drag-hint/DragHint';
function getHoverPosition(monitor, container) {
    let position = 0;
    if (monitor.isOver() && container) {
        const offset = monitor.getClientOffset().y;
        const children = container.children;
        let rects = [];
        for (let i = 0; i < children.length; i++) {
            const child = children[i];
            const rect = child.getBoundingClientRect();
            rects.push(rect);
        }
        rects = rects.filter((rect) => rect.height > 10);
        let firstBelowIndex = rects.findIndex((rect) => rect.top > offset);
        if (firstBelowIndex === -1) {
            firstBelowIndex = rects.length;
        }
        if (firstBelowIndex === 0) {
            position = 0;
        }
        else {
            const rectIndex = firstBelowIndex - 1;
            const rect = rects[rectIndex];
            const middle = rect.top + rect.height / 2;
            position = middle > offset ? rectIndex : rectIndex + 1;
        }
    }
    return position;
}
// this is to be used in the target container
export function useDraggableLibraryItemDrop(containerRef, dropItem, accept, items, isNewItemDragging) {
    const [hoverPosition, setHoverPosition] = useState(-1);
    const isZeroCards = items.length === 0;
    const [{ isDraggingOver }, drop] = useDrop({
        accept,
        collect(monitor) {
            const isOver = monitor.isOver();
            return {
                isDraggingOver: isOver,
            };
        },
        hover(item, monitor) {
            const container = containerRef.current;
            const position = getHoverPosition(monitor, container);
            setHoverPosition(position);
        },
        drop(item, monitor) {
            const container = containerRef.current;
            setHoverPosition(-1);
            const position = isZeroCards ? 0 : getHoverPosition(monitor, container);
            if (container) {
                dropItem(position, item);
            }
        },
    });
    const isDraggingOutside = isNewItemDragging && !isDraggingOver;
    const itemsWithPreview = useMemo(function () {
        const newSlides = items.map((it) => it);
        if (isDraggingOver)
            newSlides.splice(hoverPosition, 0, { preview: true });
        return newSlides;
    }, [items, hoverPosition, isDraggingOver]);
    drop(containerRef);
    return {
        isDraggingOver,
        isZeroCards,
        isDraggingOutside,
        itemsWithPreview,
    };
}
// this is used in the items of source container
export function useDraggableLibraryItem(type, itemProperties, updateDragStatus, itemRef, canDrag = true) {
    const [, drag] = useDrag({
        item: Object.assign({ type }, itemProperties),
        begin() {
            updateDragStatus(true);
        },
        end() {
            updateDragStatus(false);
        },
        canDrag() {
            return canDrag;
        },
    });
    drag(itemRef);
}
// used to render items with a line indicating where the dropped item will be placed
// it will also show hint if something can be dropped.
// TODO: send text as props
export function renderDropContainerItems(dropConfig, isDragging, renderItem) {
    if (dropConfig.isZeroCards && !isDragging) {
        return React.createElement(DragHint, { text: 'Drop something here' });
    }
    if (dropConfig.isZeroCards && dropConfig.isDraggingOver) {
        return React.createElement(DragHint, { text: 'Drop to Add' });
    }
    if (dropConfig.isDraggingOutside) {
        return React.createElement(DragHint, { text: 'Drop Here' });
    }
    return dropConfig.itemsWithPreview.map((item, index) => {
        if (!item) {
            return;
        }
        if (item.preview) {
            return (React.createElement("div", { key: `preview-${index}`, style: {
                    width: '100%',
                    height: 2,
                    background: 'green',
                    margin: 3,
                } }));
        }
        return renderItem(item, index);
    });
}
//# sourceMappingURL=DraggableLibraryItem.js.map