import React, { useCallback, useEffect, useLayoutEffect, useMemo, useRef, useState, } from 'react';
import { Editor as TinyMCEEditor } from '@tinymce/tinymce-react';
import uuid from 'uuid';
import { useDebouncedValue } from '../../hooks';
import { addChips, createPlugins, removeChips } from './plugins';
import './HTMLEditor.scss';
export const HTMLEditor = (props) => {
    const { value, onChange, style, variables, useVariablesPlugin = false, variablesPluginName = 'Variables', useQuestionnaireLinkPlugin = false, } = props;
    const editorRef = useRef(null);
    const editorId = useMemo(() => {
        return uuid.v4();
    }, []);
    const varRef = useRef([]);
    function setup() {
        createPlugins(varRef, useVariablesPlugin, variablesPluginName, useQuestionnaireLinkPlugin);
    }
    useEffect(() => {
        varRef.current = variables || [];
    }, [variables]);
    const [immediateCurrentValue, setCurrentValue] = useState(value);
    // sometimes tinymce creates multiple updates very fast.
    // call on change when it is stable
    const currentValue = useDebouncedValue(immediateCurrentValue, 1000);
    useEffect(() => {
        if (currentValue !== value) {
            onChange(currentValue);
        }
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [currentValue]);
    useLayoutEffect(() => {
        var _a, _b;
        const editorComponent = editorRef.current;
        if (currentValue !== value && editorComponent) {
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            (_b = (_a = editorComponent.editor) === null || _a === void 0 ? void 0 : _a.setContent) === null || _b === void 0 ? void 0 : _b.call(_a, addChips(value, varRef.current));
            setCurrentValue(value);
        }
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [value]);
    const editorInitialValue = useMemo(() => {
        return addChips(currentValue, variables !== null && variables !== void 0 ? variables : []);
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, []);
    const onEditorChange = useCallback((value) => {
        setCurrentValue(removeChips(value));
    }, []);
    // have provided initial value because value doesn't work
    // the editor doesn't work as a controlled component
    const questionnaireToolbarItems = [];
    if (useQuestionnaireLinkPlugin) {
        questionnaireToolbarItems.push('questionnairelink');
    }
    if (useVariablesPlugin) {
        questionnaireToolbarItems.push('htmleditorvars');
    }
    const questionnaireToolbarItemsStr = questionnaireToolbarItems.join(' | ');
    const questionnairePluginItems = [];
    if (useQuestionnaireLinkPlugin) {
        questionnairePluginItems.push('questionnairelink');
    }
    if (useVariablesPlugin) {
        questionnairePluginItems.push('htmleditorvars');
    }
    const questionnairePluginItemsStr = questionnairePluginItems.join(' ');
    return (React.createElement("div", { style: style, className: 'flex flex-col items-stretch' },
        React.createElement(TinyMCEEditor, { ref: editorRef, id: editorId, initialValue: editorInitialValue, init: {
                height: 200,
                menubar: false,
                setup,
                plugins: [
                    'advlist autolink lists link image charmap print preview anchor',
                    'searchreplace visualblocks fullscreen',
                    'insertdatetime media table paste code help wordcount ' +
                        questionnairePluginItemsStr,
                ],
                toolbar: 'undo redo | formatselect | fontsizeselect |bold italic backcolor | link |' +
                    'alignleft aligncenter alignright alignjustify | ' +
                    'bullist numlist outdent indent | removeformat | help | ' +
                    questionnaireToolbarItemsStr,
            }, tinymceScriptSrc: "https://cdnjs.cloudflare.com/ajax/libs/tinymce/5.6.1/tinymce.min.js", onEditorChange: onEditorChange })));
};
//# sourceMappingURL=HTMLEditor.js.map