import React, { Component, createRef, Fragment } from 'react';
import * as PropTypes from 'prop-types';
import './MarkDownEditor.scss';
import { getCaretPosition, getSuggestionText, getTextAtCaret, insertTextAtCaret, replaceHtmlAtCaret, spanClose, spanOpen, transformInnerHTMLToValue, transformValueToInnerHTMl, } from './utilities';
export class MarkDownEditor extends Component {
    constructor(props) {
        super(props);
        this.caretPosition = 0;
        this.editorRef = createRef();
        this.suggestionsListRef = createRef();
        this.handleEvent = this.handleEvent.bind(this);
        this.hoverOnOption = this.hoverOnOption.bind(this);
        this.selectSuggestion = this.selectSuggestion.bind(this);
        this.clickOnOption = this.clickOnOption.bind(this);
        this.handleFocusEvent = this.handleFocusEvent.bind(this);
        this.handleBlurEvent = this.handleBlurEvent.bind(this);
        this.scrollToSuggestion = this.scrollToSuggestion.bind(this);
        this.handleCaretChange = this.handleCaretChange.bind(this);
        this.callOnChange = this.callOnChange.bind(this);
        this.handleCut = this.handleCut.bind(this);
        this.handlePaste = this.handlePaste.bind(this);
        this.createVariableClicked = this.createVariableClicked.bind(this);
        this.defaultSuggestionState = {
            suggestionText: '',
            shownSuggestions: [],
            // not used
            suggestionCaretPosition: 0,
            selectedSuggestionIndex: 0,
            suggestionMode: false,
            showSuggestions: false,
        };
        this.state = Object.assign(Object.assign({}, this.defaultSuggestionState), { value: '' });
    }
    componentDidMount() {
        this.editorRef.current.addEventListener('change', this.handleEvent, false);
        this.editorRef.current.addEventListener('paste', this.handlePaste, false);
        this.editorRef.current.addEventListener('cut', this.handleCut, false);
        this.editorRef.current.addEventListener('keydown', this.handleEvent, false);
        this.editorRef.current.addEventListener('keypress', this.handleEvent, false);
        this.editorRef.current.addEventListener('input', this.handleEvent, false);
        this.editorRef.current.addEventListener('focus', this.handleFocusEvent, false);
        this.editorRef.current.addEventListener('blur', this.handleBlurEvent, false);
        this.editorRef.current.addEventListener('keydown', this.handleCaretChange, false);
        this.editorRef.current.addEventListener('click', this.handleCaretChange, false);
        this.editorRef.current.addEventListener('focus', this.handleCaretChange, false);
        if (this.props.value !== undefined) {
            this.handleValueChange();
        }
    }
    componentDidUpdate(prevProps, prevState, snapshot) {
        if (this.props.value !== undefined &&
            this.props.value !== prevProps.value) {
            this.handleValueChange();
        }
    }
    scrollToSuggestion(suggestionId) {
        const suggestionsList = this.suggestionsListRef.current;
        if (suggestionsList) {
            const elementById = document.getElementById(`suggestion-${suggestionId}`);
            // TODO better calculation based on view boxes
            elementById === null || elementById === void 0 ? void 0 : elementById.scrollIntoView();
        }
    }
    handleFocusEvent() {
        this.handleSuggestionsDisplay();
    }
    handleCaretChange(event) {
        if (event.type === 'keydown') {
            if (this.state.suggestionMode) {
                switch (event.which) {
                    //esc
                    case 27:
                        return;
                    // up arrow
                    case 38:
                        return;
                    //down arrow
                    case 40:
                        return;
                }
            }
            if (event.which === 27) {
                return;
            }
        }
        setTimeout(this.callOnChange);
        this.handleSuggestionsDisplay();
    }
    handleCut() {
        setTimeout(this.callOnChange);
    }
    handlePaste(event) {
        // Stop data actually being pasted into div
        event.stopPropagation();
        event.preventDefault();
        // Get pasted data via clipboard API
        const clipboardData = event.clipboardData || window.clipboardData;
        // TODO handle multiline
        let value = clipboardData.getData('Text').trim();
        if (!this.props.multiline) {
            value = value.replace(/[\r\n]+/g, ' ');
        }
        insertTextAtCaret(value, this.editorRef.current);
        this.callOnChange();
        return false;
    }
    callOnChange() {
        const element = this.editorRef.current;
        const childNodes = element.childNodes;
        const value = transformInnerHTMLToValue(childNodes);
        this.setState({ value }, () => {
            this.props.onChange(value, this.props.data);
        });
    }
    handleValueChange() {
        const value = this.props.value;
        if (value !== this.state.value) {
            this.setState({ value });
            this.editorRef.current.innerHTML = transformValueToInnerHTMl(value !== null && value !== void 0 ? value : '');
        }
    }
    handleBlurEvent() {
        this.caretPosition = getCaretPosition();
        setTimeout(() => {
            // to make sure multiple consecutive text nodes are merged
            this.editorRef.current.innerHTML = transformValueToInnerHTMl(this.state.value);
            this.closeSuggestions();
            typeof this.props.onBlur === 'function' && this.props.onBlur();
        }, 0);
    }
    handleSuggestionsDisplay() {
        setTimeout(() => {
            const caretPosition = getCaretPosition();
            const text = getTextAtCaret(this.editorRef.current);
            for (let i = caretPosition - 1; i >= 0; i--) {
                // TODO: handle other characters not allowed during variable creation
                if (text[i] === ' ') {
                    break;
                }
                if (text[i] === '#') {
                    this.setState({
                        suggestionText: text.slice(i + 1, caretPosition),
                        suggestionCaretPosition: caretPosition,
                        suggestionMode: true,
                        showSuggestions: true,
                    });
                    this.updateSuggestionText(this.editorRef.current);
                    return;
                }
            }
            this.closeSuggestions();
        });
    }
    handleEvent(event) {
        if (event.type === 'keydown') {
            if (this.state.suggestionMode) {
                let selectedSuggestionIndex;
                switch (event.which) {
                    // up arrow
                    case 38:
                        event.preventDefault();
                        selectedSuggestionIndex = Math.max(0, this.state.selectedSuggestionIndex - 1);
                        this.setState({
                            selectedSuggestionIndex,
                        });
                        this.scrollToSuggestion(selectedSuggestionIndex);
                        return;
                    //down arrow
                    case 40: {
                        event.preventDefault();
                        const isCreateVariableEnabled = this.props.allowVariableCreation &&
                            this.state.suggestionText &&
                            // make sure that no other variable exists with same exact text
                            !this.state.shownSuggestions.find((suggestion) => suggestion === this.state.suggestionText);
                        // if variable creation is enabled, there will be one
                        // extra entry in the dropdown
                        selectedSuggestionIndex = Math.min(isCreateVariableEnabled
                            ? this.state.shownSuggestions.length
                            : this.state.shownSuggestions.length - 1, this.state.selectedSuggestionIndex + 1);
                        this.setState({
                            selectedSuggestionIndex,
                        });
                        this.scrollToSuggestion(selectedSuggestionIndex);
                        return;
                    }
                    //esc
                    case 27:
                        this.setState(Object.assign({}, this.defaultSuggestionState));
                        event.preventDefault();
                        return;
                    //space
                    // case 32:
                    //   this.selectVariable(event);
                    //   return
                    //enter
                    case 13: {
                        const isCreateVariableEnabled = this.props.allowVariableCreation &&
                            this.state.suggestionText &&
                            // make sure that no other variable exists with same exact text
                            !this.state.shownSuggestions.find((suggestion) => suggestion === this.state.suggestionText);
                        // if
                        if (this.state.selectedSuggestionIndex ===
                            this.state.shownSuggestions.length &&
                            isCreateVariableEnabled) {
                            event.preventDefault();
                            this.createVariableClicked();
                        }
                        else {
                            this.selectVariable(event);
                        }
                        return;
                    }
                    default:
                        this.updateSuggestionText(event.target);
                }
            }
            else {
                if (event.which === 13) {
                    // enter
                    event.preventDefault();
                    if (this.props.multiline) {
                        insertTextAtCaret('\n', this.editorRef.current);
                        this.callOnChange();
                    }
                }
                else if (event.which === 8 && window.getSelection()) {
                    // // backspace
                    // TODO: fix delete span in firefox
                    // this code is to handle the case where cursor is to the right of variable span and backspace is clicked
                    // expectation: span should be removed from the html
                    // TODO: handle similar case for delete when the cursor is to the left of span
                    // const editor = this.editorRef.current;
                    // const caretPosition = getCaretPosition(editor);
                    // const anchorNode = window.getSelection().anchorNode;
                    // if (
                    //   caretPosition === 0 &&
                    //   anchorNode.previousElementSibling &&
                    //   anchorNode.previousElementSibling instanceof HTMLSpanElement
                    // ) {
                    //   editor.removeChild(anchorNode.previousElementSibling);
                    //   event.preventDefault();
                    //   return;
                    // }
                    // // if there is no text after caret
                    // const editorChildren = editor.children;
                    // const editorLastChild =
                    //   editorChildren.length && editorChildren[editorChildren.length - 1];
                    // if (
                    //   anchorNode === editor &&
                    //   editorLastChild &&
                    //   editorLastChild instanceof HTMLSpanElement
                    // ) {
                    //   editorChildren.removeChild(editorLastChild);
                    //   return;
                    // }
                }
            }
            switch (event.key) {
                case '#':
                    this.showSuggestions();
                    break;
            }
        }
    }
    selectVariable(event) {
        const shownSuggestions = this.state.shownSuggestions;
        const selectedSuggestionIndex = this.state.selectedSuggestionIndex;
        if (selectedSuggestionIndex < shownSuggestions.length &&
            shownSuggestions.length) {
            event.preventDefault();
            this.selectSuggestion(event.target);
        }
    }
    showSuggestions() {
        this.setState({
            suggestionMode: true,
            showSuggestions: true,
            shownSuggestions: this.props.contentVariables.map((a) => a.variableName),
        });
    }
    createVariableClicked() {
        this.props.createVariable &&
            this.props.createVariable(this.state.suggestionText);
    }
    render() {
        return (React.createElement(Fragment, null,
            React.createElement("input", { id: this.props.id, name: this.props.name, type: this.props.type, style: {
                    display: 'none',
                }, value: this.state.value }),
            React.createElement("div", { className: 'MarkDownEditor', style: this.props.style },
                this.props.title && (React.createElement("div", { className: 'title' }, this.props.title)),
                React.createElement("div", { key: 'editor', className: 'editor', ref: this.editorRef, contentEditable: this.props.disabled ? undefined : true, "data-div-placeholder-content": this.state.value ? true : undefined, "data-placeholder": this.props.placeholder, style: {
                        borderRadius: '7px',
                        height: this.props.multiline ? 150 : undefined,
                        border: !this.props.isValid ? '2px solid red' : undefined,
                    } }),
                this.state.showSuggestions && (React.createElement("div", { className: `suggestions-list ${this.state.showSuggestions.length === 0 ? 'hidden' : ''}`, ref: this.suggestionsListRef },
                    this.state.shownSuggestions.map((suggestion, index) => {
                        const selected = this.state.selectedSuggestionIndex === index;
                        return (React.createElement("div", { className: `suggestion ${selected ? ' selected' : ''}`, "data-selected": selected, key: suggestion, "data-option": true, "data-index": index, id: `suggestion-${index}`, onMouseEnter: this.hoverOnOption, onMouseDown: this.clickOnOption }, suggestion));
                    }),
                    this.props.allowVariableCreation &&
                        this.state.suggestionText &&
                        // make sure that no other variable exists with same exact text
                        !this.state.shownSuggestions.find((suggestion) => suggestion === this.state.suggestionText) && (React.createElement("div", { className: `suggestion ${this.state.selectedSuggestionIndex ===
                            this.state.shownSuggestions.length
                            ? ' selected'
                            : ''}`, "data-selected": this.state.selectedSuggestionIndex ===
                            this.state.shownSuggestions.length, key: `create-variable-${this.state.suggestionText}`, "data-option": true, "data-index": this.state.shownSuggestions.length, id: `suggestion-${this.state.shownSuggestions.length}`, onMouseEnter: this.hoverOnOption, onMouseDown: this.createVariableClicked },
                        "Create variable \"",
                        this.state.suggestionText,
                        "\"")))))));
    }
    clickOnOption(e) {
        e.preventDefault();
        const shownSuggestions = this.state.shownSuggestions;
        const selectedSuggestionIndex = this.state.selectedSuggestionIndex;
        this.editorRef.current.focus();
        if (selectedSuggestionIndex < shownSuggestions.length &&
            shownSuggestions.length) {
            this.selectSuggestion(this.editorRef.current);
        }
    }
    selectSuggestion() {
        const shownSuggestions = this.state.shownSuggestions;
        const selectedSuggestionIndex = this.state.selectedSuggestionIndex;
        replaceHtmlAtCaret(`${spanOpen}#${shownSuggestions[selectedSuggestionIndex]}${spanClose}`, this.state.suggestionText.length + 1);
        this.setState(Object.assign({}, this.defaultSuggestionState));
    }
    closeSuggestions() {
        this.setState(Object.assign({}, this.defaultSuggestionState));
    }
    hoverOnOption(ev) {
        const index = parseInt(ev.target.getAttribute('data-index'), 10);
        this.setState({ selectedSuggestionIndex: index });
    }
    componentWillUnmount() {
        this.editorRef.current.removeEventListener('change', this.handleEvent, false);
        this.editorRef.current.removeEventListener('paste', this.handlePaste, false);
        this.editorRef.current.removeEventListener('cut', this.handleCut, false);
        this.editorRef.current.removeEventListener('keydown', this.handleEvent, false);
        this.editorRef.current.removeEventListener('keypress', this.handleEvent, false);
        this.editorRef.current.removeEventListener('input', this.handleEvent, false);
        this.editorRef.current.removeEventListener('focus', this.handleFocusEvent, false);
        this.editorRef.current.removeEventListener('blur', this.handleBlurEvent, false);
        this.editorRef.current.removeEventListener('keydown', this.handleCaretChange, false);
        this.editorRef.current.removeEventListener('click', this.handleCaretChange, false);
        this.editorRef.current.removeEventListener('focus', this.handleCaretChange, false);
    }
    updateSuggestionText(element) {
        setTimeout(() => {
            const prevSelectedSuggestion = this.state.showSuggestions[this.state.selectedSuggestionIndex];
            const suggestionText = getSuggestionText(element);
            this.setState({
                suggestionText,
                shownSuggestions: this.props.contentVariables
                    .filter((suggestion) => suggestion.variableName.startsWith(suggestionText) &&
                    suggestion.variableType !== 'IMAGE')
                    .map((a) => a.variableName),
                selectedSuggestionIndex: Math.max(0, this.state.shownSuggestions.indexOf(prevSelectedSuggestion)),
            });
        });
    }
}
MarkDownEditor.propTypes = {
    data: PropTypes.any,
    placeholder: PropTypes.string,
    title: PropTypes.string,
    onChange: PropTypes.func,
    onBlur: PropTypes.func,
    id: PropTypes.string,
    type: PropTypes.string,
    name: PropTypes.string,
    contentVariables: PropTypes.array,
    value: PropTypes.string,
    multiline: PropTypes.bool,
    isValid: PropTypes.bool,
    disabled: PropTypes.bool,
    style: PropTypes.object,
    allowVariableCreation: PropTypes.bool,
    createVariable: PropTypes.func,
};
MarkDownEditor.defaultProps = {
    placeholder: 'Start typing...',
    isValid: true,
    onChange: function () {
        // empty function
    },
    multiline: false,
    allowVariableCreation: false,
};
//# sourceMappingURL=MarkDownEditor.js.map