import { __awaiter, __decorate, __metadata } from "tslib";
import { action, computed, observable, reaction } from 'mobx';
import { BusinessService, LoginService } from '../services';
import { applyNetworkResponse, applyNetworkResponseForPromise, getDefaultNetworkResponse, http, } from '../utils';
import * as storage from '../utils/storage';
const SESSION_ID_HEADER = 'session_id';
export class LoginStore {
    constructor() {
        // FIXME: [FIXTURES] this is temporary to work with fixtures
        this.businessIdsResponse = getDefaultNetworkResponse();
        this.loginResponse = getDefaultNetworkResponse();
        this.setBusinessIds = (businessIds) => {
            this.businessIdsResponse = {
                data: businessIds,
                error: null,
                isLoading: false,
            };
        };
        this.updateSavedLogin = () => __awaiter(this, void 0, void 0, function* () {
            try {
                applyNetworkResponse(this.loginResponse, true, null, null);
                const savedLogin = yield storage.getLogin();
                if (savedLogin) {
                    applyNetworkResponse(this.loginResponse, false, savedLogin, null);
                }
            }
            catch (e) {
                applyNetworkResponse(this.loginResponse, false, null, e);
            }
        });
        this.login = (loginFormData) => {
            const promise = LoginService.login(loginFormData.username, loginFormData.password, loginFormData.businessId);
            applyNetworkResponseForPromise(this.loginResponse, promise.then((resp) => ({
                userId: resp.userEntity.externalId,
                sessionId: resp.shiroSession.id,
                businessId: resp.userEntity.businessId,
            })));
        };
        this.logout = () => {
            const loginResponseData = this.loginResponse.data;
            if (!loginResponseData)
                return;
            LoginService.logout(loginResponseData.sessionId)
                .then(() => {
                applyNetworkResponse(this.loginResponse, false, null, null);
            })
                .catch(() => {
                applyNetworkResponse(this.loginResponse, false, null, null);
            });
        };
        this.addReactions();
        // FIXME: [FIXTURES] this is temporary to work with fixtures.
        applyNetworkResponseForPromise(this.businessIdsResponse, BusinessService.getBusinesses().then((resp) => resp.map((business) => business.externalId)));
        this.updateSavedLogin();
        // observe local storage to logout if logged out from other tab (other document)
        window.addEventListener('storage', this.updateSavedLogin);
        http.addErrorInterceptor((response) => {
            if (response.status === 401 && this.loginResponse.data) {
                applyNetworkResponse(this.loginResponse, false, null, {
                    message: 'Session timed out',
                });
            }
        });
    }
    get isLoggedIn() {
        // FIXME: [FIXTURES] this is temporary to work with business
        const businessIds = this.businessIdsResponse.data;
        return businessIds !== null && businessIds.length > 0;
    }
    get businessId() {
        // FIXME: [FIXTURES] this is temporary to work with business
        const businessIds = this.businessIdsResponse.data || [
            'ea1ca4b6-a9cf-43b7-8fe3-7f74429c10e2',
        ];
        if (!businessIds)
            throw new Error('No Business Id');
        return businessIds[0];
    }
    addReactions() {
        // whenever login response changes, update storage.
        reaction(() => this.loginResponse.data, () => {
            const loginResponse = this.loginResponse.data;
            if (loginResponse) {
                // eslint-disable-next-line @typescript-eslint/camelcase
                http.addHeader(SESSION_ID_HEADER, loginResponse.sessionId);
                storage.saveLogin(loginResponse);
            }
            else {
                http.removeHeader(SESSION_ID_HEADER);
                storage.deleteLogin();
            }
        });
    }
}
__decorate([
    observable,
    __metadata("design:type", Object)
], LoginStore.prototype, "businessIdsResponse", void 0);
__decorate([
    observable,
    __metadata("design:type", Object)
], LoginStore.prototype, "loginResponse", void 0);
__decorate([
    computed,
    __metadata("design:type", Boolean),
    __metadata("design:paramtypes", [])
], LoginStore.prototype, "isLoggedIn", null);
__decorate([
    computed,
    __metadata("design:type", String),
    __metadata("design:paramtypes", [])
], LoginStore.prototype, "businessId", null);
__decorate([
    action,
    __metadata("design:type", Object)
], LoginStore.prototype, "setBusinessIds", void 0);
//# sourceMappingURL=LoginStore.js.map