import { VALID_URL_REGEX } from '../constants';
const MAX_PIXEL_MOVEMENT_ALLOWED = 5;
function getDefaultPos() {
    return {
        distance: 0,
        originalX: 0,
        originalY: 0,
    };
}
export function getClickHandlers(clickFunction, 
// eslint-disable-next-line @typescript-eslint/no-empty-function
eventActions = () => { }) {
    let position = getDefaultPos();
    let down = false;
    function start(event) {
        if (event.button === 0) {
            eventActions(event);
            position.distance = 0;
            down = true;
            position.originalX = event.pageX;
            position.originalY = event.pageY;
        }
    }
    function move(event) {
        if (event.button === 0) {
            eventActions(event);
            if (down) {
                const dist = (event.pageX - position.originalX) *
                    (event.pageX - position.originalX) +
                    (event.pageY - position.originalY) *
                        (event.pageY - position.originalY);
                position.distance = Math.max(dist, position.distance);
            }
        }
    }
    function end(event) {
        if (event.button === 0) {
            eventActions(event);
            if (down && position.distance < MAX_PIXEL_MOVEMENT_ALLOWED) {
                clickFunction(event);
            }
            down = false;
        }
        position = getDefaultPos();
    }
    return {
        onMouseDown: start,
        onMouseUp: end,
        onMouseMove: move,
    };
}
export function getRandomColor() {
    const letters = '0123456789ABCDEF';
    let color = '#';
    for (let i = 0; i < 6; i++) {
        color += letters[Math.floor(Math.random() * 16)];
    }
    return color;
}
export function classNames(...args) {
    return args
        .map((arg) => {
        if (typeof arg === 'string') {
            return arg;
        }
        if (typeof arg === 'object') {
            return classNames(...Object.keys(arg).filter((key) => arg[key]));
        }
        return '';
    })
        .filter((it) => it !== '')
        .join(' ');
}
export function classNamesWithModules(styles) {
    function classNamesFn(...args) {
        return args
            .map((arg) => {
            if (typeof arg === 'string') {
                return styles[arg] || arg;
            }
            if (typeof arg === 'object') {
                return classNamesFn(...Object.keys(arg).filter((key) => arg[key]));
            }
            return '';
        })
            .filter((it) => it !== '')
            .join(' ');
    }
    return classNamesFn;
}
export function debounce(func, delay) {
    let debounceTimer;
    return function (...args) {
        clearTimeout(debounceTimer);
        // eslint-disable-next-line
        // @ts-ignore
        debounceTimer = setTimeout(() => func.apply(this, args), delay);
    };
}
export function htmlUnescape(str) {
    return str
        .replace(/&quot;/g, '"')
        .replace(/&#39;/g, "'")
        .replace(/&lt;/g, '<')
        .replace(/&gt;/g, '>')
        .replace(/&amp;/g, '&')
        .replace(/&nbsp;/g, ' ')
        .replace(new RegExp(String.fromCharCode(160), 'g'), ' ');
}
// I needed the opposite function today, so adding here too:
export function htmlEscape(str) {
    return str
        .replace(/&/g, '&amp;')
        .replace(/"/g, '&quot;')
        .replace(/'/g, '&#39;')
        .replace(/</g, '&lt;')
        .replace(/>/g, '&gt;');
}
export function isMacintosh() {
    return navigator.platform.indexOf('Mac') > -1;
}
export function getKeyCode(event) {
    let code;
    /* eslint-disable @typescript-eslint/no-explicit-any */
    if (event.which !== undefined) {
        code = event.which;
    }
    else if (event.keyCode !== undefined) {
        code = event.keyCode;
    }
    /* eslint-enable @typescript-eslint/no-explicit-any */
    return code;
}
export function isValidURL(str) {
    const res = str.match(VALID_URL_REGEX);
    return res !== null;
}
export function startTimer(callback, interval) {
    let count = 0;
    const initial = Date.now();
    let expected = initial + interval;
    let timeoutId;
    function step() {
        const now = Date.now();
        count = now - initial;
        callback(count);
        const dt = now - expected;
        expected += interval;
        timeoutId = setTimeout(step, Math.max(0, interval - dt)); // take into account drift
    }
    timeoutId = setTimeout(step, interval);
    return () => {
        clearTimeout(timeoutId);
    };
}
export function validateEmail(email) {
    const re = /^(([^<>()[\]\\.,;:\s@"]+(\.[^<>()[\]\\.,;:\s@"]+)*)|(".+"))@((\[[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\])|(([a-zA-Z\-0-9]+\.)+[a-zA-Z]{2,}))$/;
    return re.test(String(email).toLowerCase());
}
//# sourceMappingURL=common.js.map