"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OperatorConfigMap = void 0;
const utils_1 = require("./utils");
const Expression_1 = require("./Expression");
const Config_1 = require("./Config");
const constants_1 = require("../../../constants");
const rulePartUtils_1 = require("./rulePartUtils");
const ConditionBuilderVariable_1 = require("../ConditionBuilderVariable");
const VAR_REGEX_SRC = constants_1.CONTEXT_VARIABLE_REGEX.source;
function getVariable(variables, variableName) {
    return variables.find((it) => it.variableName === variableName);
}
function getRule(variables, variableName, operator, config, literal) {
    const variable = getVariable(variables, variableName);
    if (!variable)
        return null;
    return {
        type: Expression_1.ExpressionType.RULE,
        parts: [
            rulePartUtils_1.getVariableRulePart(variableName),
            rulePartUtils_1.getOperatorRulePart(operator, variable),
            rulePartUtils_1.getLiteralRulePart(config, variable, literal),
        ],
    };
}
exports.OperatorConfigMap = {
    [Config_1.Operator.BELONGS]: {
        getRule(condition, variables) {
            const regexp = new RegExp(`^${VAR_REGEX_SRC} % \\[ #[\\-_0-9a-zA-Z @.]+# (, #[\\-_0-9a-zA-Z @.]+#)* ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' % ');
                const variableName = split[0];
                const operator = Config_1.Operator.BELONGS;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'is in',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `${variableName} % ${utils_1.getLiteralStringValue(rhs)}`;
        },
        supportedTypes: [ConditionBuilderVariable_1.ConditionBuilderVariableType.CATEGORY],
        getLiteralType(variableType) {
            if (variableType === ConditionBuilderVariable_1.ConditionBuilderVariableType.CATEGORY) {
                return ConditionBuilderVariable_1.ConditionBuilderVariableType.LIST_STRING;
            }
            return ConditionBuilderVariable_1.ConditionBuilderVariableType.STRING;
        },
    },
    [Config_1.Operator.NOT_BELONGS]: {
        getRule(condition, variables) {
            const notMatch = /^! \( .+ \)$/;
            const isNot = condition.match(notMatch);
            if (!isNot)
                return null;
            condition = condition.slice(4, condition.length - 2);
            const regexp = new RegExp(`^${VAR_REGEX_SRC} % \\[ #[\\-_0-9a-zA-Z @.]+# (, #[\\-_0-9a-zA-Z @.]+#)* ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' % ');
                const variableName = split[0];
                const operator = Config_1.Operator.NOT_BELONGS;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'is not in',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `! ( ${variableName} % ${utils_1.getLiteralStringValue(rhs)} )`;
        },
        supportedTypes: [ConditionBuilderVariable_1.ConditionBuilderVariableType.CATEGORY],
        getLiteralType(variableType) {
            if (variableType === ConditionBuilderVariable_1.ConditionBuilderVariableType.CATEGORY) {
                return ConditionBuilderVariable_1.ConditionBuilderVariableType.LIST_STRING;
            }
            return ConditionBuilderVariable_1.ConditionBuilderVariableType.STRING;
        },
    },
    [Config_1.Operator.BELONGS_RANGE]: {
        getRule(condition, variables) {
            const regexp = new RegExp(`^${VAR_REGEX_SRC} % \\[ \\d+(\\.\\d+)? - \\d+(\\.\\d+)? ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' % ');
                const variableName = split[0];
                const operator = Config_1.Operator.BELONGS_RANGE;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'is in range',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `${variableName} % ${utils_1.getLiteralStringValue(rhs)}`;
        },
        supportedTypes: [
            ConditionBuilderVariable_1.ConditionBuilderVariableType.INTEGER,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.REAL,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.DATE,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.DATE_TIME,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.TIME,
        ],
        getLiteralType(variableType) {
            if (variableType === ConditionBuilderVariable_1.ConditionBuilderVariableType.TIME ||
                variableType === ConditionBuilderVariable_1.ConditionBuilderVariableType.DATE_TIME ||
                variableType === ConditionBuilderVariable_1.ConditionBuilderVariableType.DATE) {
                return 'DATE_RANGE';
            }
            return 'RANGE';
        },
    },
    [Config_1.Operator.NOT_BELONGS_RANGE]: {
        getRule(condition, variables) {
            const notMatch = /^! \( .+ \)$/;
            const isNot = condition.match(notMatch);
            if (!isNot)
                return null;
            condition = condition.slice(4, condition.length - 2);
            const regexp = new RegExp(`^${VAR_REGEX_SRC} % \\[ \\d+(\\.\\d+)? - \\d+(\\.\\d+)? ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' % ');
                const variableName = split[0];
                const operator = Config_1.Operator.NOT_BELONGS_RANGE;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'is not in range',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `! ( ${variableName} % ${utils_1.getLiteralStringValue(rhs)} )`;
        },
        supportedTypes: [
            ConditionBuilderVariable_1.ConditionBuilderVariableType.INTEGER,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.REAL,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.DATE,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.DATE_TIME,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.TIME,
        ],
        getLiteralType(variableType) {
            if (variableType === ConditionBuilderVariable_1.ConditionBuilderVariableType.TIME ||
                variableType === ConditionBuilderVariable_1.ConditionBuilderVariableType.DATE_TIME ||
                variableType === ConditionBuilderVariable_1.ConditionBuilderVariableType.DATE) {
                return 'DATE_RANGE';
            }
            return 'RANGE';
        },
    },
    [Config_1.Operator.EQUALS]: {
        getRule(condition, variables) {
            const regexp = new RegExp(`^${VAR_REGEX_SRC}  %  \\[ .+ ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split('  %  ');
                const variableName = split[0];
                const operator = Config_1.Operator.EQUALS;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: '=',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `${variableName}  %  [ ${utils_1.getLiteralStringValue(rhs)} ]`;
        },
        supportedTypes: [
            ConditionBuilderVariable_1.ConditionBuilderVariableType.STRING,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.BOOLEAN,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.CATEGORY,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.INTEGER,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.REAL,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.DATE,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.TIME,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.DATE_TIME,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.IMAGE,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.VIDEO,
        ],
        getLiteralType(variableType) {
            return variableType;
        },
    },
    [Config_1.Operator.NOT_EQUALS]: {
        getRule(condition, variables) {
            const notMatch = /^! \( .+ \)$/;
            const isNot = condition.match(notMatch);
            if (!isNot)
                return null;
            condition = condition.slice(4, condition.length - 2);
            const regexp = new RegExp(`^${VAR_REGEX_SRC}  %  \\[ .+ ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split('  %  ');
                const variableName = split[0];
                const operator = Config_1.Operator.NOT_EQUALS;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: '!=',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `! ( ${variableName}  %  [ ${utils_1.getLiteralStringValue(rhs)} ] )`;
        },
        supportedTypes: [
            ConditionBuilderVariable_1.ConditionBuilderVariableType.STRING,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.BOOLEAN,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.CATEGORY,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.INTEGER,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.REAL,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.DATE,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.TIME,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.DATE_TIME,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.IMAGE,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.VIDEO,
        ],
        getLiteralType(variableType) {
            return variableType;
        },
    },
    [Config_1.Operator.STARTSWITH]: {
        getRule(condition, variables) {
            const regexp = new RegExp(`^${VAR_REGEX_SRC} \\|% #[\\-_0-9a-zA-Z @.]+#`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' |% ');
                const variableName = split[0];
                const operator = Config_1.Operator.STARTSWITH;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'starts with',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `${variableName} |% ${utils_1.getLiteralStringValue(rhs)}`;
        },
        supportedTypes: [ConditionBuilderVariable_1.ConditionBuilderVariableType.STRING],
        getLiteralType() {
            return ConditionBuilderVariable_1.ConditionBuilderVariableType.STRING;
        },
    },
    [Config_1.Operator.NOT_STARTSWITH]: {
        getRule(condition, variables) {
            const notMatch = /^! \( .+ \)$/;
            const isNot = condition.match(notMatch);
            if (!isNot)
                return null;
            condition = condition.slice(4, condition.length - 2);
            const regexp = new RegExp(`^${VAR_REGEX_SRC} \\|% #[\\-_0-9a-zA-Z @.]+#`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' |% ');
                const variableName = split[0];
                const operator = Config_1.Operator.NOT_STARTSWITH;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'does not start with',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `! ( ${variableName} |% ${utils_1.getLiteralStringValue(rhs)} )`;
        },
        supportedTypes: [ConditionBuilderVariable_1.ConditionBuilderVariableType.STRING],
        getLiteralType() {
            return ConditionBuilderVariable_1.ConditionBuilderVariableType.STRING;
        },
    },
    [Config_1.Operator.ENDSWITH]: {
        getRule(condition, variables) {
            const regexp = new RegExp(`^${VAR_REGEX_SRC} %\\| #[\\-_0-9a-zA-Z @.]+#`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' %| ');
                const variableName = split[0];
                const operator = Config_1.Operator.ENDSWITH;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'ends with',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `${variableName} %| ${utils_1.getLiteralStringValue(rhs)}`;
        },
        supportedTypes: [ConditionBuilderVariable_1.ConditionBuilderVariableType.STRING],
        getLiteralType() {
            return ConditionBuilderVariable_1.ConditionBuilderVariableType.STRING;
        },
    },
    [Config_1.Operator.NOT_ENDSWITH]: {
        getRule(condition, variables) {
            const notMatch = /^! \( .+ \)$/;
            const isNot = condition.match(notMatch);
            if (!isNot)
                return null;
            condition = condition.slice(4, condition.length - 2);
            const regexp = new RegExp(`^${VAR_REGEX_SRC} %\\| #[\\-_0-9a-zA-Z @.]+#`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' %| ');
                const variableName = split[0];
                const operator = Config_1.Operator.NOT_ENDSWITH;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'does not end with',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `! ( ${variableName} %| ${utils_1.getLiteralStringValue(rhs)} )`;
        },
        supportedTypes: [ConditionBuilderVariable_1.ConditionBuilderVariableType.STRING],
        getLiteralType() {
            return ConditionBuilderVariable_1.ConditionBuilderVariableType.STRING;
        },
    },
    [Config_1.Operator.EXACTMATCH]: {
        getRule(condition, variables) {
            const regexp = new RegExp(`^${VAR_REGEX_SRC} == \\[ .+ ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' == ');
                const variableName = split[0];
                const operator = Config_1.Operator.EXACTMATCH;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'is',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `${variableName} == ${utils_1.getLiteralStringValue(rhs)}`;
        },
        supportedTypes: [
            ConditionBuilderVariable_1.ConditionBuilderVariableType.LIST_STRING,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.LIST_INTEGER,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.LIST_REAL,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.CATEGORY_MULTIPLE,
        ],
        getLiteralType(variableType) {
            if (variableType === ConditionBuilderVariable_1.ConditionBuilderVariableType.CATEGORY_MULTIPLE) {
                return ConditionBuilderVariable_1.ConditionBuilderVariableType.LIST_STRING;
            }
            return variableType;
        },
    },
    [Config_1.Operator.NOT_EXACTMATCH]: {
        getRule(condition, variables) {
            const notMatch = /^! \( .+ \)$/;
            const isNot = condition.match(notMatch);
            if (!isNot)
                return null;
            condition = condition.slice(4, condition.length - 2);
            const regexp = new RegExp(`^${VAR_REGEX_SRC} == \\[ .+ ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' == ');
                const variableName = split[0];
                const operator = Config_1.Operator.NOT_EXACTMATCH;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'is not',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `! ( ${variableName} == ${utils_1.getLiteralStringValue(rhs)} )`;
        },
        supportedTypes: [
            ConditionBuilderVariable_1.ConditionBuilderVariableType.LIST_STRING,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.LIST_INTEGER,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.LIST_REAL,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.CATEGORY_MULTIPLE,
        ],
        getLiteralType(variableType) {
            if (variableType === ConditionBuilderVariable_1.ConditionBuilderVariableType.CATEGORY_MULTIPLE) {
                return ConditionBuilderVariable_1.ConditionBuilderVariableType.LIST_STRING;
            }
            return variableType;
        },
    },
    [Config_1.Operator.ALLMATCH]: {
        getRule(condition, variables) {
            const regexp = new RegExp(`^${VAR_REGEX_SRC} \\|> \\[ .+ ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' |> ');
                const variableName = split[0];
                const operator = Config_1.Operator.ALLMATCH;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'are all in range',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `${variableName} |> ${utils_1.getLiteralStringValue(rhs)}`;
        },
        supportedTypes: [
            ConditionBuilderVariable_1.ConditionBuilderVariableType.LIST_INTEGER,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.LIST_REAL,
        ],
        getLiteralType() {
            return 'RANGE';
        },
    },
    [Config_1.Operator.ANYMATCH]: {
        getRule(condition, variables) {
            const regexp = new RegExp(`^${VAR_REGEX_SRC} <\\| \\[ .+ ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' <| ');
                const variableName = split[0];
                const operator = Config_1.Operator.ANYMATCH;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'has some in range',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `${variableName} <| ${utils_1.getLiteralStringValue(rhs)}`;
        },
        supportedTypes: [
            ConditionBuilderVariable_1.ConditionBuilderVariableType.LIST_INTEGER,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.LIST_REAL,
        ],
        getLiteralType() {
            return 'RANGE';
        },
    },
    [Config_1.Operator.LESSTHAN]: {
        getRule(condition, variables) {
            const regexp = new RegExp(`^${VAR_REGEX_SRC} % \\[ \\* - .+ ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' % ');
                const variableName = split[0];
                const operator = Config_1.Operator.LESSTHAN;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: '<',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `${variableName} % [ * - ${utils_1.getLiteralStringValue(rhs)} ]`;
        },
        supportedTypes: [
            ConditionBuilderVariable_1.ConditionBuilderVariableType.INTEGER,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.REAL,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.DATE_TIME,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.DATE,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.TIME,
        ],
        getLiteralType(variableType) {
            return variableType;
        },
    },
    [Config_1.Operator.GREATERTHAN_EQUALS]: {
        getRule(condition, variables) {
            const notMatch = /^! \( .+ \)$/;
            const isNot = condition.match(notMatch);
            if (!isNot)
                return null;
            condition = condition.slice(4, condition.length - 2);
            const regexp = new RegExp(`^${VAR_REGEX_SRC} % \\[ \\* - .+ ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' % ');
                const variableName = split[0];
                const operator = Config_1.Operator.GREATERTHAN_EQUALS;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: '>=',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `! ( ${variableName} % [ * - ${utils_1.getLiteralStringValue(rhs)} ] )`;
        },
        supportedTypes: [
            ConditionBuilderVariable_1.ConditionBuilderVariableType.INTEGER,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.REAL,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.DATE_TIME,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.DATE,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.TIME,
        ],
        getLiteralType(variableType) {
            return variableType;
        },
    },
    [Config_1.Operator.GREATERTHAN]: {
        getRule(condition, variables) {
            const regexp = new RegExp(`^${VAR_REGEX_SRC} % \\[ .+ - \\* ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' % ');
                const variableName = split[0];
                const operator = Config_1.Operator.LESSTHAN;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: '>',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `${variableName} % [ ${utils_1.getLiteralStringValue(rhs)} - * ]`;
        },
        supportedTypes: [
            ConditionBuilderVariable_1.ConditionBuilderVariableType.INTEGER,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.REAL,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.DATE_TIME,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.DATE,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.TIME,
        ],
        getLiteralType(variableType) {
            return variableType;
        },
    },
    [Config_1.Operator.LESSTHAN_EQUALS]: {
        getRule(condition, variables) {
            const notMatch = /^! \( .+ \)$/;
            const isNot = condition.match(notMatch);
            if (!isNot)
                return null;
            condition = condition.slice(4, condition.length - 2);
            const regexp = new RegExp(`^${VAR_REGEX_SRC} % \\[ .+ - \\* ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' % ');
                const variableName = split[0];
                const operator = Config_1.Operator.LESSTHAN_EQUALS;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: '<=',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `! ( ${variableName} % [ ${utils_1.getLiteralStringValue(rhs)} - * ] )`;
        },
        supportedTypes: [
            ConditionBuilderVariable_1.ConditionBuilderVariableType.INTEGER,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.REAL,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.DATE_TIME,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.DATE,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.TIME,
        ],
        getLiteralType(variableType) {
            return variableType;
        },
    },
    [Config_1.Operator.CONTAINS_STRING]: {
        getRule(condition, variables) {
            const regexp = new RegExp(`^#[\\-_0-9a-zA-Z @.]+# % ${VAR_REGEX_SRC}$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' % ');
                const variableName = split[1];
                const operator = Config_1.Operator.CONTAINS_STRING;
                const literal = split[0];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'contains',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `${utils_1.getLiteralStringValue(rhs)} % ${variableName}`;
        },
        supportedTypes: [ConditionBuilderVariable_1.ConditionBuilderVariableType.STRING],
        getLiteralType(variableType) {
            return variableType;
        },
    },
    [Config_1.Operator.NOT_CONTAINS_STRING]: {
        getRule(condition, variables) {
            const notMatch = /^! \( .+ \)$/;
            const isNot = condition.match(notMatch);
            if (!isNot)
                return null;
            condition = condition.slice(4, condition.length - 2);
            const regexp = new RegExp(`^#[\\-_0-9a-zA-Z @.]+# % ${VAR_REGEX_SRC}$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' % ');
                const variableName = split[1];
                const operator = Config_1.Operator.NOT_CONTAINS_STRING;
                const literal = split[0];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'does not contain',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `${utils_1.getLiteralStringValue(rhs)} % ${variableName}`;
        },
        supportedTypes: [ConditionBuilderVariable_1.ConditionBuilderVariableType.STRING],
        getLiteralType(variableType) {
            return variableType;
        },
    },
    [Config_1.Operator.CONTAINS_ANY_OF]: {
        getRule(condition, variables) {
            const regexp = new RegExp(`${VAR_REGEX_SRC} <% \\[ .+ ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' <% ');
                const variableName = split[0];
                const operator = Config_1.Operator.CONTAINS_ANY_OF;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'contains any of',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `${variableName} <% ${utils_1.getLiteralStringValue(rhs)}`;
        },
        supportedTypes: [
            ConditionBuilderVariable_1.ConditionBuilderVariableType.LIST_INTEGER,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.LIST_STRING,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.LIST_REAL,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.CATEGORY_MULTIPLE,
        ],
        getLiteralType(variableType) {
            if (variableType === ConditionBuilderVariable_1.ConditionBuilderVariableType.CATEGORY_MULTIPLE) {
                return ConditionBuilderVariable_1.ConditionBuilderVariableType.LIST_STRING;
            }
            return variableType;
        },
    },
    [Config_1.Operator.NOT_CONTAINS_ANY_OF]: {
        getRule(condition, variables) {
            const notMatch = /^! \( .+ \)$/;
            const isNot = condition.match(notMatch);
            if (!isNot)
                return null;
            condition = condition.slice(4, condition.length - 2);
            const regexp = new RegExp(`${VAR_REGEX_SRC} <% \\[ .+ ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' <% ');
                const variableName = split[0];
                const operator = Config_1.Operator.NOT_CONTAINS_ANY_OF;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'does not contain any of',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `! ( ${variableName} <% ${utils_1.getLiteralStringValue(rhs)} )`;
        },
        supportedTypes: [
            ConditionBuilderVariable_1.ConditionBuilderVariableType.LIST_INTEGER,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.LIST_STRING,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.LIST_REAL,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.CATEGORY_MULTIPLE,
        ],
        getLiteralType(variableType) {
            if (variableType === ConditionBuilderVariable_1.ConditionBuilderVariableType.CATEGORY_MULTIPLE) {
                return ConditionBuilderVariable_1.ConditionBuilderVariableType.LIST_STRING;
            }
            return variableType;
        },
    },
    [Config_1.Operator.CONTAINS_ALL_OF]: {
        getRule(condition, variables) {
            const regexp = new RegExp(`${VAR_REGEX_SRC} %> \\[ .+ ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' %> ');
                const variableName = split[0];
                const operator = Config_1.Operator.CONTAINS_ALL_OF;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'contains all of',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `${variableName} %> ${utils_1.getLiteralStringValue(rhs)}`;
        },
        supportedTypes: [
            ConditionBuilderVariable_1.ConditionBuilderVariableType.LIST_INTEGER,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.LIST_STRING,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.LIST_REAL,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.CATEGORY_MULTIPLE,
        ],
        getLiteralType(variableType) {
            if (variableType === ConditionBuilderVariable_1.ConditionBuilderVariableType.CATEGORY_MULTIPLE) {
                return ConditionBuilderVariable_1.ConditionBuilderVariableType.LIST_STRING;
            }
            return variableType;
        },
    },
    [Config_1.Operator.NOT_CONTAINS_ALL_OF]: {
        getRule(condition, variables) {
            const notMatch = /^! \( .+ \)$/;
            const isNot = condition.match(notMatch);
            if (!isNot)
                return null;
            condition = condition.slice(4, condition.length - 2);
            const regexp = new RegExp(`${VAR_REGEX_SRC} %> \\[ .+ ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' %> ');
                const variableName = split[0];
                const operator = Config_1.Operator.NOT_CONTAINS_ALL_OF;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'does not contain all of',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `! ( ${variableName} %> ${utils_1.getLiteralStringValue(rhs)} )`;
        },
        supportedTypes: [
            ConditionBuilderVariable_1.ConditionBuilderVariableType.LIST_INTEGER,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.LIST_STRING,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.LIST_REAL,
            ConditionBuilderVariable_1.ConditionBuilderVariableType.CATEGORY_MULTIPLE,
        ],
        getLiteralType(variableType) {
            if (variableType === ConditionBuilderVariable_1.ConditionBuilderVariableType.CATEGORY_MULTIPLE) {
                return ConditionBuilderVariable_1.ConditionBuilderVariableType.LIST_STRING;
            }
            return variableType;
        },
    },
    [Config_1.Operator.EXACTEQUAL]: {
        getRule(condition, variables) {
            const regexp = new RegExp(`^${VAR_REGEX_SRC} === \\[ .+ ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' === ');
                const variableName = split[0];
                const operator = Config_1.Operator.EXACTEQUAL;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'is',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `${variableName} === ${utils_1.getLiteralStringValue(rhs)}`;
        },
        supportedTypes: [ConditionBuilderVariable_1.ConditionBuilderVariableType.CATEGORY_RANKING],
        getLiteralType(variableType) {
            return variableType;
        },
    },
    [Config_1.Operator.NOT_EXACTEQUAL]: {
        getRule(condition, variables) {
            const notMatch = /^! \( .+ \)$/;
            const isNot = condition.match(notMatch);
            if (!isNot)
                return null;
            condition = condition.slice(4, condition.length - 2);
            const regexp = new RegExp(`^${VAR_REGEX_SRC} === \\[ .+ ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' === ');
                const variableName = split[0];
                const operator = Config_1.Operator.NOT_EXACTEQUAL;
                const literal = split[1];
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'is not',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            return `! ( ${variableName} === ${utils_1.getLiteralStringValue(rhs)} )`;
        },
        supportedTypes: [ConditionBuilderVariable_1.ConditionBuilderVariableType.CATEGORY_RANKING],
        getLiteralType(variableType) {
            return variableType;
        },
    },
    [Config_1.Operator.RANKING_ORDER]: {
        getRule(condition, variables) {
            const regexp = new RegExp(`^#\\d+# < #\\d+# % ${VAR_REGEX_SRC}`);
            const matches = condition.match(regexp);
            if (matches) {
                let split = condition.split(' < ');
                split = [split[0], ...split[1].split(' % ')];
                const variableName = split[2];
                const operator = Config_1.Operator.RANKING_ORDER;
                const literal = `${split[0]}:${split[1]}`;
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'has order',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            const [option1, option2] = utils_1.getLiteralStringValue(rhs).split(':');
            return `${option1} < ${option2} % ${variableName}`;
        },
        supportedTypes: [ConditionBuilderVariable_1.ConditionBuilderVariableType.CATEGORY_RANKING],
        getLiteralType() {
            return 'RANKING_ORDER';
        },
    },
    [Config_1.Operator.NOT_RANKING_ORDER]: {
        getRule(condition, variables) {
            const notMatch = /^! \( .+ \)$/;
            const isNot = condition.match(notMatch);
            if (!isNot)
                return null;
            condition = condition.slice(4, condition.length - 2);
            const regexp = new RegExp(`^#\\d+# < #\\d+# % ${VAR_REGEX_SRC}`);
            const matches = condition.match(regexp);
            if (matches) {
                let split = condition.split(' < ');
                split = [split[0], ...split[1].split(' % ')];
                const variableName = split[2];
                const operator = Config_1.Operator.NOT_RANKING_ORDER;
                const literal = `${split[0]}:${split[1]}`;
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: "doesn't have order",
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            const [option1, option2] = utils_1.getLiteralStringValue(rhs).split(':');
            return `! ( ${option1} < ${option2} % ${variableName} )`;
        },
        supportedTypes: [ConditionBuilderVariable_1.ConditionBuilderVariableType.CATEGORY_RANKING],
        getLiteralType() {
            return 'RANKING_ORDER';
        },
    },
    [Config_1.Operator.RANKING_POSITION]: {
        getRule(condition, variables) {
            const regexp = new RegExp(`^${VAR_REGEX_SRC} <|> #\\d# \\[ #[\\-_0-9a-zA-Z @.]+# (, #[\\-_0-9a-zA-Z @.]+#)* ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' <|> ');
                const variableName = split[0];
                const literal = split[1];
                const operator = Config_1.Operator.RANKING_POSITION;
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: 'has',
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            const stringValue = utils_1.getLiteralStringValue(rhs);
            return `${variableName} <|> ${stringValue}`;
        },
        supportedTypes: [ConditionBuilderVariable_1.ConditionBuilderVariableType.CATEGORY_RANKING],
        getLiteralType() {
            return 'RANKING_POSITION';
        },
    },
    [Config_1.Operator.NOT_RANKING_POSITION]: {
        getRule(condition, variables) {
            const notMatch = /^! \( .+ \)$/;
            const isNot = condition.match(notMatch);
            if (!isNot)
                return null;
            condition = condition.slice(4, condition.length - 2);
            const regexp = new RegExp(`^${VAR_REGEX_SRC} <|> #\\d# \\[ #[\\-_0-9a-zA-Z @.]+# (, #[\\-_0-9a-zA-Z @.]+#)* ]$`);
            const matches = condition.match(regexp);
            if (matches) {
                const split = condition.split(' <|> ');
                const variableName = split[0];
                const literal = split[1];
                const operator = Config_1.Operator.NOT_RANKING_POSITION;
                return getRule(variables, variableName, operator, this, literal);
            }
            return null;
        },
        label: "doesn't have",
        getCondition: (variableName, otherParts) => {
            const rhs = otherParts[0];
            const stringValue = utils_1.getLiteralStringValue(rhs);
            return `${variableName} <|> ${stringValue}`;
        },
        supportedTypes: [ConditionBuilderVariable_1.ConditionBuilderVariableType.CATEGORY_RANKING],
        getLiteralType() {
            return 'RANKING_POSITION';
        },
    },
};
//# sourceMappingURL=OperatorConfigMap.js.map