"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateExpression = exports.convertConditionToHTML = exports.sanitizeExpression = exports.convertConditionToExpression = exports.getEmptyGroup = exports.getEmptyRule = exports.getNextRuleParts = exports.getExpressionCondition = exports.getRuleCondition = exports.getLiteralStringValue = void 0;
const tslib_1 = require("tslib");
const RulePart_1 = require("./RulePart");
const Expression_1 = require("./Expression");
const Config_1 = require("./Config");
const OperatorConfigMap_1 = require("./OperatorConfigMap");
const moment_1 = tslib_1.__importDefault(require("moment"));
const constants_1 = require("../../../constants");
const rulePartUtils_1 = require("./rulePartUtils");
const ConditionBuilderVariable_1 = require("../ConditionBuilderVariable");
const utils_1 = require("../../../utils");
function getLiteralStringValue(part) {
    const partValue = part === null || part === void 0 ? void 0 : part.value;
    if (!part || part.type !== RulePart_1.RulePartType.LITERAL) {
        return Config_1.BLANK_MARKER;
    }
    if (typeof part.value === 'string' || partValue instanceof Date) {
        if (part.conversion === 'string') {
            return partValue ? `#${partValue}#` : Config_1.BLANK_MARKER;
        }
        else if (part.conversion === 'boolean') {
            return partValue ? `#${partValue}#` : Config_1.BLANK_MARKER;
        }
        else if (part.conversion === 'number') {
            return partValue ? partValue.toString() : Config_1.BLANK_MARKER;
        }
        else if (part.conversion === 'date') {
            const dateValue = partValue;
            return `${dateValue.getTime()}`;
        }
    }
    else if (Array.isArray(partValue)) {
        if (partValue.length === 0) {
            return `[ ${Config_1.BLANK_MARKER} ]`;
        }
        else if (part.conversion === 'string') {
            const strArray = partValue;
            const options = strArray
                .map((it) => (it ? `#${it}#` : Config_1.BLANK_MARKER))
                .join(' , ');
            return `[ ${options} ]`;
        }
        else if (part.conversion === 'number') {
            const strArray = partValue;
            const options = strArray
                .map((it) => (it ? it.toString() : Config_1.BLANK_MARKER))
                .join(' , ');
            return `[ ${options} ]`;
        }
    }
    else if (typeof partValue === 'object' && partValue.type === 'range') {
        const value = partValue;
        return `[ ${value.min || Config_1.BLANK_MARKER} - ${value.max || Config_1.BLANK_MARKER} ]`;
    }
    else if (typeof partValue === 'object' &&
        partValue.type === 'ranking_order') {
        const option1 = partValue.option1;
        const option2 = partValue.option2;
        return partValue.reverse
            ? `#${option2}#:#${option1}#`
            : `#${option1}#:#${option2}#`;
    }
    else if (typeof partValue === 'object' &&
        partValue.type === 'ranking_position') {
        const option = partValue.option;
        const positions = partValue.positions.join(', ');
        return `#${option}# [ ${positions} ]`;
    }
    return Config_1.BLANK_MARKER;
}
exports.getLiteralStringValue = getLiteralStringValue;
function getRuleCondition(rule) {
    const parts = rule.parts;
    if (parts.length === 0) {
        return Config_1.BLANK_MARKER;
    }
    if (parts.length === 1) {
        const part = parts[0];
        if (part.type === RulePart_1.RulePartType.VARIABLE && part.value) {
            return part.value;
        }
    }
    if (parts.length >= 2) {
        const variablePart = parts[0];
        const operatorPart = parts[1];
        if (variablePart.type === RulePart_1.RulePartType.VARIABLE && variablePart.value) {
            if (operatorPart.type === RulePart_1.RulePartType.OPERATOR && operatorPart.value) {
                const operator = operatorPart.value;
                const operatorConfig = OperatorConfigMap_1.OperatorConfigMap[operator];
                const condition = operatorConfig.getCondition(variablePart.value, parts.slice(2));
                return condition !== null && condition !== void 0 ? condition : Config_1.BLANK_MARKER;
            }
            else {
                return `${variablePart.value} ${Config_1.BLANK_MARKER}`;
            }
        }
    }
    return Config_1.BLANK_MARKER;
}
exports.getRuleCondition = getRuleCondition;
function getExpressionCondition(expression) {
    if ((expression === null || expression === void 0 ? void 0 : expression.type) === Expression_1.ExpressionType.GROUP) {
        const conjunction = expression.conjunction;
        const conjunctionConfig = Config_1.ConjunctionConfigMap[conjunction];
        return conjunctionConfig.getCombinedCondition(expression.subExpressions.map(getExpressionCondition));
    }
    if ((expression === null || expression === void 0 ? void 0 : expression.type) === Expression_1.ExpressionType.RULE) {
        return getRuleCondition(expression);
    }
    return Config_1.BLANK_MARKER;
}
exports.getExpressionCondition = getExpressionCondition;
function getNextRuleParts(partialRule, variables) {
    if (partialRule.parts.length === 0) {
        return [
            {
                type: RulePart_1.RulePartType.VARIABLE,
                isAnswerBased: false,
                value: undefined,
            },
        ];
    }
    const variablePart = partialRule.parts[0];
    const variableName = variablePart.value;
    const variable = variables.find((item) => item.variableName === variableName);
    if (!variableName || !variable) {
        return [];
    }
    const variableType = variable.variableType;
    if (partialRule.parts.length === 1) {
        return [
            {
                type: RulePart_1.RulePartType.OPERATOR,
                allowedOperators: Config_1.operators.filter((operator) => {
                    const supportedTypes = OperatorConfigMap_1.OperatorConfigMap[operator].supportedTypes;
                    return supportedTypes.includes(variableType);
                }),
                value: undefined,
                variable: variable,
            },
        ];
    }
    const operatorPart = partialRule.parts[1];
    const operator = operatorPart.value;
    if (!operator) {
        return [];
    }
    const operatorConfig = OperatorConfigMap_1.OperatorConfigMap[operator];
    if (partialRule.parts.length === 2 &&
        operatorConfig.supportedTypes.includes(variableType)) {
        return [rulePartUtils_1.getLiteralRulePart(operatorConfig, variable)];
    }
    return [];
}
exports.getNextRuleParts = getNextRuleParts;
function getEmptyRule() {
    return {
        type: Expression_1.ExpressionType.RULE,
        parts: [
            {
                type: RulePart_1.RulePartType.VARIABLE,
                isAnswerBased: false,
                value: undefined,
            },
        ],
    };
}
exports.getEmptyRule = getEmptyRule;
function getEmptyGroup() {
    return {
        type: Expression_1.ExpressionType.GROUP,
        subExpressions: [],
        conjunction: Object.keys(Config_1.Conjunction)[0],
    };
}
exports.getEmptyGroup = getEmptyGroup;
const AND_MARKER = '&';
const OR_MARKER = '||';
const NOT_MARKER = '! (';
function divideConditionToParts(condition) {
    const trimmedCondition = condition.trim();
    const returnValue = [];
    let starting = 0;
    let numParenthesis = 0;
    let includeCloseParenthesis = false;
    let spreadChildren = false;
    // operators with negation start with '! ('
    // note single space after !
    if (trimmedCondition.startsWith('! (')) {
        return [trimmedCondition];
    }
    for (let i = 0; i < trimmedCondition.length; i++) {
        const char = trimmedCondition[i];
        if (numParenthesis === 0) {
            if (char === NOT_MARKER[0] &&
                trimmedCondition[i + 1] === NOT_MARKER[1] &&
                trimmedCondition[i + 2] === NOT_MARKER[2]) {
                starting = i;
                numParenthesis += 1;
                i += 2;
                includeCloseParenthesis = true;
                spreadChildren = true;
            }
            if (char === '(') {
                returnValue.push(trimmedCondition.slice(starting, i));
                starting = i + 1;
            }
            if (char === AND_MARKER) {
                returnValue.push(trimmedCondition.slice(starting, i));
                returnValue.push('&');
                starting = i + 1;
            }
            if (char === OR_MARKER[0] && trimmedCondition[i + 1] === OR_MARKER[1]) {
                returnValue.push(trimmedCondition.slice(starting, i));
                returnValue.push('||');
                starting = i + 2;
            }
        }
        if (char === '(') {
            numParenthesis += 1;
        }
        if (char === ')') {
            numParenthesis -= 1;
            if (numParenthesis === 0) {
                const children = divideConditionToParts(trimmedCondition.slice(starting, includeCloseParenthesis ? i + 1 : i));
                if (spreadChildren) {
                    children.forEach((child) => returnValue.push(child));
                }
                else {
                    returnValue.push(children);
                }
                includeCloseParenthesis = false;
                spreadChildren = false;
                starting = i + 1;
            }
        }
    }
    returnValue.push(trimmedCondition.slice(starting, trimmedCondition.length));
    return returnValue
        .map((it) => {
        if (typeof it === 'string') {
            return it.trim();
        }
        return it;
    })
        .filter((it) => it);
}
function getRuleFromCondition(condition, variables) {
    for (let i = 0; i < Config_1.operators.length; i++) {
        const operator = Config_1.operators[i];
        const rule = OperatorConfigMap_1.OperatorConfigMap[operator].getRule(condition, variables);
        if (rule) {
            return rule;
        }
    }
    return getEmptyRule();
}
function convertConditionPartsToExpression(conditionParts, variables) {
    if (typeof conditionParts === 'string') {
        return getRuleFromCondition(conditionParts, variables);
    }
    if (conditionParts.length === 0) {
        return getEmptyRule();
    }
    let conjunction;
    if (conditionParts.length === 1) {
        conjunction = Config_1.Conjunction.AND;
    }
    else {
        conjunction = conditionParts[1] === '&' ? Config_1.Conjunction.AND : Config_1.Conjunction.OR;
    }
    const newConditionParts = conditionParts.filter((it) => it !== '&' && it !== '||');
    return {
        type: Expression_1.ExpressionType.GROUP,
        subExpressions: newConditionParts
            .map((it) => convertConditionPartsToExpression(it, variables))
            .filter((it) => it),
        conjunction,
    };
}
function removeOuterParenthesis(condition) {
    let returnValue = condition.trim();
    if (returnValue.startsWith('(')) {
        returnValue = returnValue.slice(1).trim();
    }
    if (returnValue.endsWith(')')) {
        returnValue = returnValue.slice(0, returnValue.length - 1).trim();
    }
    return returnValue;
}
function convertConditionToExpression(condition, variables) {
    const conditionParts = divideConditionToParts(removeOuterParenthesis(condition));
    const expression = convertConditionPartsToExpression(conditionParts, variables);
    if (!expression || expression.type !== Expression_1.ExpressionType.GROUP) {
        return {
            type: Expression_1.ExpressionType.GROUP,
            conjunction: Config_1.Conjunction.AND,
            subExpressions: expression ? [expression] : [],
        };
    }
    return expression;
}
exports.convertConditionToExpression = convertConditionToExpression;
function sanitizeExpression(expression, variables) {
    if (expression.type === Expression_1.ExpressionType.GROUP) {
        const groupParts = expression.subExpressions.map((it) => sanitizeExpression(it, variables));
        if (groupParts.length === 0) {
            return '<span class="border-gray-700 border border-red-500 text-transparent" style=" display: inline-block;">blank</span>';
        }
        const stringRep = groupParts.join(` ${Config_1.ConjunctionConfigMap[expression.conjunction].label} `);
        return `( ${stringRep} )`;
    }
    const variable = variables.find((variable) => { var _a; return variable.variableName === ((_a = expression.parts[0]) === null || _a === void 0 ? void 0 : _a.value); });
    const parts = expression.parts.map((part) => {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o;
        if (part.type === RulePart_1.RulePartType.VARIABLE) {
            return ((_a = part.value) !== null && _a !== void 0 ? _a : Config_1.BLANK_MARKER).replace(constants_1.CONTEXT_VARIABLE_REGEX, (varName) => {
                var _a;
                const variableLabel = (_a = variable === null || variable === void 0 ? void 0 : variable.displayVariableName) !== null && _a !== void 0 ? _a : varName.slice(1);
                return `<span class="bg-green-400 text-white rounded-lg px-4">${variableLabel}</span>`;
            });
        }
        if (part.type === RulePart_1.RulePartType.OPERATOR) {
            const operator = part.value;
            return operator ? OperatorConfigMap_1.OperatorConfigMap[operator].label : Config_1.BLANK_MARKER;
        }
        if (part.type === RulePart_1.RulePartType.LITERAL) {
            if (typeof part.value === 'string') {
                if (part.literalType === ConditionBuilderVariable_1.ConditionBuilderVariableType.BOOLEAN) {
                    if (part.value === 'true') {
                        return 'Yes';
                    }
                    else {
                        return 'No';
                    }
                }
                if (variable &&
                    part.value &&
                    variable.categoryValues &&
                    !utils_1.isStringArray(variable.categoryValues)) {
                    const categoryValue = variable.categoryValues.find((it) => it.value === part.value);
                    if (variable.isCorrectableNodeType) {
                        return (_b = categoryValue === null || categoryValue === void 0 ? void 0 : categoryValue.label) !== null && _b !== void 0 ? _b : part.value;
                    }
                    return (_c = categoryValue === null || categoryValue === void 0 ? void 0 : categoryValue.label) !== null && _c !== void 0 ? _c : Config_1.BLANK_MARKER;
                }
                return part.value ? `${part.value}` : Config_1.BLANK_MARKER;
            }
            else if (part.value instanceof Date) {
                if (part.literalType === ConditionBuilderVariable_1.ConditionBuilderVariableType.DATE) {
                    return moment_1.default(part.value).format('DD/MM/yyyy');
                }
                else if (part.literalType === ConditionBuilderVariable_1.ConditionBuilderVariableType.TIME) {
                    return moment_1.default(part.value).format('HH:mm');
                }
                else if (part.literalType === ConditionBuilderVariable_1.ConditionBuilderVariableType.DATE_TIME) {
                    return moment_1.default(part.value).format('DD/MM/yyyy HH:mm');
                }
            }
            else if (Array.isArray(part.value)) {
                let value = part.value;
                if (utils_1.isStringArray(value) &&
                    variable &&
                    part.value &&
                    variable.categoryValues &&
                    !utils_1.isStringArray(variable.categoryValues)) {
                    value = value.map((valueItem) => {
                        var _a, _b;
                        const categoryValue = variable.categoryValues.find((it) => it.value === valueItem);
                        if (variable.isCorrectableNodeType) {
                            return (_a = categoryValue === null || categoryValue === void 0 ? void 0 : categoryValue.label) !== null && _a !== void 0 ? _a : valueItem;
                        }
                        return (_b = categoryValue === null || categoryValue === void 0 ? void 0 : categoryValue.label) !== null && _b !== void 0 ? _b : Config_1.BLANK_MARKER;
                    });
                }
                const options = value.map((item) => {
                    if (typeof item === 'string') {
                        return item ? `${item}` : Config_1.BLANK_MARKER;
                    }
                    return '';
                });
                return `[ ${options.join(', ')} ]`;
            }
            else if (typeof part.value === 'object') {
                if (part.value.type === 'range') {
                    const value = part.value;
                    const min = new Date(parseInt(part.value.min));
                    const max = new Date(parseInt(part.value.max));
                    if (part.variableType === ConditionBuilderVariable_1.ConditionBuilderVariableType.DATE) {
                        const minStr = part.value.min
                            ? moment_1.default(min).format('DD/MM/yyyy')
                            : Config_1.BLANK_MARKER;
                        const maxStr = part.value.max
                            ? moment_1.default(max).format('DD/MM/yyyy')
                            : Config_1.BLANK_MARKER;
                        return `${minStr} - ${maxStr}`;
                    }
                    else if (part.variableType === ConditionBuilderVariable_1.ConditionBuilderVariableType.TIME) {
                        const minStr = part.value.min
                            ? moment_1.default(min).format('HH:mm')
                            : Config_1.BLANK_MARKER;
                        const maxStr = part.value.max
                            ? moment_1.default(max).format('HH:mm')
                            : Config_1.BLANK_MARKER;
                        return `${minStr} - ${maxStr}`;
                    }
                    else if (part.variableType === ConditionBuilderVariable_1.ConditionBuilderVariableType.DATE_TIME) {
                        const minStr = part.value.min
                            ? moment_1.default(min).format('DD/MM/yyyy HH:mm')
                            : Config_1.BLANK_MARKER;
                        const maxStr = part.value.max
                            ? moment_1.default(max).format('DD/MM/yyyy HH:mm')
                            : Config_1.BLANK_MARKER;
                        return `${minStr} - ${maxStr}`;
                    }
                    return `${value.min || Config_1.BLANK_MARKER} - ${value.max || Config_1.BLANK_MARKER}`;
                }
                if (part.value.type === 'ranking_order') {
                    const value = part.value;
                    let suggestions;
                    if (!part.suggestions ||
                        ((_d = part.suggestions) === null || _d === void 0 ? void 0 : _d.length) === 0 ||
                        typeof part.suggestions[0] === 'string') {
                        suggestions = (_f = (_e = part.suggestions) === null || _e === void 0 ? void 0 : _e.map((suggestion) => ({
                            value: suggestion,
                            label: suggestion,
                        }))) !== null && _f !== void 0 ? _f : [];
                    }
                    else {
                        suggestions = part.suggestions;
                    }
                    const option1 = suggestions === null || suggestions === void 0 ? void 0 : suggestions.find((suggestion) => {
                        return suggestion.value === value.option1;
                    });
                    const option2 = suggestions === null || suggestions === void 0 ? void 0 : suggestions.find((suggestion) => {
                        return suggestion.value === value.option2;
                    });
                    const verb = value.reverse ? 'after' : 'before';
                    const option1Str = (_g = option1 === null || option1 === void 0 ? void 0 : option1.label) !== null && _g !== void 0 ? _g : Config_1.BLANK_MARKER;
                    const option2Str = (_h = option2 === null || option2 === void 0 ? void 0 : option2.label) !== null && _h !== void 0 ? _h : Config_1.BLANK_MARKER;
                    return `"${option1Str}" ${verb} "${option2Str}"`;
                }
                if (part.value.type === 'ranking_position') {
                    const value = part.value;
                    let suggestions;
                    if (!part.suggestions ||
                        ((_j = part.suggestions) === null || _j === void 0 ? void 0 : _j.length) === 0 ||
                        typeof part.suggestions[0] === 'string') {
                        suggestions = (_l = (_k = part.suggestions) === null || _k === void 0 ? void 0 : _k.map((suggestion) => ({
                            value: suggestion,
                            label: suggestion,
                        }))) !== null && _l !== void 0 ? _l : [];
                    }
                    else {
                        suggestions = part.suggestions;
                    }
                    const option = suggestions === null || suggestions === void 0 ? void 0 : suggestions.find((suggestion) => {
                        return suggestion.value === value.option;
                    });
                    const positionsStr = value.positions.map((it) => `${it}`).join(', ') || Config_1.BLANK_MARKER;
                    const optionStr = (_m = option === null || option === void 0 ? void 0 : option.label) !== null && _m !== void 0 ? _m : Config_1.BLANK_MARKER;
                    return `"${optionStr}" at one of positions [ ${positionsStr} ]"`;
                }
            }
        }
        return ((_o = part.value) === null || _o === void 0 ? void 0 : _o.toString()) || Config_1.BLANK_MARKER;
    });
    return `${parts.join(' ')}`.replace(/\$#blank#\$/g, '<span class="border-gray-700 border border-red-500 text-transparent" ' +
        'style=" display: inline-block;">blank</span>');
}
exports.sanitizeExpression = sanitizeExpression;
/**
 * This function can be used to get the HTML representation of condition
 * This should only be used when condition builder is not also part of the UI
 * If condition builder is part of the UI, it has a callback which gives the HTML representation
 * Note: This is a costly operation. Don't call in every render. Memoize the result
 * @param condition The backend representation of the condition
 * @param variables List of variables in condition (Keep in mind if all the variables used in condition
 * are not sent, the HTML might not be correct)
 */
function convertConditionToHTML(condition, variables) {
    const actVariables = variables.map((it) => (Object.assign(Object.assign({}, it), { variableName: `$${it.variableName}` })));
    const expression = convertConditionToExpression(condition, actVariables);
    return sanitizeExpression(expression, actVariables);
}
exports.convertConditionToHTML = convertConditionToHTML;
function isValueInSuggestions(suggestions, value) {
    return (suggestions.findIndex((suggestion) => {
        if (typeof suggestion === 'string') {
            return value === suggestion;
        }
        return value === suggestion.value;
    }) !== -1);
}
function validateExpression(expression) {
    if (expression.type === Expression_1.ExpressionType.RULE) {
        if (expression.parts.length < 3) {
            return false;
        }
        const literalPart = expression.parts[2];
        if (literalPart.type !== RulePart_1.RulePartType.LITERAL) {
            return false;
        }
        const value = literalPart.value;
        const suggestions = literalPart.suggestions;
        if (value === undefined) {
            return true;
        }
        if (typeof value === 'string') {
            if (suggestions) {
                return isValueInSuggestions(suggestions, value);
            }
            return value !== '';
        }
        if (Array.isArray(value)) {
            if (suggestions) {
                return value.reduce((acc, item) => {
                    return acc && isValueInSuggestions(suggestions, item);
                }, true);
            }
            return true;
        }
        if (value instanceof Date) {
            // can't think of any validations
            return true;
        }
        // it is a min max object for range
        // can be a number range or date range
        // the value is always a number (dates are stored as timestamp converted to string.)
        if (value.type === 'range') {
            try {
                const min = parseInt(value.min);
                const max = parseInt(value.max);
                return min <= max;
            }
            catch (e) {
                //  we are catching because if the user is in the middle of typing, the number might not be valid
                //  hence the expression is invalid
                return false;
            }
        }
        if (value.type === 'ranking_order') {
            return (value.option1 !== value.option2 &&
                isValueInSuggestions(suggestions !== null && suggestions !== void 0 ? suggestions : [], value.option1) &&
                isValueInSuggestions(suggestions !== null && suggestions !== void 0 ? suggestions : [], value.option2));
        }
        if (value.type === 'ranking_position') {
            return (isValueInSuggestions(suggestions !== null && suggestions !== void 0 ? suggestions : [], value.option) &&
                value.positions.reduce((acc, positionStr) => {
                    const pos = parseInt(positionStr);
                    return acc && pos >= 1 && pos <= ((suggestions === null || suggestions === void 0 ? void 0 : suggestions.length) || 0);
                }, true));
        }
        return true;
    }
    // typescript is unable to find the right overload if I don't specify types of the inner function
    return expression.subExpressions.reduce((acc, subExp) => {
        return acc && validateExpression(subExp);
    }, true);
}
exports.validateExpression = validateExpression;
//# sourceMappingURL=utils.js.map