"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.useDragSortableItem = void 0;
const react_dnd_1 = require("react-dnd");
function getPositions(containerRef, monitor) {
    const hoverBoundingRect = containerRef.current.getBoundingClientRect();
    // Get vertical middle
    const targetY = (hoverBoundingRect.bottom - hoverBoundingRect.top) / 2;
    // Determine mouse position
    const clientOffset = monitor.getClientOffset();
    // Get pixels to the top
    const clientY = clientOffset.y - hoverBoundingRect.top;
    return { targetY, clientY };
}
/**
 *
 * @param currentItemIndex {number}
 * index of current item for sorting purposes
 *
 * @param move {function(from:number, to:number): void}
 * moves item from first index to second index
 *
 * @param itemRef {React.Ref} ref to item
 *
 * @param handleRef {React.Ref}
 * ref to handle(the dom element responsible for dragging) usually same as itemRef
 *
 * @param itemType {string} item type to be used by react DnD
 *
 * @returns {boolean} whether something is dragging over current item
 */
function useDragSortableItem(currentItemIndex, move, itemRef, handleRef, itemType) {
    const [, drop] = react_dnd_1.useDrop({
        accept: [itemType],
        hover(item, monitor) {
            if (!itemRef.current) {
                return;
            }
            const clientIndex = item.index;
            const targetIndex = currentItemIndex;
            if (clientIndex === targetIndex) {
                return;
            }
            const { targetY, clientY } = getPositions(itemRef, monitor);
            if ((clientIndex < targetIndex && clientY < targetY) ||
                (clientIndex > targetIndex && clientY > targetY)) {
                return false;
            }
            move(clientIndex, targetIndex);
            item.index = targetIndex;
        },
    });
    const [{ isDraggingOver }, drag, preview] = react_dnd_1.useDrag({
        item: { index: currentItemIndex, type: itemType },
        collect: (monitor) => {
            return {
                isDraggingOver: monitor.isDragging(),
            };
        },
    });
    drop(itemRef);
    drag(handleRef);
    preview(itemRef);
    return isDraggingOver;
}
exports.useDragSortableItem = useDragSortableItem;
//# sourceMappingURL=useDragSortableItem.js.map