<?php

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://talque.com
 * @since      1.0.0
 *
 * @package    Talque_View
 * @subpackage Talque_View/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Talque_View
 * @subpackage Talque_View/includes
 * @author     Volker Braun <volker.braun@real-life-interaction.de>
 */
class Talque_View {
    
    /**
     * The loader that's responsible for maintaining and registering all hooks that power
     * the plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      Talque_View_Loader    $loader    Maintains and registers all hooks for the plugin.
     */
    protected $loader;
    
    /**
     * The unique identifier of this plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string    $plugin_name    The string used to uniquely identify this plugin.
     */
    protected $plugin_name;
    
    /**
     * The current version of the plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string    $version    The current version of the plugin.
     */
    protected $version;

    /**
     * Define the core functionality of the plugin.
     *
     * Set the plugin name and the plugin version that can be used throughout the plugin.
     * Load the dependencies, define the locale, and set the hooks for the admin area and
     * the public-facing side of the site.
     *
     * @since    1.0.0
     */
    public function __construct() {
        $this->version = TALQUE_VIEW_VERSION;
        $this->plugin_name = 'talque-view';
        $this->load_dependencies();
        $this->set_locale();
        $this->define_admin_hooks();
        $this->define_public_hooks();
    }
    
    /**
     * Dependency injection
     */
    private function load_dependencies() {
        $plugin = plugin_dir_path(dirname(__FILE__));
        require_once $plugin . 'includes/class-talque-view-loader.php';
        require_once $plugin . 'includes/class-talque-view-i18n.php';
        require_once $plugin . 'admin/class-talque-view-admin.php';
        require_once $plugin . 'public/class-talque-view-settings.php';
        require_once $plugin . 'public/class-talque-view-api.php';
        require_once $plugin . 'public/class-talque-view-public.php';
        require_once $plugin . 'public/class-talque-view-posts.php';
        require_once $plugin . 'public/class-talque-view-shortcodes.php';
        require_once $plugin . 'public/class-talque-view-filter.php';
        $this->loader = new Talque_View_Loader();
    }

    /**
     * Define the locale for this plugin for internationalization.
     *
     * Uses the Talque_View_i18n class in order to set the domain and to register the hook
     * with WordPress.
     *
     * @since    1.0.0
     * @access   private
     */
    private function set_locale() {
        $plugin_i18n = new Talque_View_i18n();
        $this->loader->add_action('plugins_loaded', $plugin_i18n, 'load_plugin_textdomain');
    }

    /**
     * Register all of the hooks related to the admin area functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_admin_hooks() {
        $plugin_admin = new Talque_View_Admin($this->get_plugin_name(), $this->get_version());
        $this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'admin_enqueue_scripts_hook');
        $this->loader->add_action('admin_init', $plugin_admin, 'admin_init_hook');
        $this->loader->add_action('admin_menu', $plugin_admin, 'admin_menu_hook');
    }

    /**
     * Register all of the hooks related to the public-facing functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_public_hooks() {
        $api = new Talque_View_API();
        $plugin_public = new Talque_View_Public( $this->get_plugin_name(), $this->get_version() );
        $plugin_posts = new Talque_View_Posts( $this->get_plugin_name(), $this->get_version(), $api );
        $plugin_shortcodes = new Talque_View_Shortcodes( $this->get_plugin_name(), $this->get_version(), $api );
        $this->loader->add_action( 'init', $plugin_posts, 'init_hook' );
        $this->loader->add_filter( 'single_template', $plugin_posts, 'single_template_filter', 99 );
        $this->loader->add_filter( 'wpml_is_redirected', $plugin_posts, 'wpml_is_redirected_filter', 10, 3 );
        $this->loader->add_action( 'wp_enqueue_scripts', $plugin_posts, 'wp_enqueue_scripts_hook' );
        $this->loader->add_action( 'updated_option', $plugin_posts, 'updated_option_hook' );
        $this->loader->add_action( 'init', $plugin_shortcodes, 'init_hook' );
    }

    /**
     * Run the loader to execute all of the hooks with WordPress.
     *
     * @since    1.0.0
     */
    public function run() {
        $this->loader->run();               
    }

    /**
     * The name of the plugin used to uniquely identify it within the context of
     * WordPress and to define internationalization functionality.
     *
     * @since     1.0.0
     * @return    string    The name of the plugin.
     */
    public function get_plugin_name() {
        return $this->plugin_name;
    }

    /**
     * The reference to the class that orchestrates the hooks with the plugin.
     *
     * @since     1.0.0
     * @return    Talque_View_Loader    Orchestrates the hooks of the plugin.
     */
    public function get_loader() {
        return $this->loader;
    }

    /**
     * Retrieve the version number of the plugin.
     *
     * @since     1.0.0
     * @return    string    The version number of the plugin.
     */
    public function get_version() {
        return $this->version;
    }

}
