<?php

/**
 * API calls to the Talque backend
 *
 * @link       https://talque.com
 * @since      1.0.0
 *
 * @package    Talque_View
 * @subpackage Talque_View/public
 */


/**
 * Custom Exception for API Error
 *
 * @package    Talque_View
 * @subpackage Talque_View/public
 * @author     Volker Braun <volker.braun@real-life-interaction.de>
 */
class Talque_View_API_Error extends Exception {
}


/**
 * API calls to the Talque backend
 *
 * @package    Talque_View
 * @subpackage Talque_View/public
 * @author     Volker Braun <volker.braun@real-life-interaction.de>
 */
class Talque_View_API {

    /**
     * The default value for $backend
     */
    const DEFAULT_BACKEND = 'PROD';
    
    /**
     * The backend, either 
     */
    private $backend;

    /**
     * Part of the url for the js/scss assets
     */
    private $asset_url;
    
    /**
     * Initialize the class and set its properties.
     */
    public function __construct() {
        $this->backend = get_option( 'talque-backend', Talque_View_API::DEFAULT_BACKEND );
        $this->asset_url = $this->init_asset_url();
        $this->load_dependencies();
    }

    private function init_asset_url() {
        if ( $this->backend === 'DEV' )
            return 'http://localhost:8380';
        $base_url = 'https://storage.googleapis.com/assets.talque.com/plugin/view';
        if ( $this->backend === 'QA' )
            return $base_url;
        $org_id = get_option( 'talque-org-id' );
        if ( $org_id )
            return $base_url . '/' . $org_id;
        return $base_url;
    }
    
    /**
     * Dependency injection
     */
    private function load_dependencies() {
    }

    /**
     * Return the desired locale
     *
     * Either returns 'DE_DE' or 'EN_US', with more locales to be added later.
     */
    private function locale() {
        // type: () -> string
        $locale = get_locale();
        $locale2 = substr( $locale, 0, 2 );  // first two chars
        switch ($locale2) {
            case 'de':
                return 'DE_DE';
            case 'en':
            default:
                return 'EN_US';
        }
    }

    /**
     * Percent-encode the slug, if neccessary
     *
     * This should not be necessary, the slugs are sent percent-encoded from
     * the talque server. But the PHP built-in webserver seems to decode
     * them. A proper lamp server does not decode them.
     *
     * Note that percent characters are not allowed in the (decoded) slug, so we
     * can fix this.
     */
    private function maybe_urlencode( $slug ) {
        $has_percent = ( strpos( $slug, '%' ) !== false );
        if ( $has_percent )
            return $slug;
        // Slug is either decoded or does not need encoding
        return urlencode( $slug );
    }

    public function get_style_url() {
        // type: () -> string
        return $this->asset_url . '/app.css';
    }
    
    public function get_script_url() {
        // type: () -> string
        return $this->asset_url . '/app.js';
    }

    private function make_initial( $post_name, $params ) {
        return $post_name . '?' . http_build_query($params);
    }
    
    private function html_snippet( $initial_route, $settings ) {
        $template =
            '<div'.
            '  class="tq-talque-view"'.
            '  initial="%s"'.
            '  %s'.
            '  backend="%s">'.
            '</div>';
        $html = sprintf(
            $template,
            $initial_route,
            $settings->html_snippet_attributes(),
            $this->backend);
        if ( WP_DEBUG ) {
            error_log($html);
        }
        return $html;
    }
    
    public function speaker_id( $speaker_id, $post_name, $settings ) {
        // type: (string) -> string
        $params = array(
            'speakerId' => $speaker_id,
        );
        $initial = $this->make_initial( $post_name, $params );
        return $this->html_snippet( $initial, $settings );
    }

    public function speaker_slug( $speaker_slug, $post_name, $settings ) {
        // type: (string) -> string
        $params = array(
            'speaker' => $this->maybe_urlencode( $speaker_slug ),
        );
        $initial = $this->make_initial( $post_name, $params );
        return $this->html_snippet( $initial, $settings );
    }

    public function speaker_list( $settings ) {
        return $this->html_snippet( 'speaker-list', $settings );
    }

    public function speaker_grid( $settings ) {
        // type: () -> string
        return $this->html_snippet( 'speaker-grid', $settings );
    }

    public function lecture_id( $lecture_id, $post_name, $settings ) {
        // type: (string) -> string
        $params = array(
            'lectureId' => $lecture_id,
        );
        $initial = $this->make_initial( $post_name, $params );
        return $this->html_snippet( $initial, $settings );
    }

    public function lecture_slug( $lecture_slug, $post_name, $settings ) {
        // type: (string) -> string
        $params = array(
            'lecture' => $this->maybe_urlencode( $lecture_slug ),
        );
        $initial = $this->make_initial( $post_name, $params );
        return $this->html_snippet( $initial, $settings );
    }

    public function lecture_list( $settings ) {
        // type: (?string) -> string
        return $this->html_snippet( 'lecture-list', $settings );
    }

    public function lecture_table( $settings ) {
        // type: (?string) -> string
        return $this->html_snippet( 'lecture-table', $settings );
    }

    public function lecture_grid( $settings ) {
        // type: (?string) -> string
        return $this->html_snippet( 'lecture-grid', $settings );
    }

    public function vendor_id( $vendor_id, $post_name, $settings ) {
        // type: (string) -> string
        $params = array(
            'vendorId' => $vendor_id,
        );
        $initial = $this->make_initial( $post_name, $params );
        return $this->html_snippet( $initial, $settings );
    }

    public function vendor_slug( $vendor_slug, $post_name, $settings ) {
        $params = array(
            'vendor' => $this->maybe_urlencode( $vendor_slug ),
        );
        $initial = $this->make_initial( $post_name, $params );
        return $this->html_snippet( $initial, $settings );
    }

    public function sponsor_grid( $settings ) {
        // type: (?Talque_View_Settings) -> string
        return $this->html_snippet( 'sponsor-grid', $settings );
    }

    public function exhibitor_grid( $settings ) {
        return $this->html_snippet( 'exhibitor-grid', $settings );
    }
}
