<?php

/**
 * Custom posts for the talque plugin.
 *
 * @link       https://talque.com
 * @since      1.0.0
 *
 * @package    Talque_View
 * @subpackage Talque_View/admin
 */

/**
 * Custom posts for the talque plugin.
 *
 * Defines the plugin name, version, and the custom post types
 *
 * @package    Talque_View
 * @subpackage Talque_View/public
 * @author     Volker Braun <volker.braun@real-life-interaction.de>
 */
class Talque_View_Posts {

    /**
     * The instance
     *
     * @var      Talque_View_Posts    $instance    The instance this plugin.
     */
    private static $instance;

    /**
     * Get the singleton
     *
     * Use this to access the plugin from templates; When rendering templates,
     * this class has already been instantiated.
     *
     * @return Talque_View_Posts
     */
    public static function get_instance()
    {
        $inst = self::$instance;
        if ( !$inst )
            throw new Error('the talque view plugin has not been instantiated');
        return $inst;
    }
    
    /**
     * The ID of this plugin.
     *
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;

    /**
     * Backend API access
     *
     * @var      Talque_View_API    $instance    The backend API.
     */
    private static $api;

    /**
     * Initialize the class and set its properties.
     *
     * @param      string    $plugin_name       The name of this plugin.
     * @param      string    $version    The version of this plugin.
     */
    public function __construct( $plugin_name, $version, $api ) {
        // Singleton
        if ( self::$instance ) {
            throw new Error('the talque view plugin can only be instantiated once');
        }
        self::$instance = $this;
        // Initialization
        $this->plugin_name = $plugin_name;
        $this->version = $version;
        self::$api = $api;
        $this->load_dependencies();
    }

    /**
     * Dependency injection
     */
    private function load_dependencies() {
    }

    /**
     * Called by the activator when the plugin is activated
     */
    public function activate_plugin_hook() {
        $this->register_custom_post_type();
        $this->insert_custom_posts();
        $this->add_rewrite_rules();
    }

    /**
     * Called by the deactivator when the plugin is deactivated
     */
    public function deactivate_plugin_hook() {
        /*
        $allposts = get_posts( array( 'post_type'=>'talque', 'numberposts' => -1 ) );
        foreach ($allposts as $eachpost) {
            error_log( 'deleting post ' . $eachpost->ID);
            wp_delete_post( $eachpost->ID, true );
        }
        unregister_post_type( 'talque' );
        */
    }
    
    /**
     * Initialize the plugin
     */
    public function init_hook() {
        $this->register_custom_post_type();
        $this->add_rewrite_rules();
        if ( WP_DEBUG ) {
            // For debugging only: flush now (too expensive in production)
            flush_rewrite_rules();
            error_log( 'flush (WP_DEBUG === true)' );
        }
    }
    
    /**
     * Called when options have been saved
     */
    public function updated_option_hook( $option ) {
        // error_log( 'updated_option_hook: ' . $option );
        if ( in_array( $option, array( 'talque-url-lecture', 'talque-url-speaker', 'talque-url-vendor' ) ) ) {
            error_log( 'url rewriting options changed, flushing rules' );
            $this->add_rewrite_rules();
            flush_rewrite_rules();
        }
    }

    /**
     * Prevent redirection
     *
     * WPML mucks about with url in a feeble attempt to translate the query; Abort!
     */
    public function wpml_is_redirected_filter( $redirect, $post_id, $q ) {
        // error_log('wpml_is_redirected_filter: '.json_encode($q) );
        if ( isset( $q->query_vars['post_type'] ) && 'talque' === $q->query_vars['post_type'] )
            return false;
        return $redirect;
    }
    
    /**
     * Register the 'talque' custom post type with Wordpress
     *
     * Note: the rewrite rules do not work if Settings -> Permalink Settings
     * -> Plain is selected (which WP treats as "no rewrite rules").
     */
    private function add_rewrite_rules() {
        add_rewrite_tag( '%viewId%', '([0-9a-zA-Z]+)' );
        add_rewrite_tag( '%orgId%', '([0-9a-zA-Z]+)' );
        add_rewrite_tag( '%speakerId%', '([0-9a-zA-Z]+)' );
        add_rewrite_tag( '%speaker%', '([^/]+)' );
        add_rewrite_tag( '%lectureId%', '([0-9a-zA-Z]+)' );
        add_rewrite_tag( '%lecture%', '([^/]+)' );
        add_rewrite_tag( '%vendorId%', '([0-9a-zA-Z]+)' );
        add_rewrite_tag( '%vendor%', '([^/]+)' );
        # The post_type=talque is not necessary, possibly remnant for supporting old versions?
        add_rewrite_rule( '^talque/sponsors/?$', 'index.php?post_type=talque&talque=sponsor-grid', 'top' );
        add_rewrite_rule( '^talque/exhibitors/?$', 'index.php?post_type=talque&talque=exhibitor-grid', 'top' );
        add_rewrite_rule( '^talque/vendor-id/([0-9a-zA-Z]+)?$', 'index.php?post_type=talque&talque=sponsor-grid&vendorId=$matches[1]', 'top' );
        add_rewrite_rule( '^talque/vendor/([^/]+)/?$', 'index.php?post_type=talque&talque=sponsor-grid&vendor=$matches[1]', 'top' );
        add_rewrite_rule( '^talque/speaker-grid/?$', 'index.php?post_type=talque&talque=speaker-grid', 'top' );
        add_rewrite_rule( '^talque/speaker-list/?$', 'index.php?post_type=talque&talque=speaker-list', 'top' );
        add_rewrite_rule( '^talque/speaker-id/([0-9a-zA-Z]+)/?$', 'index.php?post_type=talque&talque=speaker-list&speakerId=$matches[1]', 'top' );
        add_rewrite_rule( '^talque/speaker/([^/]+)/?$', 'index.php?post_type=talque&talque=speaker-list&speaker=$matches[1]', 'top' );
        add_rewrite_rule( '^talque/lecture-grid/?$', 'index.php?post_type=talque&talque=lecture-grid', 'top' );
        add_rewrite_rule( '^talque/lecture-list/?$', 'index.php?post_type=talque&talque=lecture-list', 'top' );
        add_rewrite_rule( '^talque/lecture-id/([0-9a-zA-Z]+)/?$', 'index.php?post_type=talque&talque=lecture-list&lectureId=$matches[1]', 'top' );
        add_rewrite_rule( '^talque/lecture/([^/]+)/?$', 'index.php?post_type=talque&talque=lecture-list&lecture=$matches[1]', 'top' );
        // Customizable urls
        $speaker = get_option( 'talque-url-speaker' );
        if ( $speaker )
            add_rewrite_rule( '^' . $speaker . '/([^/]+)/?$', 'index.php?post_type=talque&talque=speaker-grid&speaker=$matches[1]', 'top' );
        $lecture = get_option( 'talque-url-lecture' );
        if ( $lecture )
            add_rewrite_rule( '^' . $lecture . '/([^/]+)/?$', 'index.php?post_type=talque&talque=lecture-list&lecture=$matches[1]', 'top' );
        $vendor = get_option( 'talque-url-vendor' );
        if ( $vendor )
            add_rewrite_rule( '^' . $vendor . '/([^/]+)/?$', 'index.php?post_type=talque&talque=sponsor-grid&vendor=$matches[1]', 'top' );
    }

    /**
     * Register the stylesheet and JavaScript for the custom post type
     */
    public function wp_enqueue_scripts_hook() {
        wp_enqueue_style(
            $this->plugin_name,
            self::$api->get_style_url(),
            array(), $this->version, 'all' );
        wp_enqueue_script(
            $this->plugin_name,
            self::$api->get_script_url(),
            array(), $this->version, false );
    }
    
    /**
     * Extends the single_template filter to find the custom post template
     */
    public function single_template_filter( $template ) {
        // type: (string) -> string
        $post_type = get_query_var( 'post_type' );
        if ($post_type != 'talque')
            return $template;
        $plugin = plugin_dir_path( dirname( __FILE__ ) );
        return $plugin . '/public/single-talque.php';
    }

    /**
     * Register the 'talque' custom post type with Wordpress
     */
    private function register_custom_post_type() {
        register_post_type(
            'talque',
            array(
                'labels'      => array(
                    'name'        => __('Talque'),
                ),
                'public'      => true,
                'has_archive' => false,
                'show_ui'     => false,
                'show_in_menu' => false,
                'show_in_nav_menus' => true,
            )
        );
    }
    
    /**
     * Insert the 'talque' custom posts
     */
    private function insert_custom_posts() {
        $speaker_list = $this->get_or_insert_post( 'speaker-list', __('Speaker List') );
        $speaker_grid = $this->get_or_insert_post( 'speaker-grid', __('Speaker Grid') );
        $lecture_list = $this->get_or_insert_post( 'lecture-list', __('Lecture List') );
        $lecture_grid = $this->get_or_insert_post( 'lecture-grid', __('Lecture Grid') );
        $lecture_table = $this->get_or_insert_post( 'lecture-table', __('Lecture Table') );
        $sponsor_grid = $this->get_or_insert_post( 'sponsor-grid', __( 'Sponsors' ) );
        $exhibitor_grid = $this->get_or_insert_post( 'exhibitor-grid', __( 'Exhibitors' ) );
        error_log(
            'talque posts: ' . 
            $speaker_list . '/' .
            $speaker_grid . '/' .
            $lecture_list . '/' .
            $lecture_grid . '/' .
            $lecture_table . '/' .
            $sponsor_grid . '/' .
            $exhibitor_grid
        );
    }

    private function get_or_insert_post( $name, $title ) {
        // type: (string, string) -> int
        $post = get_page_by_title( $title, OBJECT, 'talque' );
        if ( $post )
            $ID = $post->ID;
        else
            $ID = NULL;
        $post_id = wp_insert_post( array(
            'ID' => $ID,
            'post_type' => 'talque',
            'post_name' => $name,
            'post_title' => $title,
            'post_status' => 'publish',
        ) );
        return $post_id;
    }

    /**
     * Return appropriate content by making a backend API call
     *
     * Throws if an error is encountered.
     */
    public function get_content( $name ) {
        // type: (string) -> string
        if ( WP_DEBUG )
            error_log( "get_content $name " . ' org-id=' . get_query_var( 'orgId' ) );
        $org_id = get_query_var( 'orgId', NULL );
        $view_id = get_query_var( 'viewId', NULL );
        $settings = Talque_View_Settings::from_wp_options( $org_id, $view_id );
        # first, handle single pages (multi page urls showing single because of url args)
        $speaker_id = get_query_var( 'speakerId' );
        if ( $speaker_id ) {
            return self::$api->speaker_id( $speaker_id, $name, $settings );
        }
        $speaker_slug = get_query_var( 'speaker' );
        if ( $speaker_slug ) {
            return self::$api->speaker_slug( $speaker_slug, $name, $settings );
        }
        $lecture_id = get_query_var( 'lectureId' );
        if ( $lecture_id ) {
            return self::$api->lecture_id( $lecture_id, $name, $settings );
        }
        $lecture_slug = get_query_var( 'lecture' );
        if ( $lecture_slug ) {
            return self::$api->lecture_slug( $lecture_slug, $name, $settings );
        }
        $vendor_id = get_query_var( 'vendorId' );
        if ( $vendor_id ) {
            return self::$api->vendor_id( $vendor_id, $name, $settings );
        }
        $vendor_slug = get_query_var( 'vendor' );
        if ( $vendor_slug ) {
            return self::$api->vendor_slug( $vendor_slug, $name, $settings );
        }
        # otherwise, handle multiple pages
        switch ( $name ) {
            case 'speaker-list':
                return self::$api->speaker_list( $settings );
            case 'speaker-grid':
                return self::$api->speaker_grid( $settings );
            case 'lecture-list':
                return self::$api->lecture_list( $settings );
            case 'lecture-table':
                return self::$api->lecture_table( $settings );
            case 'lecture-grid':
                return self::$api->lecture_grid( $settings );
            case 'sponsor-grid':
                return self::$api->sponsor_grid( $settings );
            case 'exhibitor-grid':
                return self::$api->exhibitor_grid( $settings );
            default:
                throw new Error( 'not handled: ' . $name );
        }
    }

    /**
     * Return content or an error message
     */
    public function get_content_or_error( $name ) {
        // type: (string) -> string
        try {
            return $this->get_content( $name );
        } catch ( Talque_View_API_Error $error ) {
            return $error->getMessage();
        }
    }
}

