<?php

/**
 * Settings for customizing the appearence of the Talque backend
 *
 * @link       https://talque.com
 * @since      1.0.0
 *
 * @package    Talque_View
 * @subpackage Talque_View/public
 */


/**
 * Settings for customizing the appearence of the Talque backend
 *
 * @package    Talque_View
 * @subpackage Talque_View/public
 * @author     Volker Braun <volker.braun@real-life-interaction.de>
 */
class Talque_View_Settings {

    private $org_id;
    private $view_id;
    private $filter;

    /**
     * Initialize the class and set its properties.
     */
    public function __construct( $org_id, $view_id, $filter ) {
        $this->org_id = $org_id;
        $this->view_id = $view_id;
        $this->filter = $filter;
    }

    /**
     * Construct, maybe taking defaults from wordpress options
     */
    public static function from_wp_options( $org_id, $view_id, $filter=NULL ) {
        if ( $org_id === NULL )
            $org_id = get_option( 'talque-org-id' );
        if ( $view_id === NULL )
            $view_id = get_option( 'talque-view-id' );
        if ( $filter === NULL )
            $filter = Talque_View_Filter::make_empty();
        return new Talque_View_Settings( $org_id, $view_id, $filter );
    }

    private static function from_shortcode( $sanitized ) {
        $org_id = isset($sanitized['org-id']) ? $sanitized['org-id'] : NULL;
        $view_id = isset($sanitized['view-id']) ? $sanitized['view-id'] : NULL;
        $filter = Talque_View_Filter::from_shortcode( $sanitized );
        return Talque_View_Settings::from_wp_options( $org_id, $view_id, $filter );
    }
    
    public static function from_lecture_shortcode( $attrs, $shortcode_tag ) {
        // Note: shortcode attrs are lower cased
        $sanitized = shortcode_atts( array(
            'org-id' => NULL,
            'view-id' => NULL,
            'timezone' => NULL,
            'tags' => NULL,
            'formats' => NULL,
            'topics' => NULL,
        ), $attrs, $shortcode_tag );
        return Talque_View_Settings::from_shortcode( $sanitized );
    }

    public static function from_speaker_shortcode( $attrs, $shortcode_tag ) {
        // Note: shortcode attrs are lower cased
        $sanitized = shortcode_atts( array(
            'org-id' => NULL,
            'view-id' => NULL,
            'timezone' => NULL,
            'badges' => NULL,
        ), $attrs, $shortcode_tag );
        return Talque_View_Settings::from_shortcode( $sanitized );
    }

    public static function from_sponsor_shortcode( $attrs, $shortcode_tag ) {
        // Note: shortcode attrs are lower cased
        $sanitized = shortcode_atts( array(
            'org-id' => NULL,
            'view-id' => NULL,
            'timezone' => NULL,
            'sponsors' => NULL,
        ), $attrs, $shortcode_tag );
        return Talque_View_Settings::from_shortcode( $sanitized );
    }

    public static function from_exhibitor_shortcode( $attrs, $shortcode_tag ) {
        // Note: shortcode attrs are lower cased
        $sanitized = shortcode_atts( array(
            'org-id' => NULL,
            'view-id' => NULL,
            'timezone' => NULL,
            'exhibitors' => NULL,
        ), $attrs, $shortcode_tag );
        return Talque_View_Settings::from_shortcode( $sanitized );
    }

    public function org_id() {
        return $this->org_id;
    }

    public function view_id() {
        return $this->view_id;
    }

    public function lecture_url() {
        $lecture = get_option( 'talque-url-lecture' );
        if ( !$lecture )
            return NULL;
        return get_home_url() . '/' . $lecture . '/{lecture}';
    }
    
    public function speaker_url() {
        $speaker = get_option( 'talque-url-speaker' );
        if ( !$speaker )
            return NULL;
        return get_home_url() . '/' . $speaker . '/{speaker}';
    }
    
    public function vendor_url() {
        $vendor = get_option( 'talque-url-vendor' );
        if ( !$vendor )
            return NULL;
        return get_home_url() . '/' . $vendor . '/{vendor}';
    }

    /**
     * Return the current locale
     *
     * For example, 'DE_DE' or 'EN_US'.
     */
    public function locale() {
        // type: () -> string
        $locale = get_locale();
        $locale2 = substr( $locale, 0, 2 );  // first two chars
        switch ($locale2) {
            case 'de':
                return 'DE_DE';
            case 'fr':
                return 'FR_FR';
            case 'en':
            default:
                return 'EN_US';
        }
    }

    public function html_snippet_attributes() {
        $template = 
            '  org-id="%s"'.
            '  view-id="%s"'.
            '  locale="%s"'.
            '  lecture-url="%s"'.
            '  speaker-url="%s"'.
            '  vendor-url="%s"'.
            '  %s';
        return sprintf(
            $template,
            $this->org_id(),
            $this->view_id(),
            $this->locale(),
            $this->lecture_url(),
            $this->speaker_url(),
            $this->vendor_url(),
            $this->filter->html_snippet_attributes()
        );
    }
}
