// Copyright 2014 The Bazel Authors. All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package com.google.devtools.build.lib.rules.java.proto;

import com.google.auto.value.AutoValue;
import com.google.common.base.Verify;
import com.google.devtools.build.lib.actions.Artifact;
import com.google.devtools.build.lib.analysis.TransitiveInfoProvider;
import com.google.devtools.build.lib.cmdline.Label;
import com.google.devtools.build.lib.collect.nestedset.NestedSet;
import com.google.devtools.build.lib.concurrent.ThreadSafety.Immutable;
import com.google.devtools.build.lib.skyframe.serialization.autocodec.AutoCodec;
import com.google.devtools.build.lib.skyframe.serialization.autocodec.AutoCodec.Strategy;

/**
 * A {@link TransitiveInfoProvider} for {@link Artifact}s created and used to generate the proto
 * extension registry. This provider is used to ensure that if multiple registries are generated
 * from a target, that the top most target produces a registry that is a superset of any child
 * registries.
 */
@AutoCodec(strategy = Strategy.AUTO_VALUE_BUILDER)
@AutoValue
@Immutable
public abstract class GeneratedExtensionRegistryProvider implements TransitiveInfoProvider {

  /** @return the rule label for which this registry was built. */
  public abstract Label getGeneratingRuleLabel();

  /** @return if this registry was generated for lite or full runtime. */
  public abstract boolean isLite();

  /** @return the class jar generated by the registry. */
  public abstract Artifact getClassJar();

  /** @return the source jar generated by the registry. */
  public abstract Artifact getSrcJar();

  /** @return the proto jars used to generate the registry. */
  public abstract NestedSet<Artifact> getInputs();

  public static Builder builder() {
    return new AutoValue_GeneratedExtensionRegistryProvider.Builder();
  }

  /** A builder for {@link GeneratedExtensionRegistryProvider}. */
  @AutoValue.Builder
  public abstract static class Builder {

    /** Sets the rule label for which this registry was built. */
    public abstract Builder setGeneratingRuleLabel(Label label);

    /** Indicates this registry was built for lite runtime if <tt>true</tt>, full otherwise. */
    public abstract Builder setLite(boolean lite);

    /** Sets the class jar containing the generated extension registry. */
    public abstract Builder setClassJar(Artifact classJar);

    /** Sets the source jar containing the generated extension registry. */
    public abstract Builder setSrcJar(Artifact srcJar);

    /** Sets the transitive set of protos used to produce the generated extension registry. */
    public abstract Builder setInputs(NestedSet<Artifact> inputs);

    abstract GeneratedExtensionRegistryProvider autoBuild();
    public GeneratedExtensionRegistryProvider build() {
      GeneratedExtensionRegistryProvider result = autoBuild();
      Verify.verify(!result.getInputs().isEmpty());
      return autoBuild();
    }
  }
}
