// Copyright 2016 The Bazel Authors. All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package com.google.devtools.build.lib.actions;

import com.google.devtools.build.lib.unix.NativePosixSystem;

import java.io.IOException;

/**
 * This class estimates the local host's resource capacity for Darwin.
 */
public class LocalHostResourceManagerDarwin {
  private static final Boolean JNI_UNAVAILABLE =
      "0".equals(System.getProperty("io.bazel.EnableJni"));
  private static final double EFFECTIVE_CPUS_PER_HYPERTHREADED_CPU = 0.6;

  private static int getLogicalCpuCount() throws IOException {
    return (int) NativePosixSystem.sysctlbynameGetLong("hw.logicalcpu");
  }

  private static int getPhysicalCpuCount() throws IOException {
    return (int) NativePosixSystem.sysctlbynameGetLong("hw.physicalcpu");
  }

  private static double getMemoryInMb() throws IOException {
    return NativePosixSystem.sysctlbynameGetLong("hw.memsize") / 1E6;
  }

  public static ResourceSet getLocalHostResources() {
    if (JNI_UNAVAILABLE) {
      return null;
    }
    try {
      int logicalCpuCount = getLogicalCpuCount();
      int physicalCpuCount = getPhysicalCpuCount();
      double ramMb = getMemoryInMb();
      boolean hyperthreading = (logicalCpuCount != physicalCpuCount);

      return ResourceSet.create(
          ramMb,
          logicalCpuCount * (hyperthreading ? EFFECTIVE_CPUS_PER_HYPERTHREADED_CPU : 1.0),
          1.0,
          Integer.MAX_VALUE);
    } catch (IOException e) {
      return null;
    }
  }
}
