#version 330

in vec2 texCoord0;
in float faceAlpha0;  // Now contains packed face flags as float
in vec3 surfaceNormal;
in vec3 toLightVector;
in vec3 FragPos;
in vec4 FragPosLightSpace;
in vec2 cloudTexCoord;  // For cloud shadow sampling
in vec2 fogMapCoord;  // For area fog sampling

out vec4 outputColor;

uniform sampler2D basic_texture;
uniform sampler2D skyTexture;  // Cloud texture for dynamic shadows
uniform bool enable_transparency;
uniform float view_distance;
uniform vec4 distanceColor;

// Balanced lighting for world objects - matched to terrain
uniform float ambientStrength = 0.55;
uniform float diffuseStrength = 0.55;
uniform float time;
uniform sampler2D shadowMap;
uniform vec3 lightDirection = vec3(-0.4, -0.85, -0.4);  // Moderate sun angle (55-60 deg elevation)
uniform bool enableShadows = false;
uniform float rainIntensity = 0.0;  // 0.0 = no rain, 1.0 = heavy rain
uniform bool enableCloudShadows = true;  // Cloud shadow modulation
uniform float visibilityMultiplier = 1.0;  // Time-of-day visibility (1.0 = day, 0.35 = night)
uniform int timeOfDay = 0;  // 0=DAY, 1=DUSK, 2=NIGHT (explicit, not derived from visibility)

// Area fog uniforms
const int MAX_FOG_ZONES = 16;
uniform int numFogZones = 0;
uniform sampler2D fogMapTexture;
uniform vec3 fogColors[MAX_FOG_ZONES];
uniform float fogLimits[MAX_FOG_ZONES];         // FLimit - max fog cap distance
uniform float fogTransparencies[MAX_FOG_ZONES]; // Transp - controls distance falloff rate
uniform float fogMaxOpacities[MAX_FOG_ZONES];
uniform float fogAltitudes[MAX_FOG_ZONES];  // YBegin - fog ceiling height (world units)

// Camera position and terrain dimensions for fog calculation
uniform vec3 cameraPos;
uniform float terrainWidth;
uniform float terrainHeight;
uniform float tileWidth;
uniform float heightmapScale = 4.0;  // 4.0 for C2, 2.0 for C1

// Camera-centric area fog - when player is inside a fog zone, all visible
// fragments are fogged based on distance and vertical position.
const float FOG_ZONE_TRANSITION_RADIUS = 256.0;  // Distance over which fog fades when entering/leaving zones

int sampleFogZone(vec3 worldPos, float terrainWorldWidth, float terrainWorldHeight) {
    vec2 uv = vec2(worldPos.x / terrainWorldWidth, worldPos.z / terrainWorldHeight);
    if (uv.x < 0.0 || uv.x > 1.0 || uv.y < 0.0 || uv.y > 1.0) return -1;
    float fogIndexRaw = texture(fogMapTexture, uv).r * 255.0;
    return int(fogIndexRaw + 0.5) - 1;
}

float estimateFogZoneDepth(vec3 worldPos, int expectedFogIndex, float terrainWorldWidth, float terrainWorldHeight) {
    vec2 uv = vec2(worldPos.x / terrainWorldWidth, worldPos.z / terrainWorldHeight);
    float sameZoneCount = 0.0;
    float totalSamples = 0.0;
    float sampleRadius = FOG_ZONE_TRANSITION_RADIUS;
    vec2 uvRadius = vec2(sampleRadius / terrainWorldWidth, sampleRadius / terrainWorldHeight);

    for (float dist = 0.25; dist <= 1.0; dist += 0.25) {
        for (float angle = 0.0; angle < 6.28; angle += 0.785) {
            vec2 offset = vec2(cos(angle), sin(angle)) * dist;
            vec2 sampleUV = uv + offset * uvRadius;
            if (sampleUV.x >= 0.0 && sampleUV.x <= 1.0 && sampleUV.y >= 0.0 && sampleUV.y <= 1.0) {
                float fogIndexRaw = texture(fogMapTexture, sampleUV).r * 255.0;
                int sampleFogIndex = int(fogIndexRaw + 0.5) - 1;
                totalSamples += 1.0;
                if (sampleFogIndex == expectedFogIndex) {
                    sameZoneCount += 1.0;
                }
            }
        }
    }
    return totalSamples > 0.0 ? sameZoneCount / totalSamples : 1.0;
}

vec3 applyAreaFog(vec3 color, float distFromCamera, vec3 fragWorldPos) {
    if (numFogZones == 0) return color;

    float terrainWorldWidth = terrainWidth * tileWidth;
    float terrainWorldHeight = terrainHeight * tileWidth;

    int cameraFogIndex = sampleFogZone(cameraPos, terrainWorldWidth, terrainWorldHeight);
    if (cameraFogIndex < 0 || cameraFogIndex >= numFogZones) return color;

    float fogCeiling = fogAltitudes[cameraFogIndex];
    float transp = fogTransparencies[cameraFogIndex];  // Already scaled to our world units
    float fogLimit = fogLimits[cameraFogIndex];        // Raw 0-255 value
    vec3 fogColor = fogColors[cameraFogIndex];

    // ========== SIMPLIFIED FOG MODEL ==========
    // Fixed visibility range: 30 tiles (480 world units)
    // Fog accumulates from 0 at player to FLimit/255 at 30 tiles

    float depthA = max(0.0, fogCeiling - cameraPos.y);      // Camera depth in fog
    float depthB = max(0.0, fogCeiling - fragWorldPos.y);   // Fragment depth in fog

    if (depthA <= 0.0 && depthB <= 0.0) return color;

    // Smooth vertical transition for camera crossing ceiling
    float ceilingTransitionHeight = 96.0;
    float cameraVerticalFade = 1.0;
    if (depthA <= 0.0) {
        float aboveAmount = cameraPos.y - fogCeiling;
        cameraVerticalFade = clamp(1.0 - aboveAmount / ceilingTransitionHeight, 0.0, 1.0);
        if (cameraVerticalFade <= 0.0) return color;
    } else if (depthA < ceilingTransitionHeight) {
        cameraVerticalFade = smoothstep(0.0, ceilingTransitionHeight, depthA);
    }

    // Camera zone edge transition
    float cameraZoneDepth = estimateFogZoneDepth(cameraPos, cameraFogIndex, terrainWorldWidth, terrainWorldHeight);
    float zoneTransitionFade = smoothstep(0.0, 0.7, cameraZoneDepth);

    const float MAX_FOG_DISTANCE = 480.0;  // 30 tiles * 16 units per tile

    // Prevent division by zero
    float safeTransp = max(transp, 1.0);

    // Distance factor: 0 at camera, 1 at 30 tiles, capped at 1
    float distanceFactor = min(distFromCamera / MAX_FOG_DISTANCE, 1.0);

    // Transp modifies accumulation rate (600 = reference clear visibility)
    float transpFactor = safeTransp / 600.0;

    // Depth factor: deeper in fog = more fog
    float depthFactor = min((depthA + depthB) / 200.0, 2.0);

    // Combine factors
    float fogValue = (distanceFactor * depthFactor * 255.0) / transpFactor;

    // Cap at FLimit (0-255 range)
    fogValue = min(fogValue, fogLimit);
    float fogDensity = fogValue / 255.0;

    fogDensity *= cameraVerticalFade * zoneTransitionFade;
    fogDensity = clamp(fogDensity, 0.0, 1.0);

    // Edge softening
    vec2 fragUV = vec2(fragWorldPos.x / terrainWorldWidth, fragWorldPos.z / terrainWorldHeight);
    if (fragUV.x >= 0.0 && fragUV.x <= 1.0 && fragUV.y >= 0.0 && fragUV.y <= 1.0) {
        int fragFogIndex = sampleFogZone(fragWorldPos, terrainWorldWidth, terrainWorldHeight);

        if (fragFogIndex >= 0 && fragFogIndex != cameraFogIndex) {
            vec2 uvSampleRadius = vec2(128.0 / terrainWorldWidth, 128.0 / terrainWorldHeight);
            float totalWeight = 0.0;
            float sameZoneWeight = 0.0;

            for (float dx = -3.0; dx <= 3.0; dx += 1.0) {
                for (float dy = -3.0; dy <= 3.0; dy += 1.0) {
                    vec2 offset = vec2(dx, dy) / 3.0;
                    float sampleDist = length(offset);
                    if (sampleDist > 1.0) continue;

                    float weight = 1.0 - sampleDist * 0.5;
                    vec2 neighborCoord = clamp(fragUV + offset * uvSampleRadius, vec2(0.0), vec2(1.0));
                    float neighborIndex = texture(fogMapTexture, neighborCoord).r * 255.0;
                    int neighborFogIndex = int(neighborIndex + 0.5) - 1;

                    totalWeight += weight;
                    if (neighborFogIndex == cameraFogIndex) sameZoneWeight += weight;
                }
            }

            float edgeFade = smoothstep(0.0, 0.5, sameZoneWeight / max(totalWeight, 0.001));
            fogDensity *= edgeFade;
        }
    }

    return mix(color, fogColor, fogDensity);
}

// Carnivores Face Flags (from Hunt.h)
const int SF_DOUBLE_SIDE  = 0x0001;  // Render both sides (no backface culling)
const int SF_DARK_BACK    = 0x0002;  // Darken backface instead of culling
const int SF_OPACITY      = 0x0004;  // Binary alpha: black (0) = transparent, else opaque
const int SF_TRANSPARENT  = 0x0008;  // Semi-transparent blending (glass, water)

// Decode face flags from packed float
int getFaceFlags() {
    return int(faceAlpha0);
}

bool hasFlag(int flag) {
    return (getFaceFlags() & flag) != 0;
}

// Bicubic interpolation weight function (Catmull-Rom spline)
float catmullRomWeight(float x) {
    float ax = abs(x);
    if (ax < 1.0) {
        return (1.5 * ax - 2.5) * ax * ax + 1.0;
    } else if (ax < 2.0) {
        return ((-0.5 * ax + 2.5) * ax - 4.0) * ax + 2.0;
    }
    return 0.0;
}

// High-quality bicubic texture sampling with adaptive sharpening
// Provides sharper results than bilinear while avoiding aliasing artifacts
vec3 bicubicSharpenTexture(sampler2D tex, vec2 uv, float distanceFromCamera, float viewDist) {
    vec2 texSize = vec2(textureSize(tex, 0));
    vec2 texelSize = 1.0 / texSize;

    // Convert to texel coordinates
    vec2 texCoord = uv * texSize - 0.5;
    vec2 texelCenter = floor(texCoord);
    vec2 f = texCoord - texelCenter;

    // Sample 4x4 grid for bicubic
    vec3 result = vec3(0.0);
    float totalWeight = 0.0;

    for (int j = -1; j <= 2; j++) {
        for (int i = -1; i <= 2; i++) {
            vec2 samplePos = (texelCenter + vec2(float(i), float(j)) + 0.5) * texelSize;
            float weight = catmullRomWeight(float(i) - f.x) * catmullRomWeight(float(j) - f.y);
            result += texture(tex, samplePos).rgb * weight;
            totalWeight += weight;
        }
    }
    result /= totalWeight;

    // Distance-adaptive sharpening: sharpen more at distance where mipmaps blur
    // Near: minimal sharpening (0.2), Far: stronger sharpening (0.6)
    float distRatio = clamp(distanceFromCamera / (viewDist * 0.7), 0.0, 1.0);
    float adaptiveSharpness = mix(0.2, 0.6, distRatio);

    // Apply additional unsharp mask for extra crispness
    vec3 center = texture(tex, uv).rgb;
    vec3 blur = vec3(0.0);
    blur += texture(tex, uv + vec2(texelSize.x, 0.0)).rgb;
    blur += texture(tex, uv - vec2(texelSize.x, 0.0)).rgb;
    blur += texture(tex, uv + vec2(0.0, texelSize.y)).rgb;
    blur += texture(tex, uv - vec2(0.0, texelSize.y)).rgb;
    blur *= 0.25;

    // Combine bicubic with adaptive unsharp mask
    vec3 sharpened = result + (result - blur) * adaptiveSharpness;

    return clamp(sharpened, 0.0, 1.0);
}

// Simple sharpening for when distance isn't available
vec3 sharpenTexture(sampler2D tex, vec2 uv) {
    vec2 texelSize = 1.0 / textureSize(tex, 0);

    // Sample center
    vec3 center = texture(tex, uv).rgb;

    // Sample neighbors for blur estimation
    vec3 up    = texture(tex, uv + vec2(0.0, texelSize.y)).rgb;
    vec3 down  = texture(tex, uv - vec2(0.0, texelSize.y)).rgb;
    vec3 left  = texture(tex, uv - vec2(texelSize.x, 0.0)).rgb;
    vec3 right = texture(tex, uv + vec2(texelSize.x, 0.0)).rgb;

    // Calculate local average (blur)
    vec3 blur = (up + down + left + right) * 0.25;

    // Unsharp mask: enhance edges by subtracting blur from original
    float sharpness = 0.4;  // Moderate sharpening (0.0 = none, 1.0 = strong)
    vec3 sharpened = center + (center - blur) * sharpness;

    return clamp(sharpened, 0.0, 1.0);
}

float ShadowCalculation(vec4 fragPosLightSpace)
{   
    // Perform perspective divide
    vec3 projCoords = fragPosLightSpace.xyz / fragPosLightSpace.w;
    
    // Transform to [0,1] range
    projCoords = projCoords * 0.5 + 0.5;
    
    // Check if fragment is outside light's view
    if (projCoords.z > 1.0 || projCoords.x < 0.0 || projCoords.x > 1.0 || 
        projCoords.y < 0.0 || projCoords.y > 1.0) {
        return 0.0;
    }
    
    // Get closest depth value from light's perspective
    float closestDepth = texture(shadowMap, projCoords.xy).r;
    
    // Get depth of current fragment from light's perspective
    float currentDepth = projCoords.z;
    
    // Calculate bias to prevent shadow acne
    float bias = 0.001;
    
    // Enhanced PCF for smoother shadows (larger sampling area)
    float shadow = 0.0;
    vec2 texelSize = 1.0 / textureSize(shadowMap, 0);
    int sampleRadius = 2; // Larger radius for smoother shadows
    int sampleCount = 0;
    
    for(int x = -sampleRadius; x <= sampleRadius; ++x)
    {
        for(int y = -sampleRadius; y <= sampleRadius; ++y)
        {
            float pcfDepth = texture(shadowMap, projCoords.xy + vec2(x, y) * texelSize).r;
            shadow += currentDepth - bias > pcfDepth ? 1.0 : 0.0;
            sampleCount++;
        }
    }
    shadow /= float(sampleCount);
    
    return shadow;
}

void main()
{
    // Calculate distance for adaptive sharpening (use fragDistance to avoid GLSL builtin conflict)
    float fragDistance = length(FragPos - cameraPos);

    // Apply bicubic sampling with distance-adaptive sharpening for maximum texture quality
    vec3 rawColor = bicubicSharpenTexture(basic_texture, texCoord0, fragDistance, view_distance);
    vec4 sC = texture(basic_texture, texCoord0);  // Keep original for alpha check
    int faceFlags = getFaceFlags();

    // ========== CARNIVORES ALPHA SYSTEM ==========
    // sfOpacity (0x0004): Binary color-key transparency - black pixels are transparent
    // sfTransparent (0x0008): Semi-transparent blending (glass, water effects)
    //
    // Check both alpha channel AND color to handle textures with/without alpha bit set.
    // Discard if: alpha is 0 OR color is near-black (for textures without proper alpha)

    float baseAlpha = 1.0;

    if (hasFlag(SF_OPACITY) && enable_transparency) {
        // Hybrid check: discard if alpha is 0 OR if RGB is near-black
        // This handles both properly alpha-flagged textures and raw color-keyed textures
        bool isTransparentByAlpha = (sC.a < 0.5);
        bool isTransparentByColor = (sC.r <= 0.004 && sC.g <= 0.004 && sC.b <= 0.004);

        if (isTransparentByAlpha || isTransparentByColor) {
            discard;
        }
    }

    if (hasFlag(SF_TRANSPARENT)) {
        // Semi-transparent blending (glass effect)
        // Original uses distance-based opacity levels
        baseAlpha = 0.5;
    }

    // Lighting calculations
    vec3 unitSurfaceNormal = normalize(surfaceNormal);
    vec3 unitToLightVector = normalize(toLightVector);

    // Simple diffuse lighting - normals are already corrected in vertex shader
    float diffuse = max(dot(unitSurfaceNormal, unitToLightVector), 0.0);
    float brightness = ambientStrength + diffuse * diffuseStrength;

    // Apply the lighting to the sharpened texture color (BGR to RGB conversion)
    vec3 textureColor = vec3(rawColor.b, rawColor.g, rawColor.r);
    vec3 finalColor = textureColor * brightness;

    // Add cloud shadows for dynamic atmosphere (matching terrain)
    if (enableCloudShadows) {
        vec4 cloudColor = texture(skyTexture, cloudTexCoord);
        float cloudLuminance = (0.299 * cloudColor.b + 0.587 * cloudColor.g + 0.114 * cloudColor.r);

        // Strong cloud contrast for clearly visible moving shadows
        float cloudBrightness = clamp((cloudLuminance - 0.2) * 3.0, 0.0, 1.0);
        // Cloud effect: bright sky = 10% brighter (1.10), dark clouds = 75% darker (0.25)
        float cloudModulation = 0.25 + (cloudBrightness * 0.85);

        finalColor = finalColor * cloudModulation;
    }

    // Apply shadows if enabled
    float shadow = ShadowCalculation(FragPosLightSpace);
    finalColor = finalColor * (1.0 - shadow * 0.1); // Softer shadow darkening

    // Use radial (spherical) distance from camera for natural circular fog falloff
    // (fragDistance already calculated above for texture sharpening)

    // Fog effect (distance fog) - apply first
    float min_distance = view_distance * 0.50;
    float max_distance = view_distance * 0.95;
    float fogFactor = 0.0;

    if (fragDistance > min_distance) {
        fogFactor = clamp((fragDistance - min_distance) / (max_distance - min_distance), 0.0, 1.0);
        fogFactor = min(fogFactor, 0.45);
    }

    finalColor = mix(finalColor, distanceColor.rgb, fogFactor);

    // Apply area fog (localized fog zones) AFTER distance fog
    // This ensures area fog is visible and not washed out by distance fog
    // Use world-space distance for proper Transp-based falloff calculation
    float worldDistance = length(FragPos - cameraPos);
    finalColor = applyAreaFog(finalColor, worldDistance, FragPos);

    // ========== RAIN ATMOSPHERE EFFECTS ==========
    if (rainIntensity > 0.0) {
        // 1. Darken the scene (overcast sky)
        float rainDarkening = 1.0 - (rainIntensity * 0.35);
        finalColor *= rainDarkening;

        // 2. Apply blue-gray tint
        vec3 rainTint = vec3(0.85, 0.88, 1.0);
        finalColor = mix(finalColor, finalColor * rainTint, rainIntensity * 0.4);

        // 3. Add wet surface specular highlights
        vec3 lightDir = normalize(-lightDirection);
        // Primary sharp specular for glossy wet surface
        float wetSpecular = pow(max(dot(unitSurfaceNormal, lightDir), 0.0), 96.0) * rainIntensity * 0.4;
        // Secondary broader specular for wet sheen
        float wetSheen = pow(max(dot(unitSurfaceNormal, lightDir), 0.0), 16.0) * rainIntensity * 0.18;
        finalColor += vec3(wetSpecular + wetSheen) * rainTint;

        // 4. Reduced visibility during rain - distance-based fog intensifies
        float rainVisibilityReduction = rainIntensity * 0.4;
        float rainFogFactor = fogFactor + (1.0 - fogFactor) * rainVisibilityReduction * (fragDistance / max_distance);
        rainFogFactor = clamp(rainFogFactor, 0.0, 0.75);
        finalColor = mix(finalColor, distanceColor.rgb * rainTint, rainFogFactor - fogFactor);
    }

    // ========== TIME-OF-DAY EFFECTS (DAWN/DUSK/NIGHT) ==========
    // Use explicit timeOfDay uniform (0=DAY, 1=DUSK, 2=NIGHT) instead of deriving from visibility
    // visibilityMultiplier still controls intensity of effects
    if (timeOfDay > 0) {
        float effectIntensity = 1.0 - visibilityMultiplier;  // How strong the effect should be

        bool isDusk = (timeOfDay == 1);
        bool isNight = (timeOfDay == 2);

        // 1. Darkening - dusk is brighter than night
        float darkening = 1.0;
        if (isDusk) {
            darkening = 1.0 - (effectIntensity * 0.50);
        } else if (isNight) {
            darkening = 1.0 - (effectIntensity * 0.90);
        }
        finalColor *= darkening;

        // 2. Desaturation - dusk keeps more color than night
        vec3 grayscale = vec3(dot(finalColor, vec3(0.299, 0.587, 0.114)));
        float desatAmount = 0.0;
        if (isDusk) {
            desatAmount = effectIntensity * 0.30;
        } else if (isNight) {
            desatAmount = effectIntensity * 0.85;
        }
        finalColor = mix(finalColor, grayscale, desatAmount);

        // 3. Color tinting - warm for dusk, cool for night (mutually exclusive)
        vec3 duskTint = vec3(1.3, 0.85, 0.55);   // Warm orange/golden sunset
        vec3 nightTint = vec3(0.4, 0.5, 0.8);    // Cool blue moonlight

        if (isDusk) {
            finalColor = mix(finalColor, finalColor * duskTint, effectIntensity * 0.7);
        } else if (isNight) {
            finalColor = mix(finalColor, finalColor * nightTint, effectIntensity * 0.6);
        }

        // 4. Depth fog - dusk has longer visibility than night
        if (isDusk) {
            float duskFogStart = 64.0;
            float duskFogEnd = 640.0;
            float duskFogAmount = smoothstep(duskFogStart, duskFogEnd, fragDistance) * effectIntensity;
            vec3 duskFogColor = vec3(0.25, 0.12, 0.06);  // Dark warm amber
            finalColor = mix(finalColor, duskFogColor, duskFogAmount * 0.8);
        } else if (isNight) {
            float nightFogStart = 16.0;
            float nightFogEnd = 320.0;
            float nightFogAmount = smoothstep(nightFogStart, nightFogEnd, fragDistance) * effectIntensity;
            vec3 nightFogColor = vec3(0.02, 0.03, 0.06); // Very dark blue-black
            finalColor = mix(finalColor, nightFogColor, nightFogAmount);
        }
    }

    // ========== HARD EDGE FADE ==========
    // Ensure geometry fully fades to sky color before reaching the clip plane
    // Uses radial distance for circular fade (fragDistance already calculated above)
    float fadeStart = view_distance * 0.70;  // Start hard fade at 70% of view distance
    float fadeEnd = view_distance * 0.95;    // Fully faded by 95% of view distance
    float hardFade = clamp((fragDistance - fadeStart) / (fadeEnd - fadeStart), 0.0, 1.0);
    // Smooth the fade curve for more gradual transition
    hardFade = smoothstep(0.0, 1.0, hardFade);

    // At dusk/night, fade to appropriate color instead of daytime distance color
    vec3 fadeTargetColor = distanceColor.rgb;
    if (timeOfDay > 0) {
        vec3 duskFadeColor = vec3(0.25, 0.12, 0.06);  // Warm amber for dusk
        vec3 nightFadeColor = vec3(0.02, 0.03, 0.06); // Dark blue for night
        float effectIntensity = 1.0 - visibilityMultiplier;

        if (timeOfDay == 1) {
            fadeTargetColor = mix(distanceColor.rgb, duskFadeColor, effectIntensity);
        } else if (timeOfDay == 2) {
            fadeTargetColor = mix(distanceColor.rgb, nightFadeColor, effectIntensity);
        }
    }
    finalColor = mix(finalColor, fadeTargetColor, hardFade);

    outputColor = vec4(finalColor, baseAlpha);
}
