#version 330

layout(location = 0) in vec3 position;
layout(location = 1) in vec2 texCoord;
layout(location = 2) in vec3 normal;
layout(location = 3) in float faceAlpha;
layout(location = 4) in mat4 instancedMatrix;

out vec2 texCoord0;
out float faceAlpha0;
out vec3 surfaceNormal;
out vec3 toLightVector;
out vec3 FragPos;
out vec4 FragPosLightSpace;
out vec2 cloudTexCoord;  // For cloud shadow sampling
out vec2 fogMapCoord;  // For area fog sampling

uniform highp mat4 MVP;
uniform mat4 model;
uniform mat4 view;
uniform mat4 projection;
uniform float terrainWidth;
uniform float terrainHeight;
uniform float tileWidth;
uniform vec3 cameraPos;
uniform float time;
uniform mat4 lightSpaceMatrix;
uniform bool enableShadows = false;

// Wind effect uniforms
uniform float windStrength = 0.0;      // 0.0 = disabled, typical range 0.1-0.5
uniform vec2 windDirection = vec2(1.0, 0.0);  // normalized wind direction (XZ plane)
uniform float windSpeed = 2.0;         // oscillation speed
uniform float windInfluence = 1.0;     // per-object influence (0 = no sway, 1 = full sway)

void main()
{
    vec3 displacedPosition = position;

    // Apply wind displacement only if wind is enabled and object allows it
    if (windStrength > 0.0 && windInfluence > 0.0) {
        // Height-based influence - only upper parts of objects sway
        // Using local Y position so bases stay grounded
        float heightFactor = smoothstep(0.0, 3.0, position.y);

        // Create natural-looking sway using multiple sine waves
        // Use world position from instance matrix for variation between instances
        vec3 instancePos = vec3(instancedMatrix[3][0], instancedMatrix[3][1], instancedMatrix[3][2]);
        float phase = instancePos.x * 0.1 + instancePos.z * 0.13;  // unique phase per instance

        float primaryWave = sin(time * windSpeed + phase + position.x * 0.3 + position.z * 0.3);
        float secondaryWave = sin(time * windSpeed * 1.3 + phase * 1.7 + position.x * 0.5) * 0.3;
        float sway = (primaryWave + secondaryWave) * windStrength * heightFactor * windInfluence;

        // Apply displacement in wind direction (XZ plane only)
        displacedPosition.x += sway * windDirection.x;
        displacedPosition.z += sway * windDirection.y;
    }

    vec4 worldPosition = model * instancedMatrix * vec4(displacedPosition, 1.0);

    // Moderate sun position for terrain lighting
    // Coming from "southwest" at roughly 55-60 degree elevation
    vec3 lightPosition = vec3(4000.0, 10000.0, 4000.0);

    // Transform normal to world space - must include instancedMatrix rotation
    // Negate normal because Carnivores uses left-handed coordinates (Z is negated on import)
    // which inverts the cross product winding order during normal calculation
    mat4 modelInstance = model * instancedMatrix;
    surfaceNormal = -mat3(transpose(inverse(modelInstance))) * normal;
    toLightVector = lightPosition - worldPosition.xyz;

    vec4 viewPosition = view * worldPosition;
    vec4 projectedPosition = projection * viewPosition;

    texCoord0 = texCoord;
    faceAlpha0 = faceAlpha;
    FragPos = worldPosition.xyz;

    // Calculate cloud texture coordinates (matching terrain shader)
    // Uses world position to ensure clouds move consistently across terrain and objects
    cloudTexCoord = vec2(1.0 - (worldPosition.z / (tileWidth * 128.0)) - (time * 0.008),
                         1.0 - (worldPosition.x / (tileWidth * 128.0)) - (time * 0.008));

    // Fog map coordinate - uses world position normalized to terrain size
    fogMapCoord = vec2(worldPosition.x / (terrainWidth * tileWidth), worldPosition.z / (terrainHeight * tileWidth));

    if (enableShadows) {
        FragPosLightSpace = lightSpaceMatrix * worldPosition;
    }

    gl_Position = projectedPosition;
}
