#version 330

in vec2 texCoord0;
in float faceAlpha0;  // Now contains packed face flags as float
in vec3 surfaceNormal;
in vec3 toLightVector;
in float distanceFromCamera;  // Horizontal distance from camera
in vec3 viewDirection;  // Direction from fragment to camera
in vec3 worldPos;  // World position
in vec2 fogMapCoord;  // For area fog sampling
in vec4 FragPosLightSpace;  // For shadow mapping

out vec4 outputColor;

uniform sampler2D basic_texture;
uniform bool enable_transparency;
uniform float view_distance;
uniform vec4 distanceColor;

// Lighting balance - lower ambient for better shadows, higher diffuse for contrast
uniform float ambientStrength = 0.45;  // Lower ambient allows shadows to show
uniform float diffuseStrength = 0.55;  // Higher diffuse for strong directional shading
uniform float time;
uniform float spawnTime;  // Time when character was spawned
uniform float maxSpawnDistance;  // Max distance before recycling (from spawner config)
uniform float customAlpha;  // Custom alpha override (1.0 = default, <1.0 = transparent)
uniform float rainIntensity = 0.0;  // 0.0 = no rain, 1.0 = heavy rain
uniform float visibilityMultiplier = 1.0;  // Time-of-day visibility (1.0 = day, 0.35 = night)
uniform int timeOfDay = 0;  // 0=DAY, 1=DUSK, 2=NIGHT (explicit, not derived from visibility)

// Shadow mapping uniforms
uniform sampler2D shadowMap;
uniform bool enableShadows = false;
uniform float shadowStrength = 0.5;  // How dark shadows appear (0=invisible, 1=black)

// Trophy room lighting - enhanced display lighting
uniform bool trophyRoomMode = false;  // Enable enhanced lighting for trophy display
uniform float rimLightStrength = 0.0;  // Rim/back lighting intensity for trophy display

// Area fog uniforms
const int MAX_FOG_ZONES = 16;
uniform int numFogZones = 0;
uniform sampler2D fogMapTexture;
uniform vec3 fogColors[MAX_FOG_ZONES];
uniform float fogLimits[MAX_FOG_ZONES];         // FLimit - max fog cap distance
uniform float fogTransparencies[MAX_FOG_ZONES]; // Transp - controls distance falloff rate
uniform float fogMaxOpacities[MAX_FOG_ZONES];
uniform float fogAltitudes[MAX_FOG_ZONES];  // YBegin - fog ceiling height (world units)

// Camera position and terrain dimensions for fog calculation
uniform vec3 cameraPos;
uniform float terrainWidth;
uniform float terrainHeight;
uniform float tileWidth;
uniform float heightmapScale = 4.0;  // 4.0 for C2, 2.0 for C1

// Camera-centric area fog - when player is inside a fog zone, all visible
// fragments are fogged based on distance and vertical position.
const float FOG_ZONE_TRANSITION_RADIUS = 256.0;  // Distance over which fog fades when entering/leaving zones

int sampleFogZone(vec3 worldPos, float terrainWorldWidth, float terrainWorldHeight) {
    vec2 uv = vec2(worldPos.x / terrainWorldWidth, worldPos.z / terrainWorldHeight);
    if (uv.x < 0.0 || uv.x > 1.0 || uv.y < 0.0 || uv.y > 1.0) return -1;
    float fogIndexRaw = texture(fogMapTexture, uv).r * 255.0;
    return int(fogIndexRaw + 0.5) - 1;
}

float estimateFogZoneDepth(vec3 worldPos, int expectedFogIndex, float terrainWorldWidth, float terrainWorldHeight) {
    vec2 uv = vec2(worldPos.x / terrainWorldWidth, worldPos.z / terrainWorldHeight);
    float sameZoneCount = 0.0;
    float totalSamples = 0.0;
    float sampleRadius = FOG_ZONE_TRANSITION_RADIUS;
    vec2 uvRadius = vec2(sampleRadius / terrainWorldWidth, sampleRadius / terrainWorldHeight);

    for (float dist = 0.25; dist <= 1.0; dist += 0.25) {
        for (float angle = 0.0; angle < 6.28; angle += 0.785) {
            vec2 offset = vec2(cos(angle), sin(angle)) * dist;
            vec2 sampleUV = uv + offset * uvRadius;
            if (sampleUV.x >= 0.0 && sampleUV.x <= 1.0 && sampleUV.y >= 0.0 && sampleUV.y <= 1.0) {
                float fogIndexRaw = texture(fogMapTexture, sampleUV).r * 255.0;
                int sampleFogIndex = int(fogIndexRaw + 0.5) - 1;
                totalSamples += 1.0;
                if (sampleFogIndex == expectedFogIndex) {
                    sameZoneCount += 1.0;
                }
            }
        }
    }
    return totalSamples > 0.0 ? sameZoneCount / totalSamples : 1.0;
}

vec3 applyAreaFog(vec3 color, float distFromCamera, vec3 fragWorldPos) {
    if (numFogZones == 0) return color;

    float terrainWorldWidth = terrainWidth * tileWidth;
    float terrainWorldHeight = terrainHeight * tileWidth;

    int cameraFogIndex = sampleFogZone(cameraPos, terrainWorldWidth, terrainWorldHeight);
    if (cameraFogIndex < 0 || cameraFogIndex >= numFogZones) return color;

    float fogCeiling = fogAltitudes[cameraFogIndex];
    float transp = fogTransparencies[cameraFogIndex];  // Already scaled to our world units
    float fogLimit = fogLimits[cameraFogIndex];        // Raw 0-255 value
    vec3 fogColor = fogColors[cameraFogIndex];

    // ========== SIMPLIFIED FOG MODEL ==========
    // Fixed visibility range: 30 tiles (480 world units)
    // Fog accumulates from 0 at player to FLimit/255 at 30 tiles

    float depthA = max(0.0, fogCeiling - cameraPos.y);      // Camera depth in fog
    float depthB = max(0.0, fogCeiling - fragWorldPos.y);   // Fragment depth in fog

    if (depthA <= 0.0 && depthB <= 0.0) return color;

    // Smooth vertical transition for camera crossing ceiling
    float ceilingTransitionHeight = 96.0;
    float cameraVerticalFade = 1.0;
    if (depthA <= 0.0) {
        float aboveAmount = cameraPos.y - fogCeiling;
        cameraVerticalFade = clamp(1.0 - aboveAmount / ceilingTransitionHeight, 0.0, 1.0);
        if (cameraVerticalFade <= 0.0) return color;
    } else if (depthA < ceilingTransitionHeight) {
        cameraVerticalFade = smoothstep(0.0, ceilingTransitionHeight, depthA);
    }

    // Camera zone edge transition with smooth external-to-internal fog handoff
    float cameraZoneDepth = estimateFogZoneDepth(cameraPos, cameraFogIndex, terrainWorldWidth, terrainWorldHeight);
    float zoneTransitionFade = smoothstep(0.0, 0.7, cameraZoneDepth);
    float edgeVisibilityBoost = (1.0 - zoneTransitionFade);

    const float MAX_FOG_DISTANCE = 480.0;  // 30 tiles * 16 units per tile

    // Prevent division by zero
    float safeTransp = max(transp, 1.0);

    // Distance factor: 0 at camera, 1 at 30 tiles, capped at 1
    float distanceFactor = min(distFromCamera / MAX_FOG_DISTANCE, 1.0);

    // Transp modifies accumulation rate (600 = reference clear visibility)
    float transpFactor = safeTransp / 600.0;

    // Depth factor: deeper in fog = more fog
    float depthFactor = min((depthA + depthB) / 200.0, 2.0);

    // Combine factors
    float fogValue = (distanceFactor * depthFactor * 255.0) / transpFactor;

    // Cap at FLimit (0-255 range)
    fogValue = min(fogValue, fogLimit);
    float fogDensity = fogValue / 255.0;

    // Maintain fog visibility at edge to match external view
    float edgeFogDensity = distanceFactor * (fogLimit / 255.0) * 0.85;
    fogDensity = mix(edgeFogDensity, fogDensity, zoneTransitionFade);
    fogDensity *= cameraVerticalFade;
    fogDensity = clamp(fogDensity, 0.0, 1.0);

    // Edge softening
    vec2 fragUV = vec2(fragWorldPos.x / terrainWorldWidth, fragWorldPos.z / terrainWorldHeight);
    if (fragUV.x >= 0.0 && fragUV.x <= 1.0 && fragUV.y >= 0.0 && fragUV.y <= 1.0) {
        int fragFogIndex = sampleFogZone(fragWorldPos, terrainWorldWidth, terrainWorldHeight);

        if (fragFogIndex >= 0 && fragFogIndex != cameraFogIndex) {
            vec2 uvSampleRadius = vec2(128.0 / terrainWorldWidth, 128.0 / terrainWorldHeight);
            float totalWeight = 0.0;
            float sameZoneWeight = 0.0;

            for (float dx = -3.0; dx <= 3.0; dx += 1.0) {
                for (float dy = -3.0; dy <= 3.0; dy += 1.0) {
                    vec2 offset = vec2(dx, dy) / 3.0;
                    float sampleDist = length(offset);
                    if (sampleDist > 1.0) continue;

                    float weight = 1.0 - sampleDist * 0.5;
                    vec2 neighborCoord = clamp(fragUV + offset * uvSampleRadius, vec2(0.0), vec2(1.0));
                    float neighborIndex = texture(fogMapTexture, neighborCoord).r * 255.0;
                    int neighborFogIndex = int(neighborIndex + 0.5) - 1;

                    totalWeight += weight;
                    if (neighborFogIndex == cameraFogIndex) sameZoneWeight += weight;
                }
            }

            float edgeFade = smoothstep(0.0, 0.5, sameZoneWeight / max(totalWeight, 0.001));
            fogDensity *= edgeFade;
        }
    }

    return mix(color, fogColor, fogDensity);
}

// ========== SHADOW CALCULATION ==========
// Poisson disk for soft shadow sampling
const vec2 poissonDisk[16] = vec2[](
    vec2(-0.94201624, -0.39906216), vec2(0.94558609, -0.76890725),
    vec2(-0.094184101, -0.92938870), vec2(0.34495938, 0.29387760),
    vec2(-0.91588581, 0.45771432), vec2(-0.81544232, -0.87912464),
    vec2(-0.38277543, 0.27676845), vec2(0.97484398, 0.75648379),
    vec2(0.44323325, -0.97511554), vec2(0.53742981, -0.47373420),
    vec2(-0.26496911, -0.41893023), vec2(0.79197514, 0.19090188),
    vec2(-0.24188840, 0.99706507), vec2(-0.81409955, 0.91437590),
    vec2(0.19984126, 0.78641367), vec2(0.14383161, -0.14100790)
);

float ShadowCalculation(vec4 fragPosLightSpace) {
    // Perform perspective divide
    vec3 projCoords = fragPosLightSpace.xyz / fragPosLightSpace.w;

    // Transform to [0,1] range (from NDC [-1,1])
    projCoords = projCoords * 0.5 + 0.5;

    // Check if outside shadow map bounds
    if (projCoords.x < 0.0 || projCoords.x > 1.0 ||
        projCoords.y < 0.0 || projCoords.y > 1.0 ||
        projCoords.z > 1.0) {
        return 0.0;  // No shadow outside the shadow map
    }

    float currentDepth = projCoords.z;

    // Bias to prevent shadow acne - scale with surface angle to light
    float bias = 0.002;

    // PCF (Percentage-Closer Filtering) with Poisson disk sampling for soft shadows
    float shadow = 0.0;
    vec2 texelSize = 1.0 / textureSize(shadowMap, 0);
    float spreadRadius = 2.0;  // Soft shadow spread

    for (int i = 0; i < 16; i++) {
        vec2 offset = poissonDisk[i] * texelSize * spreadRadius;
        float pcfDepth = texture(shadowMap, projCoords.xy + offset).r;
        shadow += currentDepth - bias > pcfDepth ? 1.0 : 0.0;
    }
    shadow /= 16.0;

    return shadow;
}

// Carnivores Face Flags (from Hunt.h)
// Note: sfPhong (0x0030) and sfEnvMap (0x0050) include the sfMortal bit (0x0010)
const int SF_DOUBLE_SIDE  = 0x0001;  // Render both sides (no backface culling)
const int SF_DARK_BACK    = 0x0002;  // Darken backface instead of culling
const int SF_OPACITY      = 0x0004;  // Binary alpha: black (0) = transparent, else opaque
const int SF_TRANSPARENT  = 0x0008;  // Semi-transparent blending (glass, water)
const int SF_MORTAL       = 0x0010;  // Vital hit area (head, heart) - gameplay only
const int SF_PHONG        = 0x0030;  // Phong specular highlight (C2 only)
const int SF_ENVMAP       = 0x0050;  // Environment map reflection (C2 only)

// Specular settings for sfPhong faces
uniform float specularStrength = 0.35;

// Debug mode - set to true to visualize mortal faces
uniform bool debugMortalFaces = false;

// Decode face flags from packed float
int getFaceFlags() {
    return int(faceAlpha0);
}

bool hasFlag(int flag) {
    return (getFaceFlags() & flag) != 0;
}

// Check if face should have specular highlights (sfPhong or sfEnvMap flags)
bool hasSpecularMaterial() {
    int flags = getFaceFlags();
    bool hasPhong = (flags & 0x0020) != 0;   // Phong-specific bit
    bool hasEnvMap = (flags & 0x0040) != 0;  // EnvMap-specific bit
    return hasPhong || hasEnvMap;
}

// Bicubic interpolation weight function (Catmull-Rom spline)
float catmullRomWeight(float x) {
    float ax = abs(x);
    if (ax < 1.0) {
        return (1.5 * ax - 2.5) * ax * ax + 1.0;
    } else if (ax < 2.0) {
        return ((-0.5 * ax + 2.5) * ax - 4.0) * ax + 2.0;
    }
    return 0.0;
}

// High-quality bicubic texture sampling with distance-adaptive sharpening
vec3 bicubicSharpenTexture(sampler2D tex, vec2 uv, float distanceFromCamera, float viewDist) {
    vec2 texSize = vec2(textureSize(tex, 0));
    vec2 texelSize = 1.0 / texSize;

    // Convert to texel coordinates
    vec2 texCoord = uv * texSize - 0.5;
    vec2 texelCenter = floor(texCoord);
    vec2 f = texCoord - texelCenter;

    // Sample 4x4 grid for bicubic
    vec3 result = vec3(0.0);
    float totalWeight = 0.0;

    for (int j = -1; j <= 2; j++) {
        for (int i = -1; i <= 2; i++) {
            vec2 samplePos = (texelCenter + vec2(float(i), float(j)) + 0.5) * texelSize;
            float weight = catmullRomWeight(float(i) - f.x) * catmullRomWeight(float(j) - f.y);
            result += texture(tex, samplePos).rgb * weight;
            totalWeight += weight;
        }
    }
    result /= totalWeight;

    // Distance-adaptive sharpening: sharpen more at distance where mipmaps blur
    // Near: minimal sharpening (0.2), Far: stronger sharpening (0.5)
    float distRatio = clamp(distanceFromCamera / (viewDist * 0.7), 0.0, 1.0);
    float adaptiveSharpness = mix(0.2, 0.5, distRatio);

    // Apply additional unsharp mask
    vec3 center = texture(tex, uv).rgb;
    vec3 blur = vec3(0.0);
    blur += texture(tex, uv + vec2(texelSize.x, 0.0)).rgb;
    blur += texture(tex, uv - vec2(texelSize.x, 0.0)).rgb;
    blur += texture(tex, uv + vec2(0.0, texelSize.y)).rgb;
    blur += texture(tex, uv - vec2(0.0, texelSize.y)).rgb;
    blur *= 0.25;

    vec3 sharpened = result + (result - blur) * adaptiveSharpness;
    return clamp(sharpened, 0.0, 1.0);
}

void main()
{
    // Calculate distance for adaptive sharpening (use distFromCam to avoid GLSL builtin conflict)
    float distFromCam = length(worldPos - cameraPos);

    // Apply bicubic sampling with distance-adaptive sharpening for maximum texture quality
    vec3 rawColor = bicubicSharpenTexture(basic_texture, texCoord0, distFromCam, view_distance);
    vec4 sC = texture(basic_texture, texCoord0);  // Keep original for alpha check
    int faceFlags = getFaceFlags();

    // Convert texture from BGR to RGB
    vec3 textureColor = vec3(rawColor.b, rawColor.g, rawColor.r);
    textureColor = clamp(textureColor, 0.0, 1.0);

    vec3 unitSurfaceNormal = normalize(surfaceNormal);
    vec3 unitToLightVector = normalize(toLightVector);
    vec3 unitViewDirection = normalize(viewDirection);

    // Simple directional lighting - normals are already corrected in vertex shader
    float diffuse = max(dot(unitSurfaceNormal, unitToLightVector), 0.0);
    float brightness = ambientStrength + diffuse * diffuseStrength;

    // ========== SHADOW MAPPING ==========
    // Apply shadows from the static shadow map (environment shadows on characters)
    float shadowFactor = 0.0;
    if (enableShadows) {
        shadowFactor = ShadowCalculation(FragPosLightSpace);
        // Reduce brightness in shadowed areas (only affects diffuse, not ambient)
        brightness = ambientStrength + diffuse * diffuseStrength * (1.0 - shadowFactor * shadowStrength);
    }

    // Apply lighting to texture color
    vec3 finalColor = textureColor * brightness;

    // ========== TROPHY ROOM ENHANCED LIGHTING ==========
    // Subtle rim lighting for edge definition only - preserves natural colors
    if (trophyRoomMode && rimLightStrength > 0.0) {
        // Rim lighting only at very edge (tight falloff)
        float rimDot = 1.0 - max(dot(unitViewDirection, unitSurfaceNormal), 0.0);
        float rim = pow(rimDot, 4.0);  // Very tight rim - edges only
        vec3 rimColor = vec3(1.0, 0.98, 0.95);  // Slightly warm white
        finalColor += rim * rimLightStrength * rimColor;
    }

    // ========== SPECULAR HIGHLIGHTS FOR SCALY/WET SKIN ==========
    // Subtle specular to simulate natural skin with some life, but not overly shiny
    // Uses Fresnel effect: highlights stronger at grazing angles (more realistic)
    {
        vec3 reflectDir = reflect(-unitToLightVector, unitSurfaceNormal);
        float specAngle = max(dot(unitViewDirection, reflectDir), 0.0);

        // Fresnel factor: surfaces facing away from view get more specular (rim lighting effect)
        float fresnel = 1.0 - max(dot(unitViewDirection, unitSurfaceNormal), 0.0);
        fresnel = pow(fresnel, 3.0);  // Tighten the fresnel falloff

        // Tighter specular for matte look - higher exponent = smaller highlight
        float spec = pow(specAngle, 32.0);  // Tighter highlight than before

        // Base strength is low for matte appearance, fresnel adds subtle rim shine
        float strength = specularStrength * 0.25;  // Reduced base strength
        float fresnelBoost = fresnel * 0.15;  // Subtle rim highlight

        // sfPhong/sfEnvMap flagged faces get moderately stronger specular
        if (hasSpecularMaterial()) {
            spec = pow(specAngle, 48.0);  // Even tighter for flagged faces
            strength = specularStrength * 0.6;  // Stronger but still restrained
            fresnelBoost = fresnel * 0.25;  // More pronounced rim for shiny materials
        }

        // Tint specular toward surface color for natural organic look (no white hotspots)
        vec3 specColor = mix(vec3(0.7), textureColor, 0.6);
        finalColor += (strength * spec + fresnelBoost) * specColor;
    }

    // ========== ATMOSPHERIC DEPTH EFFECTS ==========
    // Match terrain shader for consistent appearance
    float distance = length(worldPos - cameraPos);

    // Exponential fog (same as terrain - rapid increase then plateaus)
    float min_distance = view_distance * 0.50;
    float max_distance = view_distance * 0.95;

    // Exponential fog formula matching terrain
    float fogDensity = 0.00012;
    float fogFactor = 1.0 - exp(-distance * fogDensity);
    fogFactor = clamp(fogFactor, 0.0, 0.65);  // Match terrain's 0.65 cap

    // Height-based atmospheric scattering (matching terrain)
    float heightFactor = clamp((200.0 - worldPos.y) / 300.0, 0.0, 0.4);
    fogFactor += heightFactor * 0.15;

    // Distance-based desaturation (matching terrain)
    float desaturationFactor = clamp(distance / (view_distance * 0.8), 0.0, 0.5);

    // Aerial perspective shift toward blue haze (matching terrain)
    // At dusk/night, use appropriate fog color instead of daytime haze
    vec3 aerialColor = vec3(0.6, 0.75, 0.9);
    if (timeOfDay > 0) {
        float effectIntensity = 1.0 - visibilityMultiplier;
        vec3 duskAerialColor = vec3(0.25, 0.12, 0.06);  // Warm amber
        vec3 nightAerialColor = vec3(0.02, 0.03, 0.06); // Dark blue
        if (timeOfDay == 1) {
            aerialColor = mix(aerialColor, duskAerialColor, effectIntensity);
        } else if (timeOfDay == 2) {
            aerialColor = mix(aerialColor, nightAerialColor, effectIntensity);
        }
    }
    float aerialBlend = clamp(distance / (view_distance * 1.1), 0.0, 0.4);

    // Apply desaturation
    vec3 desaturatedColor = mix(finalColor, vec3(dot(finalColor, vec3(0.299, 0.587, 0.114))), desaturationFactor);

    // Apply aerial perspective
    vec3 aerialBlended = mix(desaturatedColor, aerialColor * desaturatedColor, aerialBlend);

    // Blend with distance fog - at dusk/night, use appropriate fog color
    vec3 fogTargetColor = distanceColor.rgb;
    if (timeOfDay > 0) {
        float effectIntensity = 1.0 - visibilityMultiplier;
        vec3 duskFogColor = vec3(0.25, 0.12, 0.06);  // Warm amber
        vec3 nightFogColor = vec3(0.02, 0.03, 0.06); // Dark blue
        if (timeOfDay == 1) {
            fogTargetColor = mix(distanceColor.rgb, duskFogColor, effectIntensity);
        } else if (timeOfDay == 2) {
            fogTargetColor = mix(distanceColor.rgb, nightFogColor, effectIntensity);
        }
    }
    finalColor = mix(aerialBlended, fogTargetColor, fogFactor);

    // Contrast reduction at distance (matching terrain)
    if (distance > min_distance) {
        float contrastReduction = clamp((distance - min_distance) / (max_distance - min_distance), 0.0, 0.3);
        // At dusk/night, use appropriate fog color instead of gray for contrast reduction
        vec3 contrastTarget = vec3(0.5);
        if (timeOfDay > 0) {
            float effectIntensity = 1.0 - visibilityMultiplier;
            vec3 duskContrastTarget = vec3(0.25, 0.12, 0.06);
            vec3 nightContrastTarget = vec3(0.02, 0.03, 0.06);
            if (timeOfDay == 1) {
                contrastTarget = mix(vec3(0.5), duskContrastTarget, effectIntensity);
            } else if (timeOfDay == 2) {
                contrastTarget = mix(vec3(0.5), nightContrastTarget, effectIntensity);
            }
        }
        finalColor = mix(finalColor, contrastTarget, contrastReduction * 0.2);
    }

    // Apply area fog (localized fog zones) AFTER distance fog
    // Skip area fog at night - night depth fog handles visibility instead
    if (timeOfDay != 2) {
        // This ensures area fog is visible and not washed out by distance fog
        // Use world-space distance for proper Transp-based falloff calculation
        float worldDistance = length(worldPos - cameraPos);
        finalColor = applyAreaFog(finalColor, worldDistance, worldPos);
    }

    // ========== RAIN ATMOSPHERE EFFECTS ==========
    if (rainIntensity > 0.0) {
        // 1. Darken the scene (overcast sky)
        float rainDarkening = 1.0 - (rainIntensity * 0.35);
        finalColor *= rainDarkening;

        // 2. Apply blue-gray tint
        vec3 rainTint = vec3(0.85, 0.88, 1.0);
        finalColor = mix(finalColor, finalColor * rainTint, rainIntensity * 0.4);

        // 3. Wet skin specular highlights (animals look wet and shiny in rain)
        // Primary sharp specular for glossy wet skin
        float wetSpecular = pow(max(dot(unitSurfaceNormal, unitToLightVector), 0.0), 96.0) * rainIntensity * 0.45;
        // Secondary broader specular for wet sheen across skin
        float wetSheen = pow(max(dot(unitSurfaceNormal, unitToLightVector), 0.0), 12.0) * rainIntensity * 0.2;
        finalColor += vec3(wetSpecular + wetSheen) * rainTint;

        // 4. Reduced visibility during rain - distance-based fog intensifies
        float rainVisibilityReduction = rainIntensity * 0.4;
        float rainFogFactor = fogFactor + (1.0 - fogFactor) * rainVisibilityReduction * (distance / max_distance);
        rainFogFactor = clamp(rainFogFactor, 0.0, 0.75);
        finalColor = mix(finalColor, distanceColor.rgb * rainTint, rainFogFactor - fogFactor);
    }

    // ========== CARNIVORES ALPHA SYSTEM ==========
    // sfOpacity (0x0004): Binary color-key transparency - black pixels are transparent
    // sfTransparent (0x0008): Semi-transparent blending (glass, water effects)
    //
    // Check both alpha channel AND color to handle textures with/without alpha bit set.

    float baseAlpha = 1.0;

    if (hasFlag(SF_OPACITY) && enable_transparency) {
        // Hybrid check: discard if alpha is 0 OR if RGB is near-black
        bool isTransparentByAlpha = (sC.a < 0.5);
        bool isTransparentByColor = (sC.r <= 0.004 && sC.g <= 0.004 && sC.b <= 0.004);

        if (isTransparentByAlpha || isTransparentByColor) {
            discard;
        }
    }

    if (hasFlag(SF_TRANSPARENT)) {
        // Semi-transparent blending (glass effect)
        // Original uses distance-based opacity: 25%, 50%, or 75%
        // We'll use a simpler approach: 50% base transparency
        // The fog system will handle distance-based fading
        baseAlpha = 0.5;
    }

    // Fade-in effect for newly spawned characters (fade from invisible to full opacity)
    float fadeInDuration = 2.0;
    float timeSinceSpawn = time - spawnTime;
    float fadeInAlpha = clamp(timeSinceSpawn / fadeInDuration, 0.0, 1.0);

    // ========== DISTANCE-BASED FADE-OUT EFFECT ==========
    // Gradually fade out characters approaching spawn distance limit
    // This prevents "pop-out" when characters are recycled by player-relative spawners
    float fadeOutAlpha = 1.0;

    if (maxSpawnDistance > 0.0) {
        // Start fading 3 tiles (48 world units) before max distance
        float fadeOutStart = maxSpawnDistance - 48.0;  // 3 tiles at 16 units/tile
        float fadeOutEnd = maxSpawnDistance;

        if (distanceFromCamera > fadeOutStart) {
            // Linear fade from 1.0 (fully visible) to 0.0 (invisible) over 3 tiles
            fadeOutAlpha = 1.0 - clamp((distanceFromCamera - fadeOutStart) / (fadeOutEnd - fadeOutStart), 0.0, 1.0);
        }
    }

    // ========== TIME-OF-DAY EFFECTS (DAWN/DUSK/NIGHT) ==========
    // Use explicit timeOfDay uniform (0=DAY, 1=DUSK, 2=NIGHT) instead of deriving from visibility
    // visibilityMultiplier still controls intensity of effects
    if (timeOfDay > 0) {
        float effectIntensity = 1.0 - visibilityMultiplier;  // How strong the effect should be

        bool isDusk = (timeOfDay == 1);
        bool isNight = (timeOfDay == 2);

        // 1. Darkening - dusk is brighter than night
        float darkening = 1.0;
        if (isDusk) {
            darkening = 1.0 - (effectIntensity * 0.50);
        } else if (isNight) {
            darkening = 1.0 - (effectIntensity * 0.90);
        }
        finalColor *= darkening;

        // 2. Desaturation - dusk keeps more color than night
        vec3 grayscale = vec3(dot(finalColor, vec3(0.299, 0.587, 0.114)));
        float desatAmount = 0.0;
        if (isDusk) {
            desatAmount = effectIntensity * 0.30;
        } else if (isNight) {
            desatAmount = effectIntensity * 0.85;
        }
        finalColor = mix(finalColor, grayscale, desatAmount);

        // 3. Color tinting - warm for dusk, cool for night (mutually exclusive)
        vec3 duskTint = vec3(1.3, 0.85, 0.55);   // Warm orange/golden sunset
        vec3 nightTint = vec3(0.4, 0.5, 0.8);    // Cool blue moonlight

        if (isDusk) {
            finalColor = mix(finalColor, finalColor * duskTint, effectIntensity * 0.7);
        } else if (isNight) {
            finalColor = mix(finalColor, finalColor * nightTint, effectIntensity * 0.6);
        }

        // 4. Depth fog - dusk has longer visibility than night
        if (isDusk) {
            float duskFogStart = 200.0;
            float duskFogEnd = 800.0;
            float duskFogAmount = smoothstep(duskFogStart, duskFogEnd, distance) * effectIntensity;
            vec3 duskFogColor = vec3(0.25, 0.12, 0.06);  // Dark warm amber
            finalColor = mix(finalColor, duskFogColor, duskFogAmount * 0.8);
        } else if (isNight) {
            float nightFogStart = 150.0;
            float nightFogEnd = 480.0;
            float nightFogAmount = smoothstep(nightFogStart, nightFogEnd, distance) * effectIntensity;
            vec3 nightFogColor = vec3(0.02, 0.03, 0.06); // Very dark blue-black
            finalColor = mix(finalColor, nightFogColor, nightFogAmount);
        }
    }

    // ========== HARD EDGE FADE ==========
    // Ensure geometry fully fades to sky color before reaching the clip plane
    // Uses radial distance for circular fade (distance already calculated above)
    float fadeStart = view_distance * 0.70;  // Start hard fade at 70% of view distance
    float fadeEnd = view_distance * 0.95;    // Fully faded by 95% of view distance
    float hardFade = clamp((distance - fadeStart) / (fadeEnd - fadeStart), 0.0, 1.0);
    // At dusk/night, fade to appropriate color instead of daytime distance color
    vec3 fadeTargetColor = distanceColor.rgb;
    if (timeOfDay > 0) {
        vec3 duskFadeColor = vec3(0.25, 0.12, 0.06);  // Warm amber for dusk
        vec3 nightFadeColor = vec3(0.02, 0.03, 0.06); // Dark blue for night
        float effectIntensity = 1.0 - visibilityMultiplier;

        if (timeOfDay == 1) {
            fadeTargetColor = mix(distanceColor.rgb, duskFadeColor, effectIntensity);
        } else if (timeOfDay == 2) {
            fadeTargetColor = mix(distanceColor.rgb, nightFadeColor, effectIntensity);
        }
    }
    // Smooth the fade curve for more gradual transition
    hardFade = smoothstep(0.0, 1.0, hardFade);
    finalColor = mix(finalColor, fadeTargetColor, hardFade);

    // Combine all alpha effects: base transparency, fade-in, fade-out, and custom
    float finalAlpha = baseAlpha * min(fadeInAlpha, fadeOutAlpha) * customAlpha;

    // ========== DEBUG: MORTAL FACE VISUALIZATION ==========
    // When enabled, highlight mortal faces in bright pink for debugging hit zones
    if (debugMortalFaces) {
        // DEBUG: Show faceAlpha value as color gradient
        // Red channel = faceAlpha0 / 20.0 (so 16 -> 0.8 red)
        // This will show us what value the shader is actually receiving
        float debugAlpha = faceAlpha0 / 20.0;
        outputColor = vec4(debugAlpha, 1.0 - debugAlpha, 0.0, 1.0);
        return;
    }

    outputColor = vec4(finalColor, finalAlpha);
}
