#version 330

layout(location = 0) in vec3 position;
layout(location = 1) in vec2 texCoord;
layout(location = 2) in vec3 normal;
layout(location = 3) in float faceAlpha;      // Unused for glTF, kept for Vertex compatibility
layout(location = 4) in mat4 instancedMatrix; // Instance transform (4 vec4 slots: 4,5,6,7)

out vec2 v_texCoord;
out vec3 v_worldPos;
out vec3 v_normal;
out vec3 v_viewDir;
out vec2 v_fogMapCoord;
out vec4 v_fragPosLightSpace;

uniform mat4 model;
uniform mat4 view;
uniform mat4 projection;
uniform vec3 cameraPos;
uniform float terrainWidth;
uniform float terrainHeight;
uniform float tileWidth;
uniform float time;
uniform mat4 lightSpaceMatrix;
uniform bool enableShadows = false;

// Wind effect uniforms (same as basic_shader)
uniform float windStrength = 0.0;
uniform vec2 windDirection = vec2(1.0, 0.0);
uniform float windSpeed = 2.0;
uniform float windInfluence = 0.0;  // 0 for most glTF objects (rigid), >0 for vegetation

void main()
{
    vec3 displacedPosition = position;

    // Apply wind displacement for vegetation (optional)
    if (windStrength > 0.0 && windInfluence > 0.0) {
        // Height-based influence - only upper parts sway
        float heightFactor = smoothstep(0.0, 3.0, position.y);

        // Use instance position for phase variation
        vec3 instancePos = vec3(instancedMatrix[3][0], instancedMatrix[3][1], instancedMatrix[3][2]);
        float phase = instancePos.x * 0.1 + instancePos.z * 0.13;

        float primaryWave = sin(time * windSpeed + phase + position.x * 0.3 + position.z * 0.3);
        float secondaryWave = sin(time * windSpeed * 1.3 + phase * 1.7 + position.x * 0.5) * 0.3;
        float sway = (primaryWave + secondaryWave) * windStrength * heightFactor * windInfluence;

        displacedPosition.x += sway * windDirection.x;
        displacedPosition.z += sway * windDirection.y;
    }

    // Calculate world position with instance transform
    mat4 modelInstance = model * instancedMatrix;
    vec4 worldPosition = modelInstance * vec4(displacedPosition, 1.0);

    v_worldPos = worldPosition.xyz;
    v_texCoord = texCoord;

    // Transform normal to world space
    // glTF uses right-handed Y-up, may need adjustment based on import
    v_normal = mat3(transpose(inverse(modelInstance))) * normal;

    // View direction for specular
    v_viewDir = normalize(cameraPos - worldPosition.xyz);

    // Fog map coordinate (same as basic_shader)
    v_fogMapCoord = vec2(worldPosition.x / (terrainWidth * tileWidth),
                         worldPosition.z / (terrainHeight * tileWidth));

    // Shadow map coordinate
    if (enableShadows) {
        v_fragPosLightSpace = lightSpaceMatrix * worldPosition;
    }

    gl_Position = projection * view * worldPosition;
}
