#version 330 core

in vec3 FragPos;
in vec2 TexCoord;
in vec4 FragPosLightSpace;
in vec3 toLightVector;
in float faceAlpha0;  // Now contains packed face flags as float

out vec4 FragColor;

uniform sampler2D basic_texture;
uniform sampler2D shadowMap;
uniform vec3 lightDirection = vec3(-0.5, -1.0, -0.5);  // Unified sun direction
// Balanced lighting for objects receiving shadows
uniform float ambientStrength = 0.35;
uniform float diffuseStrength = 0.65;
uniform vec3 cameraPos;
uniform float view_distance = 1000.0;

// Carnivores Face Flags (from Hunt.h)
const int SF_DOUBLE_SIDE  = 0x0001;  // Render both sides (no backface culling)
const int SF_DARK_BACK    = 0x0002;  // Darken backface instead of culling
const int SF_OPACITY      = 0x0004;  // Binary alpha: black (0) = transparent, else opaque
const int SF_TRANSPARENT  = 0x0008;  // Semi-transparent blending (glass, water)

// Decode face flags from packed float
int getFaceFlags() {
    return int(faceAlpha0);
}

bool hasFlag(int flag) {
    return (getFaceFlags() & flag) != 0;
}

// Bicubic interpolation weight function (Catmull-Rom spline)
float catmullRomWeight(float x) {
    float ax = abs(x);
    if (ax < 1.0) {
        return (1.5 * ax - 2.5) * ax * ax + 1.0;
    } else if (ax < 2.0) {
        return ((-0.5 * ax + 2.5) * ax - 4.0) * ax + 2.0;
    }
    return 0.0;
}

// High-quality bicubic texture sampling with adaptive sharpening
vec3 bicubicSharpenTexture(sampler2D tex, vec2 uv, float distanceFromCamera, float viewDist) {
    vec2 texSize = vec2(textureSize(tex, 0));
    vec2 texelSize = 1.0 / texSize;

    // Convert to texel coordinates
    vec2 texCoord = uv * texSize - 0.5;
    vec2 texelCenter = floor(texCoord);
    vec2 f = texCoord - texelCenter;

    // Sample 4x4 grid for bicubic
    vec3 result = vec3(0.0);
    float totalWeight = 0.0;

    for (int j = -1; j <= 2; j++) {
        for (int i = -1; i <= 2; i++) {
            vec2 samplePos = (texelCenter + vec2(float(i), float(j)) + 0.5) * texelSize;
            float weight = catmullRomWeight(float(i) - f.x) * catmullRomWeight(float(j) - f.y);
            result += texture(tex, samplePos).rgb * weight;
            totalWeight += weight;
        }
    }
    result /= totalWeight;

    // Distance-adaptive sharpening
    float distRatio = clamp(distanceFromCamera / (viewDist * 0.7), 0.0, 1.0);
    float adaptiveSharpness = mix(0.2, 0.6, distRatio);

    // Apply additional unsharp mask
    vec3 center = texture(tex, uv).rgb;
    vec3 blur = vec3(0.0);
    blur += texture(tex, uv + vec2(texelSize.x, 0.0)).rgb;
    blur += texture(tex, uv - vec2(texelSize.x, 0.0)).rgb;
    blur += texture(tex, uv + vec2(0.0, texelSize.y)).rgb;
    blur += texture(tex, uv - vec2(0.0, texelSize.y)).rgb;
    blur *= 0.25;

    vec3 sharpened = result + (result - blur) * adaptiveSharpness;
    return clamp(sharpened, 0.0, 1.0);
}

float ShadowCalculation(vec4 fragPosLightSpace, vec3 lightDir)
{
    // Perform perspective divide
    vec3 projCoords = fragPosLightSpace.xyz / fragPosLightSpace.w;

    // Transform to [0,1] range
    projCoords = projCoords * 0.5 + 0.5;

    // Check if fragment is outside light's view
    if (projCoords.z > 1.0 || projCoords.x < 0.0 || projCoords.x > 1.0 ||
        projCoords.y < 0.0 || projCoords.y > 1.0) {
        return 0.0; // No shadow outside light frustum
    }

    // Get closest depth value from light's perspective
    float closestDepth = texture(shadowMap, projCoords.xy).r;

    // Get depth of current fragment from light's perspective
    float currentDepth = projCoords.z;

    // Calculate bias to prevent shadow acne (simplified since we don't have normals)
    float bias = 0.005;

    // PCF shadow sampling for slightly soft but defined edges
    // Using a tighter kernel for sharper shadows on world objects
    float shadow = 0.0;
    vec2 texelSize = 1.0 / textureSize(shadowMap, 0);

    // 3x3 kernel with half-texel offsets for sharper results
    for(int x = -1; x <= 1; ++x)
    {
        for(int y = -1; y <= 1; ++y)
        {
            float pcfDepth = texture(shadowMap, projCoords.xy + vec2(x, y) * texelSize * 0.75).r;
            shadow += currentDepth - bias > pcfDepth ? 1.0 : 0.0;
        }
    }
    shadow /= 9.0;

    return shadow;
}

void main()
{
    // Calculate distance for adaptive sharpening (use fragDistance to avoid GLSL builtin conflict)
    float fragDistance = length(FragPos - cameraPos);

    // Apply bicubic sampling with distance-adaptive sharpening
    vec3 rawColor = bicubicSharpenTexture(basic_texture, TexCoord, fragDistance, view_distance);
    vec4 color = texture(basic_texture, TexCoord);  // Keep original for alpha check

    // ========== CARNIVORES ALPHA SYSTEM ==========
    // Check both alpha channel AND color to handle textures with/without alpha bit set.
    float baseAlpha = 1.0;

    if (hasFlag(SF_OPACITY)) {
        // Hybrid check: discard if alpha is 0 OR if RGB is near-black
        bool isTransparentByAlpha = (color.a < 0.5);
        bool isTransparentByColor = (color.r <= 0.004 && color.g <= 0.004 && color.b <= 0.004);

        if (isTransparentByAlpha || isTransparentByColor) {
            discard;
        }
    }

    if (hasFlag(SF_TRANSPARENT)) {
        // Semi-transparent blending
        baseAlpha = 0.5;
    }

    // Convert from BGR to RGB using sharpened texture
    vec3 textureColor = vec3(rawColor.b, rawColor.g, rawColor.r);

    // Simple lighting calculation (no normals available in this shader)
    vec3 lightDir = normalize(-lightDirection);

    // Ambient lighting
    vec3 ambient = ambientStrength * textureColor;

    // Simple directional diffuse without normals
    vec3 diffuse = diffuseStrength * textureColor;

    // Calculate shadow
    float shadow = ShadowCalculation(FragPosLightSpace, lightDir);

    // Combine lighting with shadow - shadows darken the diffuse component
    // 0.22 strength for soft, subtle shadows
    vec3 lighting = ambient + (1.0 - shadow * 0.22) * diffuse;

    FragColor = vec4(lighting, baseAlpha);
}