#version 330 core

// Shadow depth shader for animated characters (dinosaurs, trophies)
// Supports GPU animation texture sampling for correct shadow silhouettes

layout(location = 0) in vec3 position;  // Base/rest pose (fallback)
// Instance matrix (4 vec4s = mat4)
layout(location = 4) in vec4 instanceMatrix0;
layout(location = 5) in vec4 instanceMatrix1;
layout(location = 6) in vec4 instanceMatrix2;
layout(location = 7) in vec4 instanceMatrix3;
layout(location = 8) in uint originalVertexIndex;  // For animation lookup

uniform mat4 lightSpaceMatrix;

// GPU Animation uniforms (same as dinosaur.vs)
uniform bool useAnimation = false;
uniform sampler2D animationTexture;
uniform float animationFrameA = 0.0;
uniform float animationFrameB = 0.0;
uniform float animationBlend = 0.0;

// Fetch vertex position from animation texture
vec3 fetchAnimatedPosition(int vertexID, float frame) {
    int row = int(frame);
    int col = vertexID;
    vec2 texSize = vec2(textureSize(animationTexture, 0));
    vec2 uv = (vec2(float(col), float(row)) + 0.5) / texSize;
    return texture(animationTexture, uv).rgb;
}

void main()
{
    vec3 animatedPosition;

    if (useAnimation) {
        // GPU Animation: Interpolate between two frames
        vec3 posA = fetchAnimatedPosition(int(originalVertexIndex), animationFrameA);
        vec3 posB = fetchAnimatedPosition(int(originalVertexIndex), animationFrameB);
        animatedPosition = mix(posA, posB, animationBlend);
    } else {
        // Static pose: use base position from vertex buffer
        animatedPosition = position;
    }

    // Reconstruct instance model matrix from 4 vec4 columns
    mat4 instanceModel = mat4(instanceMatrix0, instanceMatrix1, instanceMatrix2, instanceMatrix3);

    vec4 worldPos = instanceModel * vec4(animatedPosition, 1.0);
    gl_Position = lightSpaceMatrix * worldPos;
}
