#version 330 core

in vec2 TexCoords;
in vec2 FragPos;

out vec4 FragColor;

uniform vec2 sunPosition;  // Sun position in NDC (-1 to 1)
uniform vec2 sunSize;      // Sun size in screen space
uniform vec3 sunColor;     // Sun color
uniform float sunIntensity;
uniform float viewFactor;  // How directly we're looking at sun (0-1)

void main()
{
    // Calculate distance from this fragment to sun center
    vec2 toSun = FragPos - sunPosition;
    float dist = length(toSun / sunSize);

    // Create sun disc with soft edge
    // Core is fully opaque, edges fade out
    float sunDisc = 1.0 - smoothstep(0.0, 1.0, dist);

    // Add bright corona effect around the disc
    float corona = 1.0 - smoothstep(0.5, 2.5, dist);
    corona = corona * corona;  // Make it fall off faster

    // Combine disc and corona
    float brightness = sunDisc + corona * 0.3;

    // Sun core should be very bright (almost white)
    vec3 coreColor = mix(sunColor, vec3(1.0), 0.7);  // Whiter in center
    vec3 edgeColor = sunColor;

    // Blend based on distance from center
    vec3 color = mix(edgeColor, coreColor, sunDisc);

    // Apply intensity
    color *= sunIntensity;

    // Alpha based on total brightness
    float alpha = brightness * sunIntensity;

    // Clamp to prevent over-saturation
    alpha = clamp(alpha, 0.0, 1.0);

    // Only render if we have some sun visibility
    if (alpha < 0.001) {
        discard;
    }

    FragColor = vec4(color, alpha);
}
