#version 330

// Enhanced terrain fragment shader for top-down map rendering
// Adds hillshading, height-based coloring, and edge fading
// No fog or weather effects - clean map visualization

in highp vec2 texCoord0;
in highp vec2 texCoord1;
in vec3 surfaceNormal;
in vec3 worldPos;
in float heightValue;
in float edgeFade;  // 0 at edges, 1 in center

out vec4 outputColor;

uniform sampler2D basic_texture;

// Directional light from upper-left (northwest) for classic map hillshading
// This creates shadows on southeast-facing slopes
const vec3 lightDirection = normalize(vec3(-0.5, -0.8, -0.5));

// Lighting parameters tuned for map readability
const float ambientStrength = 0.55;      // Base illumination
const float diffuseStrength = 0.45;      // Hillshade intensity
const float heightBrightness = 0.15;     // How much height affects brightness
const float slopeContrast = 0.3;         // Additional contrast for steep slopes

// Edge fade color (dark blue ocean)
const vec3 fadeColor = vec3(0.1, 0.15, 0.25);

void main()
{
    // Sample both texture layers and blend
    vec4 tex0 = texture(basic_texture, texCoord0);
    vec4 tex1 = texture(basic_texture, texCoord1);
    vec4 baseColor = mix(tex0, tex1, 0.5);

    // Normalize the surface normal
    vec3 norm = normalize(surfaceNormal);

    // === HILLSHADING ===
    // Classic cartographic hillshading - light from northwest
    float hillshade = max(dot(norm, -lightDirection), 0.0);

    // === HEIGHT-BASED BRIGHTNESS ===
    // Higher terrain is slightly brighter (like atmospheric perspective in reverse)
    float heightBoost = heightValue * heightBrightness;

    // === SLOPE ENHANCEMENT ===
    // Steeper slopes (where normal points more horizontal) get more contrast
    float slopeFactor = 1.0 - abs(norm.y);  // 0 = flat, 1 = vertical cliff
    float slopeShading = 1.0 - (slopeFactor * slopeContrast * (1.0 - hillshade));

    // === COMBINE LIGHTING ===
    float lighting = ambientStrength + (diffuseStrength * hillshade);
    lighting += heightBoost;
    lighting *= slopeShading;

    // Clamp to valid range
    lighting = clamp(lighting, 0.3, 1.2);

    // === SUBTLE ELEVATION TINT ===
    // Very slight color shift: low areas slightly blue-green, high areas slightly warm
    vec3 lowTint = vec3(0.95, 0.98, 1.0);   // Slight cool tint for valleys
    vec3 highTint = vec3(1.0, 0.98, 0.95);  // Slight warm tint for peaks
    vec3 elevationTint = mix(lowTint, highTint, heightValue);

    // Terrain color with lighting
    vec3 terrainColor = baseColor.rgb * lighting * elevationTint;

    // === EDGE FADE ===
    // Smoothly fade to ocean color at terrain edges
    float smoothFade = smoothstep(0.0, 1.0, edgeFade);
    vec3 finalColor = mix(fadeColor, terrainColor, smoothFade);

    outputColor = vec4(finalColor, 1.0);
}
