#version 330 core

in vec2 TexCoords;
out vec4 FragColor;

uniform sampler2D screenTexture;
uniform sampler2D depthTexture;
uniform bool isUnderwater;
uniform float waterDepth;
uniform float time;
uniform float waterLevel;    // Y coordinate of water surface
uniform float cameraY;       // Camera Y position

// Area fog uniforms
uniform bool areaFogEnabled = false;
const int MAX_FOG_ZONES = 16;
uniform int numFogZones = 0;
uniform sampler2D fogMapTexture;
uniform vec3 fogColors[MAX_FOG_ZONES];
uniform float fogLimits[MAX_FOG_ZONES];
uniform float fogTransparencies[MAX_FOG_ZONES];
uniform float fogMaxOpacities[MAX_FOG_ZONES];
uniform float fogAltitudes[MAX_FOG_ZONES];       // Fog ceiling heights
uniform float fogGroundLevels[MAX_FOG_ZONES];    // Ground level in each fog zone

// Camera and terrain info for world position reconstruction
uniform vec3 cameraPos;
uniform float terrainWidth;
uniform float terrainHeight;
uniform float tileWidth;
uniform float heightmapScale = 4.0;
uniform float nearPlane = 0.1;
uniform float farPlane = 10000.0;
uniform mat4 inverseViewProjection;

// Heightmap for terrain occlusion checking
uniform sampler2D heightmapTexture;

// Underwater effect parameters
const vec3 underwaterTint = vec3(0.4, 0.7, 0.9);
const float underwaterFogDensity = 0.015;
const float chromaticAberration = 0.002;
const float distortionStrength = 0.008;
const float distortionSpeed = 0.5;
const float underwaterFogStart = 5.0;
const float underwaterFogEnd = 300.0;

// Depth of Field uniforms
uniform bool dofEnabled = false;
uniform float dofFocalDistance = 100.0;  // Focus distance in world units
uniform float dofFocalRange = 50.0;      // Range where things stay sharp
uniform float dofMaxBlur = 4.0;          // Maximum blur radius in pixels

// Panini projection uniforms (for wide FOV distortion correction)
uniform bool paniniEnabled = false;
uniform float paniniStrength = 0.0;      // 0 = rectilinear, 1 = full cylindrical
uniform float fovRadians = 1.396;        // Field of view in radians
uniform float aspectRatio = 1.777;       // Screen aspect ratio (width/height)

// Floating particle parameters
const float particleDensity = 80.0;
const float particleSize = 0.012;
const float particleSpeed = 0.08;
const float particleBrightness = 0.8;

// Area fog constants
const float FOG_ZONE_TRANSITION_RADIUS = 512.0;  // Increased for smoother transitions
const float FOG_HORIZON_FADE_START = 200.0;      // Distance where external fog starts appearing
const float FOG_HORIZON_FADE_END = 800.0;        // Distance where external fog reaches full strength

// Reconstruct world position from depth buffer
vec3 reconstructWorldPosition(vec2 screenUV, float depth) {
    vec4 ndc = vec4(screenUV * 2.0 - 1.0, depth * 2.0 - 1.0, 1.0);
    vec4 worldPos = inverseViewProjection * ndc;
    return worldPos.xyz / worldPos.w;
}

// Linearize depth value
float linearizeDepth(float depth) {
    return (2.0 * nearPlane * farPlane) / (farPlane + nearPlane - depth * (farPlane - nearPlane));
}

// =============================================================================
// DEPTH OF FIELD
// Applies blur based on distance from focal plane
// =============================================================================

// Calculate circle of confusion (blur amount) based on depth
// Only blurs objects BEYOND the focal distance - near objects stay sharp
float calculateCoC(float linearDepth) {
    // Only apply blur to objects beyond the focal point
    // Near objects (closer than focal distance) stay sharp
    float diff = linearDepth - dofFocalDistance;
    if (diff < 0.0) {
        return 0.0;  // No blur for near objects
    }
    float coc = diff / dofFocalRange;
    return clamp(coc, 0.0, 1.0) * dofMaxBlur;
}

// Disk blur with variable radius for depth of field
vec3 dofBlur(sampler2D tex, vec2 uv, float blurRadius) {
    if (blurRadius < 0.5) {
        return texture(tex, uv).rgb;
    }

    vec3 color = vec3(0.0);
    float totalWeight = 0.0;

    // Screen size for proper pixel offsets
    vec2 texelSize = 1.0 / vec2(textureSize(tex, 0));

    // Sample in a disk pattern for natural bokeh
    const int RINGS = 3;
    const int SAMPLES_PER_RING = 8;

    // Center sample
    color += texture(tex, uv).rgb;
    totalWeight += 1.0;

    for (int ring = 1; ring <= RINGS; ring++) {
        float ringRadius = blurRadius * float(ring) / float(RINGS);
        float ringWeight = 1.0 - float(ring - 1) / float(RINGS); // Outer rings slightly less weight

        for (int s = 0; s < SAMPLES_PER_RING; s++) {
            float angle = float(s) * 6.28318530718 / float(SAMPLES_PER_RING);
            // Rotate each ring slightly for better coverage
            angle += float(ring) * 0.5;

            vec2 offset = vec2(cos(angle), sin(angle)) * ringRadius * texelSize;
            color += texture(tex, uv + offset).rgb * ringWeight;
            totalWeight += ringWeight;
        }
    }

    return color / totalWeight;
}

// Apply depth of field effect
vec3 applyDepthOfField(vec3 originalColor, vec2 uv, float depth) {
    if (!dofEnabled) {
        return originalColor;
    }

    float linearDepth = linearizeDepth(depth);
    float coc = calculateCoC(linearDepth);

    if (coc < 0.5) {
        return originalColor;
    }

    return dofBlur(screenTexture, uv, coc);
}

// =============================================================================
// PANINI PROJECTION (Barrel Distortion Correction)
// Reduces the "fishbowl" stretching effect at wide FOVs.
// Standard rectilinear projection stretches objects toward screen edges.
// This applies a subtle barrel distortion to counteract that stretching.
// =============================================================================

vec2 applyPaniniProjection(vec2 screenUV) {
    if (!paniniEnabled || paniniStrength < 0.001) {
        return screenUV;
    }

    // Convert to centered coordinates (-1 to 1)
    vec2 centered = screenUV * 2.0 - 1.0;

    // Correct for aspect ratio (work in square space)
    centered.x *= aspectRatio;

    // Distance from center
    float r = length(centered);

    if (r < 0.001) {
        return screenUV;
    }

    // Apply barrel distortion to counteract rectilinear edge stretching
    // This pulls edge pixels inward, making objects at edges appear less stretched
    //
    // The FOV affects how much correction is needed:
    // Higher FOV = more rectilinear stretching = need more barrel correction
    float fovFactor = fovRadians / 1.57079632679;  // Normalize to 90 degrees

    // Barrel distortion: r' = r * (1 + k * r^2)
    // Negative k pulls edges inward (barrel), positive pushes outward (pincushion)
    // We want barrel (negative) to counteract rectilinear stretching
    float k = -paniniStrength * 0.15 * fovFactor;

    // Apply radial distortion
    float r2 = r * r;
    float distortionFactor = 1.0 + k * r2;

    // Apply the correction
    vec2 corrected = centered * distortionFactor;

    // Undo aspect ratio correction
    corrected.x /= aspectRatio;

    // Convert back to UV space
    vec2 correctedUV = corrected * 0.5 + 0.5;

    // Clamp to valid range
    return clamp(correctedUV, 0.0, 1.0);
}

// Sample terrain height at world XZ position
float sampleTerrainHeight(vec2 worldXZ, float terrainWorldWidth, float terrainWorldHeight) {
    vec2 uv = vec2(worldXZ.x / terrainWorldWidth, worldXZ.y / terrainWorldHeight);
    if (uv.x < 0.0 || uv.x > 1.0 || uv.y < 0.0 || uv.y > 1.0) return 0.0;
    // Heightmap stores actual terrain heights directly as floats (R32F format)
    return texture(heightmapTexture, uv).r;
}

// Sample fog zone from fog map at XZ position
int sampleFogZoneXZ(vec2 worldXZ, float terrainWorldWidth, float terrainWorldHeight) {
    vec2 uv = vec2(worldXZ.x / terrainWorldWidth, worldXZ.y / terrainWorldHeight);
    if (uv.x < 0.0 || uv.x > 1.0 || uv.y < 0.0 || uv.y > 1.0) return -1;
    float fogIndexRaw = texture(fogMapTexture, uv).r * 255.0;
    return int(fogIndexRaw + 0.5) - 1;
}

// Sample fog zone from fog map (3D position, uses XZ)
int sampleFogZone(vec3 worldPos, float terrainWorldWidth, float terrainWorldHeight) {
    return sampleFogZoneXZ(worldPos.xz, terrainWorldWidth, terrainWorldHeight);
}

// Calculate signed distance to fog zone boundary (positive = inside, negative = outside)
// OPTIMIZED: Uses only 4 cardinal directions at 3 radii = 12 samples total
// Smoothing is done via the smoothstep in the caller, not via excessive sampling
float calculateFogZoneSDF(vec3 worldPos, int queryZoneIndex, float terrainWorldWidth, float terrainWorldHeight, out int nearestZone, out float nearestZoneDist) {
    vec2 uv = vec2(worldPos.x / terrainWorldWidth, worldPos.z / terrainWorldHeight);

    int currentZone = sampleFogZoneXZ(worldPos.xz, terrainWorldWidth, terrainWorldHeight);
    bool insideQueryZone = (currentZone == queryZoneIndex && queryZoneIndex >= 0);

    nearestZone = currentZone;
    nearestZoneDist = 0.0;

    float minDistToOther = FOG_ZONE_TRANSITION_RADIUS;
    float minDistToQuery = FOG_ZONE_TRANSITION_RADIUS;

    // Only 3 radii at 4 cardinal directions = 12 samples (was 120!)
    // Radii chosen to cover near/mid/far zones
    float searchRadii[3] = float[3](32.0, 160.0, 400.0);

    for (int r = 0; r < 3; r++) {
        float sampleRadius = searchRadii[r];
        vec2 uvRadius = vec2(sampleRadius / terrainWorldWidth, sampleRadius / terrainWorldHeight);

        // Sample 4 cardinal directions only
        for (int a = 0; a < 4; a++) {
            float angle = float(a) * 1.5708;  // 0, 90, 180, 270 degrees
            vec2 offset = vec2(cos(angle), sin(angle)) * uvRadius;
            vec2 sampleUV = uv + offset;

            if (sampleUV.x >= 0.0 && sampleUV.x <= 1.0 && sampleUV.y >= 0.0 && sampleUV.y <= 1.0) {
                float fogIndexRaw = texture(fogMapTexture, sampleUV).r * 255.0;
                int sampleZone = int(fogIndexRaw + 0.5) - 1;

                if (insideQueryZone) {
                    if (sampleZone != queryZoneIndex) {
                        minDistToOther = min(minDistToOther, sampleRadius);
                    }
                } else {
                    if (sampleZone == queryZoneIndex && queryZoneIndex >= 0) {
                        minDistToQuery = min(minDistToQuery, sampleRadius);
                    }
                    if (sampleZone >= 0 && sampleZone != currentZone) {
                        if (sampleRadius < nearestZoneDist || nearestZoneDist == 0.0) {
                            nearestZone = sampleZone;
                            nearestZoneDist = sampleRadius;
                        }
                    }
                }
            }
        }
    }

    if (insideQueryZone) {
        return minDistToOther;
    } else {
        return -minDistToQuery;
    }
}

// Simplified version: estimate how deep inside a fog zone (0 = at edge, 1 = deep inside)
float estimateFogZoneDepth(vec3 worldPos, int expectedFogIndex, float terrainWorldWidth, float terrainWorldHeight) {
    int nearestZone;
    float nearestDist;
    float sdf = calculateFogZoneSDF(worldPos, expectedFogIndex, terrainWorldWidth, terrainWorldHeight, nearestZone, nearestDist);

    // Convert SDF to 0-1 depth factor
    if (sdf > 0.0) {
        // Inside zone - deeper = higher value
        return clamp(sdf / FOG_ZONE_TRANSITION_RADIUS, 0.0, 1.0);
    } else {
        // Outside zone
        return 0.0;
    }
}

// =============================================================================
// ANALYTICAL HEIGHT FOG
// Based on the approach from https://iquilezles.org/articles/fog/
// For height-bounded fog, we analytically integrate fog density along the ray
// =============================================================================

// Calculate analytical fog for a ray segment through height-bounded fog
// fogFloor: bottom of fog (ground level)
// fogCeiling: top of fog
// Returns fog optical depth (how much fog accumulated)
float calculateHeightFogOpticalDepth(
    vec3 rayOrigin,      // Camera position
    vec3 rayDir,         // Normalized ray direction
    float rayLength,     // Distance to fragment
    float fogFloor,      // Bottom of fog volume
    float fogCeiling,    // Top of fog volume
    float fogDensity     // Base fog density (from transp parameter)
) {
    // Early out if ray doesn't intersect the fog height band
    float startY = rayOrigin.y;
    float endY = rayOrigin.y + rayDir.y * rayLength;

    // Both endpoints above fog ceiling - no fog
    if (startY >= fogCeiling && endY >= fogCeiling) return 0.0;

    // Both endpoints below fog floor - this could happen but fog should still apply
    // In our case, fog extends from floor to ceiling

    // Calculate where ray enters and exits the fog volume (in terms of t along ray)
    float tEnter = 0.0;
    float tExit = rayLength;

    // Clamp to fog ceiling
    if (startY > fogCeiling) {
        // Ray starts above fog, calculate entry point
        if (abs(rayDir.y) > 0.0001) {
            tEnter = (fogCeiling - startY) / rayDir.y;
        }
    }
    if (endY > fogCeiling) {
        // Ray ends above fog, calculate exit point
        if (abs(rayDir.y) > 0.0001) {
            tExit = (fogCeiling - startY) / rayDir.y;
        }
    }

    // Clamp to fog floor (optional - if fog has a bottom)
    if (startY < fogFloor) {
        if (abs(rayDir.y) > 0.0001) {
            tEnter = max(tEnter, (fogFloor - startY) / rayDir.y);
        }
    }
    if (endY < fogFloor) {
        if (abs(rayDir.y) > 0.0001) {
            tExit = min(tExit, (fogFloor - startY) / rayDir.y);
        }
    }

    // Ensure valid range
    tEnter = max(tEnter, 0.0);
    tExit = min(tExit, rayLength);

    if (tExit <= tEnter) return 0.0;

    // Calculate the fog path length
    float fogPathLength = tExit - tEnter;

    // Calculate average height within fog for density variation
    // Fog is denser at the bottom (near floor) and thinner at the top (near ceiling)
    float yEnter = startY + rayDir.y * tEnter;
    float yExit = startY + rayDir.y * tExit;
    float avgY = (yEnter + yExit) * 0.5;

    // Height-based density falloff: denser at bottom, thinner at top
    float fogThickness = fogCeiling - fogFloor;
    float heightInFog = clamp((avgY - fogFloor) / max(fogThickness, 1.0), 0.0, 1.0);
    float heightDensityFactor = 1.0 - heightInFog * 0.7; // 100% at bottom, 30% at top

    // Calculate optical depth
    float opticalDepth = fogPathLength * fogDensity * heightDensityFactor;

    return opticalDepth;
}

// =============================================================================
// SIMPLE AREA FOG CALCULATION
// No SDF edge detection - hard cut at fog zone boundaries
// =============================================================================

vec3 applyUnifiedAreaFog(vec3 color, float depth, vec2 screenUV) {
    if (!areaFogEnabled || numFogZones == 0) return color;

    float terrainWorldWidth = terrainWidth * tileWidth;
    float terrainWorldHeight = terrainHeight * tileWidth;

    // Handle sky pixels
    bool isSkyPixel = (depth > 0.9999);

    // Reconstruct fragment world position (or create a far point for sky)
    vec3 fragWorldPos;
    float rayLength;

    if (isSkyPixel) {
        vec3 ndcFar = vec3(screenUV * 2.0 - 1.0, 0.999);
        vec4 worldFar = inverseViewProjection * vec4(ndcFar, 1.0);
        vec3 farPoint = worldFar.xyz / worldFar.w;
        vec3 rayDir = normalize(farPoint - cameraPos);
        rayLength = 2000.0;
        fragWorldPos = cameraPos + rayDir * rayLength;
    } else {
        fragWorldPos = reconstructWorldPosition(screenUV, depth);
        rayLength = length(fragWorldPos - cameraPos);
    }

    vec3 rayDir = normalize(fragWorldPos - cameraPos);

    // Get camera and fragment fog zones via direct sampling
    int cameraFogIndex = sampleFogZone(cameraPos, terrainWorldWidth, terrainWorldHeight);
    int fragFogIndex = sampleFogZone(fragWorldPos, terrainWorldWidth, terrainWorldHeight);

    // Determine which fog zone to use
    int activeFogZone = -1;
    bool cameraInFog = false;

    if (cameraFogIndex >= 0 && cameraFogIndex < numFogZones) {
        // Camera is inside a fog zone - check if below ceiling
        float ceilingProximity = fogAltitudes[cameraFogIndex] - cameraPos.y;
        if (ceilingProximity > 0.0) {
            // Camera is below fog ceiling - we're in fog
            activeFogZone = cameraFogIndex;
            cameraInFog = true;
        }
    }

    // If camera not in fog, check if fragment is
    if (!cameraInFog && fragFogIndex >= 0 && fragFogIndex < numFogZones) {
        activeFogZone = fragFogIndex;
    }

    // No fog zone found
    if (activeFogZone < 0) return color;

    // Get fog parameters for active zone
    float fogCeiling = fogAltitudes[activeFogZone];
    float fogFloor = fogGroundLevels[activeFogZone];
    float transp = max(fogTransparencies[activeFogZone], 1.0);
    float fogLimit = fogLimits[activeFogZone];
    vec3 fogColor = fogColors[activeFogZone];
    float maxOpacity = fogLimit / 255.0;

    // ==========================================================================
    // SKY FOG - Only when camera is in fog and not looking up
    // ==========================================================================
    if (isSkyPixel) {
        if (cameraInFog && rayDir.y < 0.1) {
            // Camera in fog, looking horizontal/down - apply some fog to sky
            float cameraHeightInFog = clamp((fogCeiling - cameraPos.y) / max(fogCeiling - fogFloor, 1.0), 0.0, 1.0);
            float viewAngleFactor = clamp(-rayDir.y + 0.5, 0.0, 1.0);
            float skyFog = maxOpacity * cameraHeightInFog * viewAngleFactor * 0.4;
            return mix(color, fogColor, clamp(skyFog, 0.0, maxOpacity * 0.5));
        }
        return color;
    }

    // ==========================================================================
    // MAIN FOG CALCULATION
    // ==========================================================================

    float transpFactor = transp / 600.0;
    float baseVisibilityDist = 480.0 * transpFactor;
    baseVisibilityDist = clamp(baseVisibilityDist, 100.0, 2000.0);

    float finalFogDensity = 0.0;

    if (cameraInFog) {
        // ==========================================================================
        // CAMERA INSIDE FOG - Simple distance + depth fog, no edge effects
        // ==========================================================================
        float depthA = max(0.0, fogCeiling - cameraPos.y);
        float depthB = max(0.0, fogCeiling - fragWorldPos.y);
        float depthSum = depthA + depthB;
        float depthFactor = min(depthSum / 200.0, 2.0);

        float distanceFactor = clamp(rayLength / baseVisibilityDist, 0.0, 1.0);

        float fogValue = (distanceFactor * depthFactor * 255.0) / transpFactor;
        fogValue = min(fogValue, fogLimit);

        finalFogDensity = fogValue / 255.0;
    }
    // External fog rendering disabled for now - only render fog when camera is inside fog zone

    finalFogDensity = clamp(finalFogDensity, 0.0, maxOpacity);

    return mix(color, fogColor, finalFogDensity);
}

// Hash function for pseudo-random particle generation
float hash(vec2 p) {
    return fract(sin(dot(p, vec2(127.1, 311.7))) * 43758.5453123);
}

// Generate floating particles
float particles(vec2 uv, float time) {
    vec2 animatedUV = uv;
    animatedUV.x += time * particleSpeed * 0.3;
    animatedUV.y += sin(time * particleSpeed + uv.x * 10.0) * 0.02;

    vec2 gridUV = animatedUV * particleDensity;
    vec2 cellID = floor(gridUV);
    vec2 cellUV = fract(gridUV);

    float particleValue = 0.0;

    for (int x = -1; x <= 1; x++) {
        for (int y = -1; y <= 1; y++) {
            vec2 neighborCell = cellID + vec2(float(x), float(y));
            float randX = hash(neighborCell);
            float randY = hash(neighborCell + vec2(1.0, 1.0));
            vec2 particlePos = vec2(randX, randY);
            vec2 offset = vec2(float(x), float(y)) + particlePos - cellUV;
            float dist = length(offset);

            if (dist < particleSize) {
                float particle = 1.0 - (dist / particleSize);
                particle = smoothstep(0.0, 1.0, particle);
                particleValue = max(particleValue, particle);
            }
        }
    }

    return particleValue;
}

// Apply underwater effects to a color with configurable intensity
vec3 applyUnderwaterEffects(vec3 baseColor, vec2 uv, float depth, float intensity, float depthBelowSurface) {
    if (intensity < 0.001) return baseColor;

    // Wave distortion scaled by intensity
    float waveX = sin(uv.y * 10.0 + time * distortionSpeed) * distortionStrength * intensity;
    float waveY = cos(uv.x * 10.0 + time * distortionSpeed * 0.7) * distortionStrength * intensity;
    vec2 distortedUV = clamp(uv + vec2(waveX, waveY), 0.0, 1.0);

    // Chromatic aberration scaled by intensity
    float chromatic = chromaticAberration * intensity;
    float r = texture(screenTexture, distortedUV + vec2(chromatic, 0.0)).r;
    float g = texture(screenTexture, distortedUV).g;
    float b = texture(screenTexture, distortedUV - vec2(chromatic, 0.0)).b;
    vec3 color = vec3(r, g, b);

    // Underwater tint
    color = mix(baseColor, color * underwaterTint, intensity);

    // Distance-based fog (further objects are foggier)
    float linearDepth = linearizeDepth(depth);
    float fogAmount = smoothstep(underwaterFogStart, underwaterFogEnd, linearDepth);

    // Near-surface density boost: make the area just below the surface foggier
    // depthBelowSurface is how far below water level this pixel is
    // At 0-20 units below surface, add extra fog density for that "murky just under surface" look
    float surfaceFogBoost = 0.0;
    if (depthBelowSurface > 0.0 && depthBelowSurface < 40.0) {
        // Peak murkiness at about 10-20 units below surface, fading in and out
        surfaceFogBoost = smoothstep(0.0, 10.0, depthBelowSurface) * smoothstep(40.0, 15.0, depthBelowSurface);
        surfaceFogBoost *= 0.4;  // Max 40% extra fog near surface
    }

    // Minimum fog amount ensures everything underwater has at least some blue tint
    // This handles edge cases like water plane backfaces or skybox seen from underwater
    float minFogAmount = 0.3;
    fogAmount = max(fogAmount + surfaceFogBoost, minFogAmount);
    fogAmount = min(fogAmount, 1.0);
    fogAmount *= intensity;

    vec3 underwaterFogColor = vec3(0.05, 0.15, 0.25);
    color = mix(color, underwaterFogColor, fogAmount);

    // Brightness reduction
    float brightnessReduction = 1.0 - (fogAmount * 0.5 * intensity);
    color *= brightnessReduction;

    // Desaturation
    float gray = dot(color, vec3(0.299, 0.587, 0.114));
    color = mix(color, mix(vec3(gray), color, 0.7), intensity);

    // Particles (only when intensity is high enough)
    if (intensity > 0.3) {
        float particleValue = particles(uv, time);
        float closeParticles = particles(uv * 0.6, time * 0.5);
        float veryCloseParticles = particles(uv * 0.4, time * 0.3);
        float combinedParticles = particleValue * 0.4 + closeParticles * 0.35 + veryCloseParticles * 0.25;
        float particleVisibility = combinedParticles * particleBrightness * (0.5 + fogAmount * 0.5) * intensity;
        vec3 particleColor = vec3(0.8, 0.9, 1.0);
        color = mix(color, particleColor, particleVisibility);
    }

    return color;
}

void main()
{
    vec2 uv = TexCoords;

    // Apply Panini projection correction to reduce fishbowl effect at wide FOVs
    vec2 correctedUV = applyPaniniProjection(uv);

    // Sample depth at corrected UV for consistency
    float depth = texture(depthTexture, correctedUV).r;

    // === CAMERA-BASED UNDERWATER EFFECTS ===
    // Only apply underwater post-processing when the camera itself is underwater.
    // Per-pixel water detection doesn't work because water levels vary across the map.

    bool cameraBelowWater = (waterLevel > 0.0) && (cameraY < waterLevel);

    vec3 color = texture(screenTexture, correctedUV).rgb;

    if (cameraBelowWater) {
        // Camera is underwater - apply full underwater effects to entire view

        // Calculate approximate depth below surface for fog density
        float depthBelowSurface = max(0.0, waterLevel - cameraY);

        // Apply underwater effects
        color = applyUnderwaterEffects(color, correctedUV, depth, 1.0, depthBelowSurface);

        // Apply vignette
        vec2 vignetteUV = uv * (1.0 - uv);
        float vignette = vignetteUV.x * vignetteUV.y * 15.0;
        vignette = pow(vignette, 0.3);
        color *= mix(0.7, 1.0, vignette);
    } else {
        // Camera is above water (or not in water area) - normal rendering
        color = applyDepthOfField(color, correctedUV, depth);
        color = applyUnifiedAreaFog(color, depth, correctedUV);
    }

    FragColor = vec4(color, 1.0);
}
