#version 330 core

layout(location = 0) in vec3 position;
layout(location = 1) in vec2 texCoord;
layout(location = 2) in vec3 normal;
layout(location = 3) in float alpha;
layout(location = 4) in uint flags;

out vec2 texCoord0;
smooth out float alpha0;
out float EdgeFactor;
out vec3 surfaceNormal;
out vec3 toLightVector;
out vec3 viewDirection;
out vec2 heightmapCoord;
out vec2 fogMapCoord;  // For area fog sampling
out vec4 clipSpace;  // For reflection texture sampling
out vec4 reflectionClipSpace;  // World position projected through reflection camera
out highp vec2 cloudTexCoord;  // For cloud shadow sampling
out vec3 worldPos;  // World position for fog ray marching
flat out uint isDangerWater;

uniform mat4 MVP;
uniform mat4 model;
uniform mat4 reflectionVP;  // Reflection camera's view-projection matrix
uniform float terrainWidth;
uniform float terrainHeight;
uniform float tileWidth;
uniform float time;
uniform vec3 cameraPos;

void main()
{
    // No wave animation - keeps water flat to avoid tile seam artifacts
    vec4 worldPosition = model * vec4(position, 1.0);

    // Unified sun position
    vec3 lightPosition = vec3(5000.0, 10000.0, 5000.0);

    surfaceNormal = mat3(transpose(inverse(model))) * normal;
    toLightVector = lightPosition - worldPosition.xyz;

    gl_Position = MVP * worldPosition;
    clipSpace = gl_Position;  // Store for screen-space calculations

    // Project world position through reflection camera for stable reflection UVs
    reflectionClipSpace = reflectionVP * worldPosition;

    // Calculate distance to the edge of the map for fade
    float edgeDistanceX = min(position.x, (terrainWidth * tileWidth) - position.x);
    float edgeDistanceY = min(position.z, (terrainHeight * tileWidth) - position.z);
    float minEdgeDistance = min(edgeDistanceX, edgeDistanceY);
    float maxEdgeDistance = tileWidth * 60.0;
    EdgeFactor = clamp(minEdgeDistance / maxEdgeDistance, 0.0, 1.0);

    texCoord0 = texCoord;
    alpha0 = alpha;

    viewDirection = normalize(cameraPos - worldPosition.xyz);

    // Heightmap coordinates for depth lookup in fragment shader
    heightmapCoord = vec2(position.x / (terrainWidth * tileWidth), position.z / (terrainHeight * tileWidth));

    // Fog map coordinate (same as heightmap coord for consistency)
    fogMapCoord = heightmapCoord;

    // Cloud texture coordinates - same as terrain for consistent cloud shadows
    cloudTexCoord = vec2(1.0 - (position.z / (tileWidth * 128.0)) - (time * 0.008),
                         1.0 - (position.x / (tileWidth * 128.0)) - (time * 0.008));

    isDangerWater = flags;

    // Pass world position for fog ray marching
    worldPos = worldPosition.xyz;
}
