# Configuration Guide

This document describes all configuration options available in `config.json` for Carnivores CE Renderer.

## Table of Contents

- [Basic Settings](#basic-settings)
- [Logging](#logging)
- [Map Configuration](#map-configuration)
- [Video Settings](#video-settings)
- [Camera Settings](#camera-settings)
- [Sun Settings](#sun-settings)
- [Wind Settings](#wind-settings)
- [Audio Settings](#audio-settings)
- [Weapons](#weapons)
- [Equipment](#equipment)
- [UI Settings](#ui-settings)
- [Cinematic Settings](#cinematic-settings)
- [Multiplayer Settings](#multiplayer-settings)
- [Debug Settings](#debug-settings)
- [Spawns](#spawns)

---

## Basic Settings

| Property | Type | Description |
|----------|------|-------------|
| `basePath` | string | Root path for all game assets. All relative paths in the config are resolved from this location. |

**Example:**
```json
{
  "basePath": "runtime/cce/"
}
```

---

## Logging

The `log` object configures file logging for various subsystems. Each category can be redirected to a file. If not specified, output goes to the default destination.

| Property | Type | Default Destination | Description |
|----------|------|---------------------|-------------|
| `console` | string | stdout | General console output messages |
| `errors` | string | stderr | Error messages |
| `startup` | string | stdout | Startup and initialization messages |
| `network` | string | stdout | Network-related messages |

All paths support `~` for home directory expansion (e.g., `~/logs/network.log`).

**Example:**
```json
{
  "log": {
    "console": "~/Desktop/carn-console.log",
    "errors": "~/Desktop/carn-errors.log",
    "startup": "~/Desktop/carn-startup.log",
    "network": "~/Desktop/carn-network.log"
  }
}
```

**Note:** You only need to specify the log categories you want to redirect to files. Unspecified categories continue to output to their default destinations (stdout/stderr).

---

## Map Configuration

The `map` object defines which map to load and its rendering properties.

| Property | Type | Description |
|----------|------|-------------|
| `type` | string | Game type: `"C1"` (Carnivores 1) or `"C2"` (Carnivores 2/Ice Age) |
| `map` | string | Path to the .MAP file |
| `rsc` | string | Path to the .RSC resource file |
| `sky` | string | Sky rendering mode: `"flat"` or `"sphere"` |
| `weather` | object | Weather configuration (optional) |

### Weather Object

| Property | Type | Default | Description |
|----------|------|---------|-------------|
| `type` | string | `"none"` | Weather type: `"none"`, `"rain"`, `"snow"` |
| `intensity` | number | `0.0` | Weather intensity (0.0 - 1.0). Higher values produce more particles and faster rain. |
| `rainAmbient` | string | | Path to rain ambient sound file (.WAV) |
| `rainAmbientVolume` | number | `0.8` | Volume for rain ambient sound (0.0 - 1.0+) |
| `hearingReductionAtMaxRain` | number | `0.4` | How much rain reduces AI hearing range at max intensity (0.0 = no reduction, 1.0 = completely deaf) |

**Note:** Wind settings (`windStrength`, `windAngle`) are configured in the separate [Wind Settings](#wind-settings) section and automatically affect weather particle movement.

**Example:**
```json
{
  "map": {
    "type": "C2",
    "map": "game/c2/AREA1.MAP",
    "rsc": "game/c2/AREA1.RSC",
    "sky": "sphere",
    "weather": {
      "type": "rain",
      "intensity": 0.7,
      "rainAmbient": "game/audio/RAIN_LOOP.WAV",
      "rainAmbientVolume": 0.8,
      "hearingReductionAtMaxRain": 0.4
    }
  }
}
```

---

## Video Settings

The `video` object controls display and rendering options.

| Property | Type | Default | Description |
|----------|------|---------|-------------|
| `fullscreen` | boolean | `false` | Enable fullscreen mode |
| `width` | number | `1024` | Window width in pixels |
| `height` | number | `768` | Window height in pixels |
| `dynamicShadows` | boolean | `true` | Enable dynamic blob shadows for characters |
| `images` | object | | Image paths for UI elements |

### Images Object

| Property | Type | Description |
|----------|------|-------------|
| `loading` | string | Path to loading screen image (.TGA) |

**Example:**
```json
{
  "video": {
    "fullscreen": false,
    "width": 1920,
    "height": 1080,
    "dynamicShadows": true,
    "images": {
      "loading": "game/images/LOADING.TGA"
    }
  }
}
```

---

## Camera Settings

The `camera` object configures the player's view.

| Property | Type | Default | Description |
|----------|------|---------|-------------|
| `fov` | number | `80.0` | Field of view in degrees |
| `aspectRatio` | string | `"Standard"` | Aspect ratio mode: `"Standard"`, `"Wide"`, `"Ultrawide"` |
| `viewDistance` | number | `16000.0` | Far clipping plane distance |
| `nearPlane` | number | `0.1` | Near clipping plane distance |

**Example:**
```json
{
  "camera": {
    "fov": 90.0,
    "aspectRatio": "Wide",
    "viewDistance": 20000.0,
    "nearPlane": 0.1
  }
}
```

---

## Sun Settings

The `sun` object controls the sun disc and lens flare effects.

| Property | Type | Default | Description |
|----------|------|---------|-------------|
| `color` | array | `[1.0, 0.95, 0.85]` | RGB color of the sun (0.0 - 1.0 each) |
| `intensity` | number | `1.0` | Overall sun brightness |
| `glareIntensity` | number | `0.8` | Lens flare brightness |
| `glareSize` | number | `0.3` | Size of the lens flare effect |
| `glareEnabled` | boolean | `true` | Enable lens flare effect |
| `discEnabled` | boolean | `true` | Enable visible sun disc |

**Example:**
```json
{
  "sun": {
    "color": [1.0, 0.9, 0.7],
    "intensity": 1.2,
    "glareIntensity": 0.6,
    "glareSize": 0.25,
    "glareEnabled": true,
    "discEnabled": true
  }
}
```

---

## Wind Settings

The `wind` object controls vegetation animation and weather particle movement.

| Property | Type | Default | Description |
|----------|------|---------|-------------|
| `enabled` | boolean | `true` | Enable wind effects |
| `strength` | number | `0.6` | Wind intensity (affects vegetation sway amplitude) |
| `speed` | number | `4.0` | Wind animation speed |
| `directionAngle` | number | `45.0` | Wind direction in degrees |

**Example:**
```json
{
  "wind": {
    "enabled": true,
    "strength": 0.8,
    "speed": 3.0,
    "directionAngle": 90.0
  }
}
```

---

## Audio Settings

### Footstep Sounds

The `footstepSounds` object configures player footstep audio.

| Property | Type | Description |
|----------|------|-------------|
| `tracks` | array | List of footstep sound file paths |
| `interval` | number | Time between footsteps in seconds |
| `volume` | number | Playback volume (0.0 - 1.0) |
| `maxDistance` | number | Maximum audible distance for AI detection |

### Water Audio

The `waterAudio` object configures underwater and water interaction sounds.

| Property | Type | Description |
|----------|------|-------------|
| `underwaterAmbient` | string | Path to underwater ambient sound |
| `waterFootsteps` | array | List of water footstep sound paths |
| `nearWaterDistance` | number | Distance threshold for water footstep sounds |
| `underwaterAmbientVolume` | number | Volume for underwater ambient |
| `waterFootstepVolume` | number | Volume for water footsteps |

### Impact Sounds

The `impactSounds` object configures projectile impact audio.

| Property | Type | Description |
|----------|------|-------------|
| `terrain` | array | Sound paths for terrain hits |
| `object` | array | Sound paths for object/model hits |
| `water` | array | Sound paths for water hits |

**Example:**
```json
{
  "footstepSounds": {
    "tracks": [
      "game/audio/STEPS/HWALK1.WAV",
      "game/audio/STEPS/HWALK2.WAV"
    ],
    "interval": 0.45,
    "volume": 1.0,
    "maxDistance": 40.0
  },
  "waterAudio": {
    "underwaterAmbient": "game/audio/A_UNDERW.WAV",
    "waterFootsteps": [
      "game/audio/STEPS/FOOTW1.WAV"
    ],
    "nearWaterDistance": 40.0,
    "underwaterAmbientVolume": 2.25,
    "waterFootstepVolume": 0.35
  }
}
```

---

## Weapons

The `weapons` object defines available weapons. Each weapon has a key (e.g., `"rifle"`, `"primary"`) and configuration.

| Property | Type | Description |
|----------|------|-------------|
| `file` | string | Path to weapon model (.CAR file) |
| `rounds` | number | Magazine capacity |
| `bulletImage` | string | Path to bullet counter image |
| `animations` | object | Animation name mappings |
| `projectiles` | object | Ballistics configuration |

### Animations Object

| Property | Type | Description |
|----------|------|-------------|
| `draw` | string | Animation name for drawing weapon |
| `holster` | string | Animation name for holstering |
| `fire` | string | Animation name for firing |
| `reload` | string | Animation name for reloading |

### Projectiles Object

| Property | Type | Description |
|----------|------|-------------|
| `muzzleFlashFrame` | number | Animation frame to trigger muzzle flash |
| `muzzleOffset` | array | [x, y, z] offset from weapon origin |
| `gunshotRange` | number | Distance at which AI hears the shot |
| `muzzleVelocity` | number | Projectile speed |
| `maxRange` | number | Maximum projectile travel distance |
| `damage` | number | Damage per projectile |
| `projectileType` | string | Type: `"ballistic"` or `"hitscan"` |
| `numProjectiles` | number | Projectiles per shot (for shotguns) |
| `spread` | number | Projectile spread angle |
| `recoil` | number | Camera kick on firing (0.0 - 1.0) |
| `sway` | number | Weapon sway amount (0.0 - 1.0) |
| `accuracy` | number | Base accuracy (0.0 - 1.0) |
| `movingAccuracyPenalty` | number | Accuracy reduction while moving |

### Depth of Field Object (`depthOfField`)

Controls depth of field blur effect when the weapon is drawn. Useful for creating focused aiming effects on scoped weapons.

| Property | Type | Default | Description |
|----------|------|---------|-------------|
| `enabled` | boolean | `false` | Enable weapon-specific DoF when drawn |
| `focalDistance` | number | `200.0` | Focus distance in world units (sharp zone center) |
| `focalRange` | number | `100.0` | Range around focal distance that stays sharp |
| `maxBlur` | number | `5.0` | Maximum blur radius in pixels for out-of-focus areas |

**Note:** When enabled, this overrides the global DoF settings while the weapon is drawn.

### Zoom Object (`zoom`)

Controls field of view changes and scope functionality when the weapon is drawn.

| Property | Type | Default | Description |
|----------|------|---------|-------------|
| `enabled` | boolean | `false` | Enable zoom functionality for this weapon |
| `fovMultiplier` | number | `1.0` | FOV multiplier when weapon is drawn (1.0 = no change, 0.5 = 2x zoom) |
| `scopedFovMultiplier` | number | `0.25` | FOV multiplier when scoped/aiming (0.25 = 4x zoom) |
| `hasScope` | boolean | `false` | Whether weapon has a scope (enables scope toggle with middle mouse) |
| `scopeOverlay` | string | | Path to scope overlay texture (.TGA) for scoped view |

**Example - Basic Rifle (no scope):**
```json
{
  "weapons": {
    "rifle": {
      "file": "game/models/rifle.car",
      "rounds": 24,
      "bulletImage": "game/images/BULLET3.TGA",
      "animations": {
        "draw": "Rf2_get",
        "holster": "Rf2_put",
        "fire": "Rf2_fire",
        "reload": "Rf2_rld"
      },
      "projectiles": {
        "muzzleVelocity": 1600.0,
        "maxRange": 960.0,
        "damage": 20,
        "projectileType": "ballistic",
        "numProjectiles": 1,
        "spread": 0.2,
        "accuracy": 0.95
      }
    }
  }
}
```

**Example - Sniper Rifle with Scope and DoF:**
```json
{
  "weapons": {
    "primary": {
      "file": "game/models/sniper.car",
      "rounds": 5,
      "bulletImage": "game/images/BULLET1.TGA",
      "animations": {
        "draw": "Snp_get",
        "holster": "Snp_put",
        "fire": "Snp_fire",
        "reload": "Snp_rld"
      },
      "projectiles": {
        "muzzleVelocity": 2400.0,
        "maxRange": 2000.0,
        "damage": 150,
        "projectileType": "ballistic",
        "numProjectiles": 1,
        "spread": 0.0,
        "recoil": 0.8,
        "accuracy": 0.99
      },
      "depthOfField": {
        "enabled": true,
        "focalDistance": 400.0,
        "focalRange": 50.0,
        "maxBlur": 8.0
      },
      "zoom": {
        "enabled": true,
        "fovMultiplier": 0.8,
        "scopedFovMultiplier": 0.15,
        "hasScope": true,
        "scopeOverlay": "game/images/SCOPE.TGA"
      }
    }
  }
}
```

**Example - Binoculars (equipment-style zoom):**
```json
{
  "weapons": {
    "primary": {
      "file": "game/models/binoculars.car",
      "rounds": 0,
      "animations": {
        "draw": "Bin_get",
        "holster": "Bin_put"
      },
      "depthOfField": {
        "enabled": true,
        "focalDistance": 600.0,
        "focalRange": 150.0,
        "maxBlur": 4.0
      },
      "zoom": {
        "enabled": true,
        "fovMultiplier": 0.5,
        "scopedFovMultiplier": 0.2,
        "hasScope": true,
        "scopeOverlay": "game/images/BINOCULAR.TGA"
      }
    }
  }
}
```

---

## Equipment

The `equipment` object defines player equipment slots and passive abilities.

### Active Equipment Slots

| Property | Type | Description |
|----------|------|-------------|
| `name` | string | Display name |
| `type` | string | Equipment type (e.g., `"dinosaurRadar"`) |
| `keySlot` | number | Number key binding (1-9) |
| `enabled` | boolean | Whether equipment is available |
| `config` | object | Type-specific configuration |

#### Dinosaur Radar Config

| Property | Type | Description |
|----------|------|-------------|
| `sweepFrequency` | number | Radar sweep interval in seconds |
| `nearMaxRadius` | number | Near detection range |
| `farMaxRadius` | number | Far detection range |
| `marginOfError` | number | Position accuracy variance |

### Passive Equipment

| Property | Type | Description |
|----------|------|-------------|
| `name` | string | Display name |
| `type` | string | Passive type (e.g., `"explorerCover"`) |
| `description` | string | Tooltip description |
| `enabled` | boolean | Whether passive is active |

**Example:**
```json
{
  "equipment": {
    "slots": [
      {
        "name": "Radar",
        "type": "dinosaurRadar",
        "keySlot": 1,
        "enabled": true,
        "config": {
          "sweepFrequency": 8.0,
          "nearMaxRadius": 350.0,
          "farMaxRadius": 900.0
        }
      }
    ],
    "passive": [
      {
        "name": "Explorer Cover",
        "type": "explorerCover",
        "description": "Dinosaurs cannot detect you. Weapons disabled.",
        "enabled": true
      }
    ]
  }
}
```

---

## UI Settings

The `ui` object configures user interface elements.

| Property | Type | Description |
|----------|------|-------------|
| `compass` | string | Path to compass model (.CAR file) |

**Example:**
```json
{
  "ui": {
    "compass": "game/models/compas.car"
  }
}
```

---

## Cinematic Settings

The `cinematic` object configures the intro camera sequence.

| Property | Type | Default | Description |
|----------|------|---------|-------------|
| `enabled` | boolean | `true` | Enable intro cinematic |
| `duration` | number | `20.0` | Total cinematic duration in seconds |
| `letterbox` | boolean | `true` | Enable cinematic letterbox bars |
| `letterboxRatio` | number | `2.39` | Target aspect ratio for letterbox |
| `letterboxFadeOut` | number | `1.5` | Letterbox fade duration after cinematic |
| `keyframes` | array | | Camera keyframe definitions |

### Keyframe Object

| Property | Type | Description |
|----------|------|-------------|
| `time` | number | Time in seconds when this keyframe is reached |
| `position` | array | [x, y, z] camera position |
| `lookAt` | array | [x, y, z] point the camera looks at |
| `easing` | string | Interpolation: `"linear"`, `"ease-in"`, `"ease-out"`, `"ease-in-out"` |

**Tip:** To create a "cut" between shots, place two keyframes with nearly identical times (e.g., 4.99 and 5.0) with matching end/start positions.

**Example:**
```json
{
  "cinematic": {
    "enabled": true,
    "duration": 15.0,
    "letterbox": true,
    "letterboxRatio": 2.39,
    "keyframes": [
      {
        "time": 0.0,
        "position": [200, 35, 220],
        "lookAt": [240, 10, 280],
        "easing": "ease-out"
      },
      {
        "time": 5.0,
        "position": [250, 30, 260],
        "lookAt": [280, 15, 300],
        "easing": "linear"
      }
    ]
  }
}
```

---

## Multiplayer Settings

The `multiplayer` object configures network gameplay, allowing multiple players to hunt together.

### Core Settings

| Property | Type | Default | Description |
|----------|------|---------|-------------|
| `enabled` | boolean | `false` | Enable multiplayer mode |
| `debug` | boolean | `false` | Show network debug window with connection stats |
| `host` | boolean | `false` | `true` to host a game, `false` to join an existing game |
| `port` | number | `7777` | Network port for hosting or connecting |
| `address` | string | `"127.0.0.1"` | Server address (only used when joining, i.e., `host: false`) |
| `playerName` | string | `"Player"` | Display name shown to other players |
| `playerModel` | string | | Path to player avatar model (.CAR file) |

### Avatar Animations (`avatarAnimations`)

Defines the animation names for the player avatar model. Other players will see these animations based on your character's state.

| Property | Type | Description |
|----------|------|-------------|
| `idle` | string | Animation when standing still |
| `walk` | string | Animation when walking |
| `run` | string | Animation when running/sprinting |
| `die` | string | Animation when killed |
| `swim` | string | Animation when swimming |
| `jump` | string | Animation when jumping |
| `attack` | string | Animation when firing a weapon |
| `reload` | string | Animation when reloading |

**Note:** Animation names must match animations present in the `playerModel` .CAR file.

### Avatar Pitch Bend (`avatarPitchBend`)

Controls whether the player avatar bends up/down based on where the player is looking. This creates a more realistic visual where other players can see if someone is aiming up or down.

| Property | Type | Default | Description |
|----------|------|---------|-------------|
| `enabled` | boolean | `false` | Enable vertical look direction bending on avatar |
| `maxAngle` | number | `0.5` | Maximum bend angle in radians (~28 degrees) |

**Example - Basic Multiplayer Setup:**
```json
{
  "multiplayer": {
    "enabled": true,
    "debug": false,
    "host": true,
    "port": 7777,
    "playerName": "Hunter1",
    "playerModel": "game/models/poacher.car"
  }
}
```

**Example - Full Configuration with Avatar Settings:**
```json
{
  "multiplayer": {
    "enabled": true,
    "debug": true,
    "host": true,
    "port": 7777,
    "playerName": "Host",
    "playerModel": "game/models/poacher.car",
    "avatarAnimations": {
      "idle": "Pr_idle1",
      "walk": "Pr_wlk",
      "run": "Pr_run",
      "die": "Pr_die",
      "swim": "Pr_run",
      "jump": "Pr_run",
      "attack": "Pr_idle3",
      "reload": "Pr_idle2"
    },
    "avatarPitchBend": {
      "enabled": true,
      "maxAngle": 0.5
    }
  }
}
```

**Example - Joining a Game:**
```json
{
  "multiplayer": {
    "enabled": true,
    "host": false,
    "address": "192.168.1.100",
    "port": 7777,
    "playerName": "Hunter2",
    "playerModel": "game/models/poacher.car",
    "avatarAnimations": {
      "idle": "Pr_idle1",
      "walk": "Pr_wlk",
      "run": "Pr_run",
      "die": "Pr_die"
    }
  }
}
```

### Networking Notes

- The host and all clients must use the same port number
- All players should load the same map for a synchronized experience
- The `playerModel` can be any valid .CAR file - players can choose different character models
- Network state synchronizes player position, rotation, animation state, and weapon status
- When `debug` is enabled, an ImGui window shows connection status, packet statistics, and player list

---

## Debug Settings

The `debug` object enables various debug visualizations.

| Property | Type | Description |
|----------|------|-------------|
| `shadowDebug` | boolean | Show shadow map debug overlay |
| `renderDebug` | boolean | Show rendering statistics |
| `audioDebug` | boolean | Show audio source debug info |
| `aiSpawnerDebug` | boolean | Show AI spawner status |
| `imguiDebug` | boolean | Show ImGui demo window |
| `audioSourceDebug` | boolean | Show 3D audio source positions |
| `physicsDebug` | boolean | Show Bullet physics debug wireframes |
| `hitZoneDebug` | boolean | Show character hit zones |
| `raycastDebug` | boolean | Show weapon raycast traces |

**Example:**
```json
{
  "debug": {
    "shadowDebug": false,
    "physicsDebug": true,
    "hitZoneDebug": true
  }
}
```

---

## Spawns

The `spawns` array defines all characters (dinosaurs, ambient wildlife, etc.) that appear in the game. Each spawn entry configures:

1. **What** to spawn (model, animations)
2. **How** to spawn it (spawner type or one-off)
3. **How it behaves** (AI configuration)
4. **Whether it's huntable** (trophy settings)

### Understanding Spawn Modes

There are **three ways** to add creatures to a map:

| Mode | Use Case | Spawner Config |
|------|----------|----------------|
| **One-Off Spawn** | Single static creature at a fixed location | No spawner config |
| **Static Spawner** | Population around a map location | `spawner` object |
| **Player-Relative Spawner** | Ambient wildlife that follows the player | `playerRelativeSpawner` object |

You choose the mode by which spawner configuration you include (or omit).

---

### Basic Spawn Properties

Every spawn entry requires these core properties:

| Property | Type | Required | Description |
|----------|------|----------|-------------|
| `file` | string | Yes | Path to character model (.CAR file) |
| `animation` | string | Yes | Default/idle animation name from the .CAR file |
| `position` | array | For spawners | [x, z] tile coordinates - spawn center for static spawner |

---

### Mode 1: One-Off Spawn (No Spawner)

A **one-off spawn** creates a single creature at a fixed location. This is useful for:
- Placing a specific creature at a point of interest
- Creating scripted encounters
- Testing AI behavior

Simply omit both `spawner` and `playerRelativeSpawner` from the entry.

**Example - Single T-Rex at a specific location:**
```json
{
  "file": "game/models/TIREX.CAR",
  "animation": "Tx_walk",
  "position": [200, 150],
  "attachAI": {
    "controller": "ClassicAI",
    "args": {
      "name": "Guardian Rex",
      "behaviorType": "APEX_PREDATOR",
      "animations": {
        "WALK": "Tx_walk",
        "RUN": "Tx_run1",
        "DIE": "Tx_die1",
        "EAT": "Tx_kill"
      },
      "character": {
        "maxHealth": 500.0,
        "scale": 1.2
      }
    }
  }
}
```

**Key Points:**
- The creature spawns once at game start at tile position [200, 150]
- If killed, it does NOT respawn
- Good for boss encounters or unique map features

---

### Mode 2: Static Spawner

A **static spawner** maintains a population of creatures around a fixed map location. This is the classic Carnivores spawning system - creatures spawn in an area regardless of where the player is.

Use this for:
- Main hunting targets (dinosaurs the player is tracking)
- Area-based populations (a herd near a lake)
- Creatures that should persist in specific map regions

#### Static Spawner Properties

| Property | Type | Default | Description |
|----------|------|---------|-------------|
| `frequency` | number | Required | Milliseconds between spawn attempts |
| `radius` | number | Required | Spawn radius in tiles around `position`, or `-1` for map-wide spawning |
| `initial` | number | `0` | Number of creatures to spawn immediately at game start |
| `maxActive` | number | Required | Maximum concurrent creatures of this type |
| `terrain` | string | `"land"` | Terrain mode: `"land"`, `"water"`, or `"any"` |
| `minHeight` | number | (none) | Minimum terrain height for land spawns |
| `maxHeight` | number | (none) | Maximum terrain height for land spawns |
| `minDepth` | number | (none) | Minimum water depth for water spawns |
| `maxDepth` | number | (none) | Maximum water depth for water spawns |

**Map-Wide Spawning:**
When `radius` is set to `-1`, the spawner ignores the `position` and finds valid spawn locations anywhere on the map that meets the terrain criteria. This is useful for distributing creatures across the entire map rather than clustering them around a specific location.

**Terrain Modes:**
- `"land"` (default): Spawns only on dry land with slope <25°, requires all 4 adjacent tiles walkable
- `"water"`: Spawns only in water tiles, optionally within depth constraints
- `"any"`: Skips terrain validation (use for flying creatures)

**Example - Velociraptor Pack:**
```json
{
  "file": "game/models/velo2.car",
  "animation": "Vel_wlk",
  "position": [256, 256],
  "spawner": {
    "frequency": 80000,
    "radius": 128,
    "initial": 10,
    "maxActive": 20
  },
  "huntable": {
    "minScore": 5,
    "maxScore": 7
  },
  "attachAI": {
    "controller": "ClassicAI",
    "args": {
      "name": "Velociraptor",
      "behaviorType": "AGGRESSIVE_PREDATOR",
      "animations": {
        "WALK": "Vel_wlk",
        "RUN": "Vel_run1",
        "DIE": "Vel_die1",
        "EAT": "Vel_eat",
        "JUMP": "Vel_jmp5"
      },
      "character": {
        "walkSpeed": 1.2,
        "runSpeed": 2.8,
        "maxHealth": 75.0
      }
    }
  }
}
```

**How Static Spawning Works:**
1. At game start, `initial` creatures spawn within `radius` tiles of `position`
2. Every `frequency` milliseconds, if count < `maxActive`, a new creature spawns
3. Creatures wander freely but tend to stay near their spawn origin
4. When killed, the population gradually replenishes via the frequency timer

**Spawn Timing Formula:**
- `frequency: 80000` = one spawn attempt every 80 seconds
- `frequency: 600000` = one spawn attempt every 10 minutes (for rare creatures)

---

### Mode 3: Player-Relative Spawner

A **player-relative spawner** maintains ambient wildlife around the player's current position. Creatures spawn in a ring around the player, are culled when too far away, and new ones spawn to maintain the target count.

Use this for:
- Ambient atmosphere (flying pteranodons, grazing herbivores)
- Creatures that should always be "nearby" for immersion
- Non-threatening background wildlife

#### Player-Relative Spawner Properties

| Property | Type | Default | Description |
|----------|------|---------|-------------|
| `targetCount` | number | Required | Desired number of creatures around the player |
| `initialRadius` | number | Required | Minimum spawn distance from player (tiles) |
| `maxRadius` | number | Required | Maximum distance before creature is culled (tiles) |
| `spawnRadius` | array | Required | [min, max] tile range for spawning new creatures |
| `cullPeriod` | number | `2000` | Milliseconds between distance checks |
| `respectDanger` | boolean | `true` | Avoid spawning dangerous creatures too close |
| `terrain` | string | `"land"` | Terrain mode: `"land"`, `"water"`, or `"any"` |
| `minHeight` | number | (none) | Minimum terrain height for land spawns |
| `maxHeight` | number | (none) | Maximum terrain height for land spawns |
| `minDepth` | number | (none) | Minimum water depth for water spawns |
| `maxDepth` | number | (none) | Maximum water depth for water spawns |

**Example - Ambient Pteranodons:**
```json
{
  "file": "game/models/PTERA.CAR",
  "animation": "Pt_fly",
  "playerRelativeSpawner": {
    "targetCount": 4,
    "initialRadius": 30,
    "maxRadius": 80,
    "spawnRadius": [35, 60],
    "cullPeriod": 3000,
    "respectDanger": false
  },
  "attachAI": {
    "controller": "ClassicAI",
    "args": {
      "name": "Pteranodon",
      "behaviorType": "FLYING_AMBIENT",
      "animations": {
        "WALK": "Pt_fly",
        "RUN": "Pt_fly",
        "DIE": "Pt_die"
      },
      "flyUpAnim": "Pt_fly",
      "flyDownAnim": "Pt_fly",
      "fallAnim": "Pt_fall",
      "cruisingAltitude": 200.0,
      "behavior": {
        "damageOnContact": false,
        "radar": false
      },
      "character": {
        "walkSpeed": 2.0,
        "maxHealth": 30.0
      }
    }
  }
}
```

**How Player-Relative Spawning Works:**
1. System maintains `targetCount` creatures around the player
2. Creatures beyond `maxRadius` tiles are removed (culled)
3. New creatures spawn in the `spawnRadius` ring to replace culled ones
4. Spawns never happen closer than `initialRadius` to avoid "pop-in"
5. `respectDanger: true` prevents dangerous AI from spawning directly behind the player

**Spawn Ring Visualization:**
```
        maxRadius (cull boundary)
            ↓
    ┌───────────────────┐
    │   spawnRadius[1]  │  ← New creatures spawn here
    │  ┌─────────────┐  │
    │  │ spawnRadius │  │
    │  │    [0]      │  │
    │  │  ┌───────┐  │  │
    │  │  │initial│  │  │
    │  │  │Radius │  │  │
    │  │  │ [P]   │  │  │  ← Player in center
    │  │  └───────┘  │  │
    │  └─────────────┘  │
    └───────────────────┘
```

---

### Huntable Configuration (`huntable`)

Adding a `huntable` object makes the creature a valid hunting target with trophy awards. Without this, the creature can still be killed but won't trigger the trophy/pickup system.

| Property | Type | Description |
|----------|------|-------------|
| `minScore` | number | Minimum trophy points awarded |
| `maxScore` | number | Maximum trophy points awarded |
| `minWeight` | number | Minimum weight for trophy display |
| `maxWeight` | number | Maximum weight for trophy display |
| `minHeight` | number | Minimum height for trophy display |
| `maxHeight` | number | Maximum height for trophy display |
| `ship` | string | Path to pickup ship model (.CAR file) |
| `shipAudio` | array | Ship voice line sound names |
| `shipArrivalAudio` | array | Ship arrival sound names |
| `shipDepartureAudio` | array | Ship departure sound names |
| `shipJetActiveAudio` | array | Jet engine loop sound names |
| `shipJetActivatedAudio` | array | Jet startup sound names |
| `shipPickupActiveAudio` | array | Trophy beam/pickup sound names |
| `awardBackground` | string | Path to trophy screen background image |

**Example - Full Huntable Config:**
```json
{
  "huntable": {
    "minScore": 11,
    "maxScore": 16,
    "minWeight": 500,
    "maxWeight": 900,
    "minHeight": 350,
    "maxHeight": 500,
    "ship": "game/models/SHIP2A.CAR",
    "shipAudio": ["Shipvoice_3"],
    "shipArrivalAudio": ["Shipvoice_1"],
    "shipDepartureAudio": ["Shipvoice_2"],
    "shipJetActiveAudio": ["Jet3"],
    "shipJetActivatedAudio": ["Jetengineson3-3"],
    "shipPickupActiveAudio": ["Transporter4"],
    "awardBackground": "game/images/TROPHY_G.TGA"
  }
}
```

---

### AI Configuration (`attachAI`)

The `attachAI` object attaches an AI controller that governs creature behavior. Currently, the only controller type is `"ClassicAI"`, which faithfully recreates original Carnivores AI behavior.

```json
{
  "attachAI": {
    "controller": "ClassicAI",
    "args": {
      // AI configuration goes here
    }
  }
}
```

#### ClassicAI Core Settings

| Property | Type | Default | Description |
|----------|------|---------|-------------|
| `name` | string | `"unnamed"` | Display name for debug/logging |
| `DEBUG` | boolean | `false` | Enable AI debug window |
| `behaviorType` | string | `"AGGRESSIVE_PREDATOR"` | AI behavior preset |

#### Behavior Types

The `behaviorType` determines the creature's base AI pattern:

| Type | Description | Example Creatures |
|------|-------------|-------------------|
| `AGGRESSIVE_PREDATOR` | Actively hunts and attacks the player. Uses jump attacks. | Velociraptor, Allosaurus |
| `APEX_PREDATOR` | Top predator with detection sequence (sniff, look, roar). Never flees. | T-Rex, Spinosaurus |
| `DEFENSIVE_CHARGER` | Peaceful until threatened, then charges. Doesn't eat prey. | Triceratops, Ceratosaurus |
| `PURE_HERBIVORE` | Always flees from player and gunshots. Never attacks. | Parasaurolophus, Stegosaurus |
| `FLYING_AMBIENT` | Airborne creature with altitude control. | Pteranodon, Dimorphodon |
| `WATER_AMBIENT` | Stays in water areas. | Brachiosaurus (wading) |

#### Animation Mappings (`animations`)

Maps AI states to animation names in the .CAR file:

| Property | Type | Description |
|----------|------|-------------|
| `WALK` | string | Walking animation |
| `RUN` | string | Running/chasing animation |
| `SLIDE` | string | Sliding/skidding animation (sharp turns) |
| `DIE` | string | Death animation |
| `EAT` | string | Feeding animation (predators) |
| `JUMP` | string | Jump attack animation |
| `SWIM` | string | Swimming animation |
| `SLEEP` | string | Sleeping/resting animation |

**Example:**
```json
{
  "animations": {
    "WALK": "Vel_wlk",
    "RUN": "Vel_run1",
    "SLIDE": "Vel_sld",
    "DIE": "Vel_die1",
    "EAT": "Vel_eat",
    "JUMP": "Vel_jmp5",
    "SWIM": "Vel_swim"
  }
}
```

#### Idle Animations

For creatures with idle behaviors (herbivores, passive predators):

| Property | Type | Description |
|----------|------|-------------|
| `idleAnims` | array | List of idle animation names to randomly play |
| `idleOdds` | number | Probability (0.0-1.0) of playing idle animation each think cycle |

**Example:**
```json
{
  "idleAnims": ["Par_idl1", "Par_idl2", "Par_grz"],
  "idleOdds": 0.1
}
```

#### Detection Animations (APEX_PREDATOR)

Apex predators have a detection sequence before attacking:

| Property | Type | Description |
|----------|------|-------------|
| `sightDetectAnims` | array | Animations when visually spotting player |
| `smellDetectAnims` | array | Animations when smelling player |
| `roarAnim` | string | Roar animation before charging |
| `detectFaceAngle` | number | Angle tolerance for facing player during detection |
| `passiveLookoutOdds` | number | Chance to play lookout animation while passive |

**Example:**
```json
{
  "sightDetectAnims": ["Tx_see1", "Tx_see2"],
  "smellDetectAnims": ["Tx_sml1", "Tx_sml2"],
  "roarAnim": "Tx_scr1",
  "detectFaceAngle": 0.4,
  "passiveLookoutOdds": 0.14
}
```

#### Flying Creature Settings (FLYING_AMBIENT)

| Property | Type | Default | Description |
|----------|------|---------|-------------|
| `cruisingAltitude` | number | `253.0` | Target altitude above terrain |
| `altitudeUpperThreshold` | number | `175.0` | Start descending above this |
| `altitudeLowerThreshold` | number | `112.5` | Start ascending below this |
| `ascentRate` | number | `6.0` | Climb speed divisor (smaller = faster) |
| `descentRate` | number | `16.0` | Dive speed divisor |
| `minAltitude` | number | `16.0` | Minimum altitude floor |
| `flyUpSpeed` | number | `1.5` | Movement speed while climbing |
| `flyDownSpeed` | number | `1.3` | Movement speed while diving |
| `flyUpAnim` | string | | Ascending animation |
| `flyDownAnim` | string | | Descending animation |
| `fallAnim` | string | | Death fall animation |

#### Behavior Object

Fine-tune creature behavior:

| Property | Type | Default | Description |
|----------|------|---------|-------------|
| `vision` | number | `60.0` | Detection range in tiles |
| `radar` | boolean | `true` | Visible on player's radar equipment |
| `damageOnContact` | boolean | `true` | Damages/kills player on touch |
| `aggression` | number | `0.5` | Attack likelihood (0.0 = passive, 1.0 = very aggressive). Affects attack probability when detecting player. |
| `minAttackChance` | number | `0.0` | Minimum attack probability (0.0-1.0). Guarantees creature will attack at least this often regardless of distance. Set to 0.9 for highly aggressive predators. |
| `afraidDuration` | number | `10.0` | Seconds to flee after being shot |
| `avoidsWater` | boolean | `true` | Won't enter water |
| `staysInWater` | boolean | `false` | Must stay in water |
| `zigFreq` | number | `824.0` | Zigzag movement frequency |
| `zigAmp` | number | `0.5` | Zigzag movement amplitude |

#### Character Object

Physical creature properties:

| Property | Type | Default | Description |
|----------|------|---------|-------------|
| `walkSpeed` | number | `0.428` | Walking speed multiplier |
| `runSpeed` | number | `1.2` | Running speed multiplier |
| `jumpSpeed` | number | `2.3` | Jump attack speed |
| `swimSpeed` | number | `0.3` | Swimming speed multiplier |
| `stepHeight` | number | `16.0` | Maximum step height for terrain |
| `scale` | number | `1.0` | Fixed scale multiplier |
| `scaleMin` | number | | Minimum random scale (use instead of `scale`) |
| `scaleMax` | number | | Maximum random scale |
| `heightOffset` | number | `0.0` | Vertical offset from ground |
| `maxHealth` | number | `100.0` | Hit points |
| `mortalHitOnly` | boolean | `false` | If true, only mortal face hits cause damage |
| `hitRadius` | number | `412.0` | Collision cylinder radius |
| `hitHeight` | number | `312.0` | Collision cylinder height |
| `hitCenterY` | number | half of hitHeight | Collision center Y offset |
| `collisionRadius` | number | `0.0` | Pathfinding collision radius in world units. When > 0, samples terrain at multiple points around the creature's footprint to prevent large creatures from clipping through narrow passages. Recommended: ~1/4 of creature's visual width. |
| `wanderRadius` | number | `503.0` | Max wander distance from spawn |
| `killFaceAngle` | number | `1.57` | Must face player within this angle (radians, ~90°) to trigger kill |

Note: Kill detection uses the AI's animation-aware bounding box (AABB). The player must be inside the AI's bounding box and the AI must be facing the player within `killFaceAngle` to trigger a kill.

#### Victim Body Configuration (`victimBody`)

When a predator kills the player, you can customize how the player's death body appears and animates. This is particularly useful for creatures like the T-Rex that have specific "kill" animations.

| Property | Type | Default | Description |
|----------|------|---------|-------------|
| `model` | string | `"DEAD.CAR"` | Path to the death body model (.CAR file) |
| `deathAnimation` | string | `"Hr_dead1"` | Animation to play during kill sequence (looped) |
| `deathDurationMs` | number | `3000` | How long to play death animation (milliseconds) |
| `finalAnimation` | string | `"Hr_dead1"` | Animation after death sequence ends (static pose) |
| `deathOffset` | array | `[0, 0, 0]` | Position offset [x, y, z] applied during death animation phase |
| `finalOffset` | array | `[0, 0, 0]` | Position offset [x, y, z] applied during final animation phase |

**Position Offsets:**
The offset arrays allow fine-tuning of the death body position relative to the AI's orientation. Offsets are applied in the AI's local coordinate space:
- `x` - Right offset relative to AI's facing direction (positive = to the AI's right)
- `y` - Up/down offset (positive = higher)
- `z` - Forward offset relative to AI's facing direction (positive = in front of the AI)

**Example - T-Rex Kill Animation:**
```json
{
  "victimBody": {
    "deathAnimation": "Hr_kill",
    "deathDurationMs": 10000,
    "finalAnimation": "Hr_dead1",
    "deathOffset": [0, 50, 0],
    "finalOffset": [0, 0, 0]
  }
}
```

When the T-Rex kills the player:
1. Death body spawns at the AI's position with matching scale and orientation, adjusted by `deathOffset`
2. Plays "Hr_kill" animation (being eaten) in a loop for 10 seconds
3. Death sound plays once (not looped)
4. After 10 seconds, transitions to "Hr_dead1" (lying dead), repositioned to ground height plus `finalOffset`

---

### Complete Examples

#### Example 1: Aggressive Predator (Velociraptor)

```json
{
  "file": "game/models/velo2.car",
  "animation": "Vel_wlk",
  "position": [256, 256],
  "spawner": {
    "frequency": 80000,
    "radius": 128,
    "initial": 15,
    "maxActive": 25
  },
  "huntable": {
    "minScore": 5,
    "maxScore": 7,
    "minWeight": 300,
    "maxWeight": 500,
    "ship": "game/models/SHIP2A.CAR",
    "awardBackground": "game/images/TROPHY_G.TGA"
  },
  "attachAI": {
    "controller": "ClassicAI",
    "args": {
      "name": "Velociraptor",
      "behaviorType": "AGGRESSIVE_PREDATOR",
      "animations": {
        "WALK": "Vel_wlk",
        "RUN": "Vel_run1",
        "SLIDE": "Vel_sld",
        "DIE": "Vel_die1",
        "EAT": "Vel_eat",
        "JUMP": "Vel_jmp5",
        "SWIM": "Vel_swim",
        "SLEEP": "Vel_slp"
      },
      "behavior": {
        "vision": 40.0,
        "damageOnContact": true,
        "aggression": 0.5,
        "afraidDuration": 8.0
      },
      "character": {
        "walkSpeed": 1.2,
        "runSpeed": 2.8,
        "jumpSpeed": 2.3,
        "scaleMin": 0.9,
        "scaleMax": 1.1,
        "maxHealth": 75.0
      }
    }
  }
}
```

#### Example 2: Apex Predator (T-Rex)

```json
{
  "file": "game/models/TIREX.CAR",
  "animation": "Tx_walk",
  "position": [256, 256],
  "spawner": {
    "frequency": 600000,
    "radius": 128,
    "initial": 1,
    "maxActive": 2
  },
  "huntable": {
    "minScore": 11,
    "maxScore": 16,
    "minWeight": 500,
    "maxWeight": 900,
    "ship": "game/models/SHIP2A.CAR",
    "awardBackground": "game/images/TROPHY_G.TGA"
  },
  "attachAI": {
    "controller": "ClassicAI",
    "args": {
      "name": "Tyrannosaurus Rex",
      "behaviorType": "APEX_PREDATOR",
      "animations": {
        "WALK": "Tx_walk",
        "RUN": "Tx_run1",
        "SWIM": "Tx_swim",
        "DIE": "Tx_die1",
        "EAT": "Tx_kill"
      },
      "sightDetectAnims": ["Tx_see1", "Tx_see2"],
      "smellDetectAnims": ["Tx_sml1", "Tx_sml2"],
      "roarAnim": "Tx_scr1",
      "detectFaceAngle": 0.4,
      "passiveLookoutOdds": 0.14,
      "victimBody": {
        "deathAnimation": "Hr_kill",
        "deathDurationMs": 10000,
        "finalAnimation": "Hr_dead1"
      },
      "behavior": {
        "vision": 60.0,
        "damageOnContact": true,
        "aggression": 1.0,
        "afraidDuration": 0.0,
        "avoidsWater": false
      },
      "character": {
        "walkSpeed": 2.5,
        "runSpeed": 8.3,
        "scaleMin": 1.0,
        "scaleMax": 1.1,
        "maxHealth": 500.0
      }
    }
  }
}
```

#### Example 3: Pure Herbivore (Parasaurolophus)

```json
{
  "file": "game/models/PARA.CAR",
  "animation": "Par_wlk",
  "position": [200, 200],
  "spawner": {
    "frequency": 60000,
    "radius": 100,
    "initial": 8,
    "maxActive": 15
  },
  "huntable": {
    "minScore": 3,
    "maxScore": 5
  },
  "attachAI": {
    "controller": "ClassicAI",
    "args": {
      "name": "Parasaurolophus",
      "behaviorType": "PURE_HERBIVORE",
      "animations": {
        "WALK": "Par_wlk",
        "RUN": "Par_run",
        "DIE": "Par_die"
      },
      "idleAnims": ["Par_idl1", "Par_idl2", "Par_grz"],
      "idleOdds": 0.1,
      "behavior": {
        "vision": 50.0,
        "damageOnContact": false,
        "radar": true,
        "afraidDuration": 12.0
      },
      "character": {
        "walkSpeed": 1.0,
        "runSpeed": 2.2,
        "maxHealth": 60.0
      }
    }
  }
}
```

#### Example 4: Ambient Flying Creature (Pteranodon)

```json
{
  "file": "game/models/PTERA.CAR",
  "animation": "Pt_fly",
  "playerRelativeSpawner": {
    "targetCount": 5,
    "initialRadius": 25,
    "maxRadius": 70,
    "spawnRadius": [30, 55],
    "cullPeriod": 2500,
    "respectDanger": false
  },
  "attachAI": {
    "controller": "ClassicAI",
    "args": {
      "name": "Pteranodon",
      "behaviorType": "FLYING_AMBIENT",
      "animations": {
        "WALK": "Pt_fly",
        "RUN": "Pt_fly",
        "DIE": "Pt_die"
      },
      "flyUpAnim": "Pt_fly",
      "flyDownAnim": "Pt_fly",
      "fallAnim": "Pt_fall",
      "cruisingAltitude": 180.0,
      "altitudeUpperThreshold": 150.0,
      "altitudeLowerThreshold": 100.0,
      "behavior": {
        "damageOnContact": false,
        "radar": false
      },
      "character": {
        "walkSpeed": 2.5,
        "maxHealth": 25.0,
        "scaleMin": 0.8,
        "scaleMax": 1.0
      }
    }
  }
}
```

#### Example 5: Defensive Charger (Triceratops)

```json
{
  "file": "game/models/TRICE.CAR",
  "animation": "Tr_wlk",
  "position": [180, 220],
  "spawner": {
    "frequency": 120000,
    "radius": 80,
    "initial": 3,
    "maxActive": 6
  },
  "huntable": {
    "minScore": 8,
    "maxScore": 12
  },
  "attachAI": {
    "controller": "ClassicAI",
    "args": {
      "name": "Triceratops",
      "behaviorType": "DEFENSIVE_CHARGER",
      "animations": {
        "WALK": "Tr_wlk",
        "RUN": "Tr_run",
        "DIE": "Tr_die"
      },
      "idleAnims": ["Tr_idl1", "Tr_grz"],
      "idleOdds": 0.08,
      "chargeThresholdDist": 256.0,
      "trampleKillDist": 24.0,
      "behavior": {
        "vision": 35.0,
        "damageOnContact": true,
        "aggression": 0.3,
        "afraidDuration": 6.0
      },
      "character": {
        "walkSpeed": 1.0,
        "runSpeed": 3.5,
        "maxHealth": 200.0,
        "hitRadius": 500.0
      }
    }
  }
}
```

---

### Tips for Configuring Spawns

1. **Balance spawn frequencies** - Rare apex predators (T-Rex) should have long frequencies (600000ms = 10 min), while common prey can spawn faster (60000ms = 1 min).

2. **Use player-relative spawners for atmosphere** - Flying creatures and non-threatening ambient wildlife work best with player-relative spawning.

3. **Set appropriate initial counts** - Use `initial` to ensure there's something to hunt when the game starts. A value of 1/4 to 1/2 of `maxActive` is typical.

4. **Kill detection uses bounding boxes** - The player must be inside the AI's animation-aware bounding box and the AI must be facing the player within `killFaceAngle` to trigger a kill. Larger creatures naturally have larger bounding boxes.

5. **Scale affects hitboxes** - When using `scaleMin`/`scaleMax`, remember the creature's bounding box and `hitRadius` scale proportionally.

6. **Test with DEBUG: true** - Enable AI debug logging to see decision-making and tweak behavior parameters.

---

## Controls Reference

| Key | Action |
|-----|--------|
| W/Up | Move forward |
| S/Down | Move backward |
| A/Left | Strafe left |
| D/Right | Strafe right |
| Space | Jump / Swim up |
| Mouse | Look around |
| Left Click | Fire weapon |
| Right Click | Draw/holster weapon |
| R | Reload |
| E | Equipment menu |
| 1-9 | Select equipment slot |
| ESC | Pause menu |
| F1 | Toggle debug UI |
| L | Toggle noclip mode |
| O | Print position info |
| P | Toggle wireframe |
| B | Toggle bounding boxes |
| X | Toggle physics debug |
