#version 330 core

// Brush preview fragment shader
// Renders a semi-transparent brush with tile grid showing affected tiles

out vec4 FragColor;

in vec3 FragPos;
in vec2 LocalPos;
in vec2 WorldPosXZ;

uniform vec4 brushColor;     // RGBA color with alpha
uniform float brushRadius;   // Brush radius in world units
uniform bool isSquare;       // True for square brush, false for circle
uniform float tileSize;      // Tile size in world units (default 16.0)

void main()
{
    float dist;

    if (isSquare) {
        // For square: use Chebyshev distance (max of abs components)
        dist = max(abs(LocalPos.x), abs(LocalPos.y));

        // Discard fragments outside the square
        if (dist > 1.0) {
            discard;
        }
    } else {
        // For circle: use Euclidean distance
        dist = length(LocalPos);

        // Discard fragments outside the circle
        if (dist > 1.0) {
            discard;
        }
    }

    // Calculate tile grid lines based on world position
    // Tiles are centered at (n * tileSize + tileSize/2, m * tileSize + tileSize/2)
    // So grid lines are at n * tileSize
    float gridX = abs(fract(WorldPosXZ.x / tileSize) - 0.5) * 2.0;  // 0 at edge, 1 at center
    float gridZ = abs(fract(WorldPosXZ.y / tileSize) - 0.5) * 2.0;  // 0 at edge, 1 at center

    // Grid line thickness (in normalized tile units)
    float gridLineWidth = 0.08;

    // Check if we're on a grid line (close to tile edge)
    bool onGridLine = gridX < gridLineWidth || gridZ < gridLineWidth;

    // Base fill alpha - very subtle
    float fillAlpha = brushColor.a * 0.3;

    // Grid line alpha - more visible
    float gridAlpha = brushColor.a * 0.8;

    // Choose alpha based on grid line
    float alpha = onGridLine ? gridAlpha : fillAlpha;

    // Add outer edge ring for brush boundary
    float ringWidth = 0.04;
    float ringStart = 1.0 - ringWidth;
    if (dist > ringStart && dist <= 1.0) {
        float ringIntensity = smoothstep(ringStart, ringStart + ringWidth * 0.5, dist);
        ringIntensity *= smoothstep(1.0, 1.0 - ringWidth * 0.5, dist);
        alpha = max(alpha, ringIntensity * brushColor.a * 1.5);
    }

    // Brighten grid lines slightly
    vec3 color = brushColor.rgb;
    if (onGridLine) {
        color = mix(brushColor.rgb, vec3(1.0), 0.3);
    }

    // Output with alpha
    FragColor = vec4(color, alpha);
}
