#version 330

layout(location = 0) in vec3 position; // Base/rest pose position (used when animation disabled)
layout(location = 1) in vec2 texCoord;
layout(location = 2) in vec3 normal;
layout(location = 3) in float faceAlpha;
layout(location = 4) in mat4 instancedMatrix;
layout(location = 8) in uint originalVertexIndex; // Original vertex index for animation lookup

out vec2 texCoord0;
out float faceAlpha0;
out vec3 surfaceNormal;
out vec3 toLightVector;
out float distanceFromCamera;  // Distance for fade-out effect
out vec3 viewDirection;  // Direction from fragment to camera for specular
out vec3 worldPos;  // World position for advanced lighting
out vec2 fogMapCoord;  // For area fog sampling
out vec4 FragPosLightSpace;  // For shadow mapping

uniform mat4 MVP;
uniform mat4 model;
uniform mat4 view;
uniform mat4 projection;
uniform vec3 cameraPos;  // Camera position for distance calculation
uniform float bodyBend; // Body bend angle for spine curvature (radians, side-to-side)
uniform float pitchBend; // Pitch bend angle for up/down look (radians, forward/back tilt)
uniform float characterScale; // Character scale multiplier (default: 1.0)
uniform float terrainWidth;  // Terrain width in tiles
uniform float terrainHeight; // Terrain height in tiles
uniform float tileWidth;     // Width of a single tile in world units
uniform vec3 sunDirection;   // Normalized direction TO the sun (from shadow manager)
uniform mat4 lightSpaceMatrix;  // For shadow mapping

// GPU ANIMATION UNIFORMS
uniform bool useAnimation;             // Enable/disable GPU animation
uniform sampler2D animationTexture;    // RGB16F texture with all animation frames
uniform float animationFrameA;         // Current frame index (e.g., 5.0)
uniform float animationFrameB;         // Next frame index (e.g., 6.0)
uniform float animationBlend;          // Blend factor between frames (0.0 to 1.0)
uniform int animationNumVertices;      // Vertices per frame (for texture lookup)

// CROSS-ANIMATION BLENDING UNIFORMS (for smooth transitions between animations)
uniform bool useCrossFade;             // Enable cross-fade blending mode
uniform sampler2D prevAnimationTexture; // Previous animation frames (texture unit 2)
uniform float prevAnimationFrameA;     // Previous animation current frame
uniform float prevAnimationFrameB;     // Previous animation next frame
uniform float prevAnimationBlend;      // Previous animation blend factor
uniform float crossFadeBlend;          // Cross-fade factor (0.0 = new only, 1.0 = prev only)

/**
 * Fetch vertex position from animation texture.
 *
 * Texture layout: Each row = one animation frame
 *                 Each column = one vertex position (RGB = XYZ)
 *
 * @param vertexID Vertex index within the frame (0 to numVertices-1)
 * @param frame Frame index (can be fractional for sub-frame precision)
 * @return vec3 Vertex position in model space
 */
vec3 fetchAnimatedPosition(int vertexID, float frame) {
    // Calculate texture coordinates for this vertex at this frame
    // Row = frame index, Column = vertex index
    int row = int(frame);
    int col = vertexID;

    // Convert to normalized texture coordinates [0, 1]
    // Add 0.5 to sample from texel center (pixel-perfect addressing)
    vec2 texSize = vec2(textureSize(animationTexture, 0));
    vec2 uv = (vec2(float(col), float(row)) + 0.5) / texSize;

    // Fetch RGB = XYZ position
    // Data is already in float format (converted from shorts / 8.0 on upload)
    vec3 animPos = texture(animationTexture, uv).rgb;

    return animPos;
}

/**
 * Fetch vertex position from PREVIOUS animation texture (for cross-fade blending).
 * Same logic as fetchAnimatedPosition but uses prevAnimationTexture.
 */
vec3 fetchPrevAnimatedPosition(int vertexID, float frame) {
    int row = int(frame);
    int col = vertexID;

    vec2 texSize = vec2(textureSize(prevAnimationTexture, 0));
    vec2 uv = (vec2(float(col), float(row)) + 0.5) / texSize;

    return texture(prevAnimationTexture, uv).rgb;
}

void main()
{
    vec3 animatedPosition;

    if (useAnimation) {
        // GPU ANIMATION: Interpolate between two frames in parallel on GPU
        // Use originalVertexIndex (from face data) to look up correct animation data
        vec3 posA = fetchAnimatedPosition(int(originalVertexIndex), animationFrameA);
        vec3 posB = fetchAnimatedPosition(int(originalVertexIndex), animationFrameB);
        vec3 currentPose = mix(posA, posB, animationBlend);

        // CROSS-ANIMATION BLENDING: Smooth transitions between different animations
        if (useCrossFade && crossFadeBlend > 0.0) {
            // Fetch previous animation pose
            vec3 prevPosA = fetchPrevAnimatedPosition(int(originalVertexIndex), prevAnimationFrameA);
            vec3 prevPosB = fetchPrevAnimatedPosition(int(originalVertexIndex), prevAnimationFrameB);
            vec3 prevPose = mix(prevPosA, prevPosB, prevAnimationBlend);

            // Blend between current and previous animation
            // crossFadeBlend: 1.0 = fully previous, 0.0 = fully current
            animatedPosition = mix(currentPose, prevPose, crossFadeBlend);
        } else {
            animatedPosition = currentPose;
        }
    } else {
        // Static pose: use base position from vertex buffer
        animatedPosition = position;
    }

    // BODY BEND: Apply spine curvature based on Z-coordinate (original C1 mechanic)
    // Original formula: fi = z/240 (front) or z/380 (back), clamped to ±1, then scaled by bend
    // This curves the spine by rotating vertices in the XZ plane
    vec3 bentPosition = animatedPosition;

    if (abs(bodyBend) > 0.001 && useAnimation) {  // Only bend animated characters
        float z = animatedPosition.z;
        float bendInfluence;

        // Calculate bend influence based on Z position (matches original C1)
        if (z > 0.0) {
            // Front of body (head/neck area)
            bendInfluence = z / 240.0;
            bendInfluence = clamp(bendInfluence, 0.0, 1.0);
        } else {
            // Back of body (tail area)
            bendInfluence = z / 380.0;
            bendInfluence = clamp(bendInfluence, -1.0, 0.0);
        }

        bendInfluence *= bodyBend;

        // Apply bend as rotation around Y-axis (side-to-side lean into turns)
        // This rotates the body in the XZ plane (horizontal), creating natural leaning
        // Original C1/C2 formula: bx = cos*x - sin*z, bz = cos*z + sin*x
        float cosB = cos(bendInfluence);
        float sinB = sin(bendInfluence);

        // Rotate in XZ plane to create side-to-side spine curve (matches original)
        bentPosition.x = animatedPosition.x * cosB - animatedPosition.z * sinB;
        bentPosition.z = animatedPosition.z * cosB + animatedPosition.x * sinB;
        // Y coordinate is preserved here, will be affected by pitch bend
    }

    // PITCH BEND: Apply forward/back tilt for up/down look direction
    // This rotates the upper body around the X-axis based on Z position
    // Positive pitchBend = looking up (head tilts back), Negative = looking down (head tilts forward)
    if (abs(pitchBend) > 0.001 && useAnimation) {
        float z = bentPosition.z;
        float pitchInfluence;

        // Only apply to the front half of the body (z > 0)
        // The back/tail should not be affected by looking up/down
        if (z > 0.0) {
            // Front of body: more influence toward head
            pitchInfluence = z / 240.0;
            pitchInfluence = clamp(pitchInfluence, 0.0, 1.0);
        } else {
            // Back of body: no pitch influence (tail stays in place)
            pitchInfluence = 0.0;
        }

        pitchInfluence *= pitchBend;

        // Rotate around X-axis (pitch) - affects Y and Z
        // Looking down (negative pitch): head moves down (-Y) and forward (+Z)
        // Looking up (positive pitch): head moves up (+Y) and backward (-Z)
        float cosP = cos(pitchInfluence);
        float sinP = sin(pitchInfluence);

        float newY = bentPosition.y * cosP - bentPosition.z * sinP;
        float newZ = bentPosition.y * sinP + bentPosition.z * cosP;
        bentPosition.y = newY;
        bentPosition.z = newZ;
    }

    // Apply character scale, then model and instanced transformations to the bent position
    vec3 scaledPosition = bentPosition * characterScale;
    mat4 fullModelMatrix = model * instancedMatrix;  // Combined transform including character rotation
    vec4 worldPosition = fullModelMatrix * vec4(scaledPosition, 1.0);

    // Transform normal to world space - MUST use full model matrix (including instancedMatrix)
    // so that normals rotate with the character and lighting changes as they turn
    mat3 normalMatrix = mat3(transpose(inverse(fullModelMatrix)));
    surfaceNormal = normalize(normalMatrix * normal);

    // Use directional sun light - sunDirection points FROM sun TO scene
    // toLightVector should point FROM surface TO light, so negate it
    // If sunDirection is not set (zero), fall back to a default overhead sun
    vec3 effectiveSunDir = sunDirection;
    if (length(effectiveSunDir) < 0.5) {
        effectiveSunDir = normalize(vec3(0.3, -0.8, 0.2));  // Default: high sun, slight angle
    }
    toLightVector = -effectiveSunDir;  // Negate: direction TO the light

    // Apply view and projection transformations
    vec4 viewPosition = view * worldPosition;
    vec4 projectedPosition = projection * viewPosition;

    // Pass through texture coordinates and face alpha
    texCoord0 = texCoord;
    faceAlpha0 = faceAlpha;

    // Calculate horizontal distance to camera for fade-out effect (ignore Y for spawner culling)
    vec2 horizontalCameraPos = vec2(cameraPos.x, cameraPos.z);
    vec2 horizontalWorldPos = vec2(worldPosition.x, worldPosition.z);
    distanceFromCamera = length(horizontalCameraPos - horizontalWorldPos);

    // Calculate view direction for specular highlights
    worldPos = worldPosition.xyz;
    viewDirection = normalize(cameraPos - worldPosition.xyz);

    // Fog map coordinate - uses world position normalized to terrain size
    fogMapCoord = vec2(worldPosition.x / (terrainWidth * tileWidth), worldPosition.z / (terrainHeight * tileWidth));

    // Calculate position in light space for shadow mapping
    FragPosLightSpace = lightSpaceMatrix * worldPosition;

    // Set the final position of the vertex
    gl_Position = projectedPosition;
}
