#version 330 core

in vec2 TexCoord;
in vec3 FragPos;
in vec3 ViewDir;
in float DistanceFromCamera;

out vec4 FragColor;

// Fog parameters for this plane
uniform vec3 fogColor;
uniform float fogCeiling;      // Y position of the fog ceiling (top of fog)
uniform float fogTransp;       // Transparency parameter - controls density falloff
uniform float fogLimit;        // Maximum fog opacity (0-255 scale)
uniform float fogDepth;        // Height of fog volume (ceiling - ground level)

// Camera info
uniform vec3 cameraPos;
uniform float view_distance;

// Fog zone mask - to clip fog to only the fog zone area
uniform sampler2D fogMapTexture;
uniform int fogZoneIndex;      // Which fog zone this plane represents (0-15)
uniform float terrainWidth;    // Terrain width in tiles
uniform float terrainHeight;   // Terrain height in tiles
uniform float tileWidth;       // Size of each tile in world units

// Edge tapering constants
const float EDGE_TAPER_RADIUS = 192.0;  // World units - distance over which fog tapers at edges
const float EDGE_SAMPLE_RADIUS = 128.0; // World units - radius for edge detection sampling

// Sample fog zone from fog map
int sampleFogZone(vec2 worldXZ) {
    float terrainWorldWidth = terrainWidth * tileWidth;
    float terrainWorldHeight = terrainHeight * tileWidth;
    vec2 uv = vec2(worldXZ.x / terrainWorldWidth, worldXZ.y / terrainWorldHeight);
    if (uv.x < 0.0 || uv.x > 1.0 || uv.y < 0.0 || uv.y > 1.0) return -1;
    float fogIndexRaw = texture(fogMapTexture, uv).r * 255.0;
    return int(fogIndexRaw + 0.5) - 1;
}

// Estimate how deep inside the fog zone this position is (0 = at edge, 1 = deep inside)
// Uses multi-scale sampling to find distance to nearest zone boundary
float estimateZoneDepth(vec2 worldXZ, float terrainWorldWidth, float terrainWorldHeight) {
    vec2 uv = vec2(worldXZ.x / terrainWorldWidth, worldXZ.y / terrainWorldHeight);

    // Sample at multiple distances and directions to estimate depth into zone
    float minNonZoneDistance = EDGE_TAPER_RADIUS;  // Start at max taper distance

    // Sample at 3 different radii for more accurate edge detection
    float radii[3] = float[3](32.0, 96.0, 192.0);  // Near, mid, far samples

    for (int r = 0; r < 3; r++) {
        float sampleRadius = radii[r];
        vec2 uvRadius = vec2(sampleRadius / terrainWorldWidth, sampleRadius / terrainWorldHeight);

        // Sample in 8 directions
        for (int i = 0; i < 8; i++) {
            float angle = float(i) * 0.785398;  // 45 degree increments
            vec2 offset = vec2(cos(angle), sin(angle)) * uvRadius;
            vec2 sampleUV = uv + offset;

            if (sampleUV.x >= 0.0 && sampleUV.x <= 1.0 && sampleUV.y >= 0.0 && sampleUV.y <= 1.0) {
                float idx = texture(fogMapTexture, sampleUV).r * 255.0;
                int sampleZone = int(idx + 0.5) - 1;

                // Found a different zone - this is an edge
                if (sampleZone != fogZoneIndex) {
                    minNonZoneDistance = min(minNonZoneDistance, sampleRadius);
                }
            } else {
                // Edge of terrain also counts as zone boundary
                minNonZoneDistance = min(minNonZoneDistance, sampleRadius);
            }
        }
    }

    // Convert distance to depth factor (0 at edge, 1 deep inside)
    return clamp(minNonZoneDistance / EDGE_TAPER_RADIUS, 0.0, 1.0);
}

void main()
{
    float terrainWorldWidth = terrainWidth * tileWidth;
    float terrainWorldHeight = terrainHeight * tileWidth;

    // Check if this fragment is within the correct fog zone
    int currentZone = sampleFogZone(FragPos.xz);
    if (currentZone != fogZoneIndex) {
        discard;  // Not in our fog zone
    }

    // Check if camera is inside this fog zone - if so, internal fog handles it
    int cameraZone = sampleFogZone(cameraPos.xz);
    if (cameraZone == fogZoneIndex && cameraPos.y < fogCeiling) {
        // Camera is inside this fog zone and below ceiling - internal fog handles rendering
        discard;
    }

    // ========== EXTERNAL FOG VISIBILITY ==========
    // The fog plane is positioned at the fog CEILING height.
    // The fog volume extends from fogCeiling down to (fogCeiling - fogDepth).
    // We render fog visibility based on the view ray passing through this volume.

    float fogGroundLevel = fogCeiling - fogDepth;
    float safeTransp = max(fogTransp, 0.5);
    float maxOpacity = fogLimit / 255.0;

    // The fragment is on the fog ceiling plane
    // Calculate how the view ray passes through the fog volume

    // For external viewing, the key is: can we see into the fog volume?
    // The plane is at ceiling height, so:
    // - From above: we're looking down at the fog top surface
    // - From the side (same height): we see the fog "wall" horizontally
    // - From below: we're looking up at the fog from underneath

    float cameraHeightRelativeToCeiling = cameraPos.y - fogCeiling;
    float cameraHeightRelativeToGround = cameraPos.y - fogGroundLevel;

    // Calculate effective fog path length
    float effectiveThickness;
    float baseOpacityMultiplier = 1.0;

    if (cameraHeightRelativeToCeiling > 0.0) {
        // Camera is ABOVE the fog ceiling - looking down at the fog surface
        // The fog plane represents the top of the fog - we see fog accumulated below
        effectiveThickness = fogDepth;  // Full depth visible looking down
        // Reduce opacity based on how high above we are (fog looks thinner from high up)
        float heightFactor = 1.0 - smoothstep(0.0, fogDepth * 3.0, cameraHeightRelativeToCeiling);
        baseOpacityMultiplier = max(0.3, heightFactor);
    } else if (cameraHeightRelativeToGround < 0.0) {
        // Camera is BELOW the fog volume - looking up at fog from underneath
        effectiveThickness = fogDepth;
        float belowFactor = 1.0 - smoothstep(0.0, fogDepth * 2.0, -cameraHeightRelativeToGround);
        baseOpacityMultiplier = max(0.2, belowFactor);
    } else {
        // Camera is AT fog ceiling height or within the fog volume
        // (but outside the fog zone horizontally - internal fog handles inside)
        // See fog "wall" horizontally - this is the key for seeing fog from a distance!
        effectiveThickness = fogDepth * 2.0;  // Horizontal view through fog is longer
        baseOpacityMultiplier = 1.0;
    }

    // ========== DISTANCE-BASED FOG DENSITY ==========
    // External fog should be clearly visible at distance to help players identify fog areas

    float baseDensity = effectiveThickness / safeTransp;

    // Distance visibility curve:
    // - Very close (< 50): fade in to avoid popping
    // - Close to medium (50-300): strong visibility - this is key for seeing fog approach
    // - Medium to far (300-600): maintain visibility
    // - Far (> view_distance * 0.7): fade out
    float nearFade = smoothstep(30.0, 100.0, DistanceFromCamera);
    float farFade = 1.0 - smoothstep(view_distance * 0.5, view_distance * 0.9, DistanceFromCamera);

    // Boost mid-range visibility significantly - this is where players need to see the fog
    float midRangeBoost = smoothstep(80.0, 250.0, DistanceFromCamera) *
                          (1.0 - smoothstep(400.0, 700.0, DistanceFromCamera));
    midRangeBoost = midRangeBoost * 0.6 + 0.4;  // Range 0.4 to 1.0

    float distanceFactor = nearFade * farFade * midRangeBoost;

    // Calculate opacity - use a higher multiplier for better visibility
    float opacity = baseDensity * 0.2 * distanceFactor * baseOpacityMultiplier;

    // Clamp to max opacity (but allow some variance based on viewing angle)
    opacity = clamp(opacity, 0.0, maxOpacity * 0.85);

    // ========== EDGE TAPERING ==========
    // Fog should taper off at the edges of fog zones, not have hard boundaries
    // Calculate how deep inside the fog zone this fragment is
    float zoneDepth = estimateZoneDepth(FragPos.xz, terrainWorldWidth, terrainWorldHeight);

    // Apply smooth tapering curve - fog fades gradually at edges
    // Use a curve that keeps fog strong in the middle but fades nicely at edges
    float edgeTaper = smoothstep(0.0, 0.6, zoneDepth);

    // Additional fine-grained edge softening using nearby samples
    vec2 fogUV = vec2(FragPos.x / terrainWorldWidth, FragPos.z / terrainWorldHeight);
    float sampleRadius = EDGE_SAMPLE_RADIUS;
    vec2 uvRadius = vec2(sampleRadius / terrainWorldWidth, sampleRadius / terrainWorldHeight);

    float sameZoneCount = 0.0;
    float totalSamples = 0.0;

    // Sample in a weighted pattern for smoother edges
    for (int dx = -2; dx <= 2; dx++) {
        for (int dy = -2; dy <= 2; dy++) {
            float dist = length(vec2(float(dx), float(dy)));
            if (dist > 2.5) continue;  // Skip corners for circular pattern

            float weight = 1.0 - dist * 0.3;  // Weight by distance
            vec2 sampleUV = fogUV + vec2(float(dx), float(dy)) * uvRadius * 0.5;

            if (sampleUV.x >= 0.0 && sampleUV.x <= 1.0 && sampleUV.y >= 0.0 && sampleUV.y <= 1.0) {
                float idx = texture(fogMapTexture, sampleUV).r * 255.0;
                int sampleZone = int(idx + 0.5) - 1;
                totalSamples += weight;
                if (sampleZone == fogZoneIndex) {
                    sameZoneCount += weight;
                }
            }
        }
    }

    float edgeSoftness = 1.0;
    if (totalSamples > 0.0) {
        edgeSoftness = sameZoneCount / totalSamples;
        edgeSoftness = smoothstep(0.0, 0.8, edgeSoftness);
    }

    // Combine both edge factors for smooth tapering
    opacity *= edgeTaper * edgeSoftness;

    // ========== HORIZONTAL APPROACH TRANSITION ==========
    // When camera is outside the fog zone but approaching horizontally,
    // fade out the external fog plane as internal fog takes over
    if (cameraZone != fogZoneIndex) {
        // Camera is outside this fog zone - check horizontal distance to zone
        // Sample around camera to see if we're close to entering
        float closestZoneDistance = 1000.0;
        vec2 cameraUV = vec2(cameraPos.x / terrainWorldWidth, cameraPos.z / terrainWorldHeight);

        // Check in 8 directions at multiple distances
        for (int i = 0; i < 8; i++) {
            float angle = float(i) * 0.785398;
            for (float dist = 32.0; dist <= 128.0; dist += 32.0) {
                vec2 offset = vec2(cos(angle), sin(angle)) * dist;
                vec2 checkPos = cameraPos.xz + offset;
                int checkZone = sampleFogZone(checkPos);
                if (checkZone == fogZoneIndex) {
                    closestZoneDistance = min(closestZoneDistance, dist);
                }
            }
        }

        // If camera is very close to this fog zone, start fading out external plane
        // (internal fog will be taking over)
        if (closestZoneDistance < 100.0) {
            float horizontalApproachFade = smoothstep(32.0, 100.0, closestZoneDistance);
            opacity *= horizontalApproachFade;
        }
    }

    // Output fog color with calculated opacity
    FragColor = vec4(fogColor, opacity);
}
