#version 330

in vec2 v_texCoord;
in vec3 v_worldPos;
in vec3 v_normal;
in vec3 v_viewDir;
in vec2 v_fogMapCoord;
in vec4 v_fragPosLightSpace;

out vec4 outputColor;

// PBR textures
uniform sampler2D baseColorTexture;           // Unit 0
uniform sampler2D metallicRoughnessTexture;   // Unit 1 (G=roughness, B=metallic)
uniform sampler2D normalTexture;              // Unit 2
uniform sampler2D occlusionTexture;           // Unit 3
uniform sampler2D emissiveTexture;            // Unit 4

// Texture presence flags
uniform bool hasBaseColorTexture = false;
uniform bool hasMetallicRoughnessTexture = false;
uniform bool hasNormalTexture = false;
uniform bool hasOcclusionTexture = false;
uniform bool hasEmissiveTexture = false;

// PBR factors (used when texture is missing, or as multipliers)
uniform vec4 baseColorFactor = vec4(1.0);
uniform float metallicFactor = 1.0;
uniform float roughnessFactor = 1.0;
uniform vec3 emissiveFactor = vec3(0.0);
uniform float normalScale = 1.0;
uniform float occlusionStrength = 1.0;

// Alpha mode: 0=OPAQUE, 1=MASK, 2=BLEND
uniform int alphaMode = 0;
uniform float alphaCutoff = 0.5;

// Transmission (KHR_materials_transmission approximation)
uniform float transmissionFactor = 0.0;
uniform vec3 attenuationColor = vec3(1.0);

// Environment (matching basic_shader)
uniform vec3 cameraPos;
uniform float view_distance;
uniform vec4 distanceColor;
uniform vec3 lightDirection = vec3(-0.4, -0.85, -0.4);
uniform float rainIntensity = 0.0;
uniform float visibilityMultiplier = 1.0;
uniform int timeOfDay = 0;  // 0=DAY, 1=DUSK, 2=NIGHT (explicit, not derived from visibility)
uniform float time;

// Shadow mapping
uniform sampler2D shadowMap;
uniform bool enableShadows = false;

// Constants
const float PI = 3.14159265359;

// ========== PBR LIGHTING FUNCTIONS ==========

// Normal Distribution Function (GGX/Trowbridge-Reitz)
float DistributionGGX(vec3 N, vec3 H, float roughness)
{
    float a = roughness * roughness;
    float a2 = a * a;
    float NdotH = max(dot(N, H), 0.0);
    float NdotH2 = NdotH * NdotH;

    float denom = (NdotH2 * (a2 - 1.0) + 1.0);
    denom = PI * denom * denom;

    return a2 / max(denom, 0.0001);
}

// Geometry function (Schlick-GGX)
float GeometrySchlickGGX(float NdotV, float roughness)
{
    float r = roughness + 1.0;
    float k = (r * r) / 8.0;

    float denom = NdotV * (1.0 - k) + k;
    return NdotV / max(denom, 0.0001);
}

// Smith's method for geometry
float GeometrySmith(vec3 N, vec3 V, vec3 L, float roughness)
{
    float NdotV = max(dot(N, V), 0.0);
    float NdotL = max(dot(N, L), 0.0);
    float ggx2 = GeometrySchlickGGX(NdotV, roughness);
    float ggx1 = GeometrySchlickGGX(NdotL, roughness);
    return ggx1 * ggx2;
}

// Fresnel (Schlick approximation)
vec3 fresnelSchlick(float cosTheta, vec3 F0)
{
    return F0 + (1.0 - F0) * pow(clamp(1.0 - cosTheta, 0.0, 1.0), 5.0);
}

// Shadow calculation (same as basic_shader)
float ShadowCalculation(vec4 fragPosLightSpace)
{
    vec3 projCoords = fragPosLightSpace.xyz / fragPosLightSpace.w;
    projCoords = projCoords * 0.5 + 0.5;

    if (projCoords.z > 1.0 || projCoords.x < 0.0 || projCoords.x > 1.0 ||
        projCoords.y < 0.0 || projCoords.y > 1.0) {
        return 0.0;
    }

    float currentDepth = projCoords.z;
    float bias = 0.001;

    // PCF for soft shadows
    float shadow = 0.0;
    vec2 texelSize = 1.0 / textureSize(shadowMap, 0);
    for(int x = -2; x <= 2; ++x) {
        for(int y = -2; y <= 2; ++y) {
            float pcfDepth = texture(shadowMap, projCoords.xy + vec2(x, y) * texelSize).r;
            shadow += currentDepth - bias > pcfDepth ? 1.0 : 0.0;
        }
    }
    shadow /= 25.0;

    return shadow;
}

void main()
{
    // ========== SAMPLE TEXTURES ==========

    // Base color
    vec4 baseColor = baseColorFactor;
    if (hasBaseColorTexture) {
        baseColor *= texture(baseColorTexture, v_texCoord);
    }

    // Alpha handling
    if (alphaMode == 1 && baseColor.a < alphaCutoff) {
        discard;
    }

    // Metallic-Roughness
    float metallic = metallicFactor;
    float roughness = roughnessFactor;
    if (hasMetallicRoughnessTexture) {
        vec4 mr = texture(metallicRoughnessTexture, v_texCoord);
        roughness *= mr.g;  // Green channel = roughness
        metallic *= mr.b;   // Blue channel = metallic
    }
    roughness = clamp(roughness, 0.04, 1.0);

    // Normal
    vec3 N = normalize(v_normal);
    // TODO: Normal mapping when tangents are available
    // if (hasNormalTexture) { ... }

    vec3 V = normalize(v_viewDir);
    vec3 L = normalize(-lightDirection);
    vec3 H = normalize(V + L);

    // ========== PBR LIGHTING ==========

    // Reflectance at normal incidence (F0)
    // Dielectrics: 0.04, metals: albedo color
    vec3 F0 = vec3(0.04);
    F0 = mix(F0, baseColor.rgb, metallic);

    // Cook-Torrance BRDF
    float NDF = DistributionGGX(N, H, roughness);
    float G = GeometrySmith(N, V, L, roughness);
    vec3 F = fresnelSchlick(max(dot(H, V), 0.0), F0);

    // Specular
    vec3 numerator = NDF * G * F;
    float denominator = 4.0 * max(dot(N, V), 0.0) * max(dot(N, L), 0.0) + 0.0001;
    vec3 specular = numerator / denominator;

    // Energy conservation
    vec3 kS = F;
    vec3 kD = vec3(1.0) - kS;
    kD *= 1.0 - metallic;  // Metals have no diffuse

    // Diffuse
    float NdotL = max(dot(N, L), 0.0);
    vec3 diffuse = kD * baseColor.rgb / PI;

    // Light contribution (using a white light)
    vec3 lightColor = vec3(1.0);
    vec3 Lo = (diffuse + specular) * lightColor * NdotL;

    // Ambient (simple IBL approximation)
    vec3 ambient = vec3(0.35) * baseColor.rgb;

    vec3 color = ambient + Lo;

    // ========== OCCLUSION ==========
    if (hasOcclusionTexture) {
        float ao = texture(occlusionTexture, v_texCoord).r;
        color = mix(color, color * ao, occlusionStrength);
    }

    // ========== EMISSIVE ==========
    vec3 emissive = emissiveFactor;
    if (hasEmissiveTexture) {
        emissive *= texture(emissiveTexture, v_texCoord).rgb;
    }
    color += emissive;

    // ========== SHADOWS ==========
    if (enableShadows) {
        float shadow = ShadowCalculation(v_fragPosLightSpace);
        color *= (1.0 - shadow * 0.5);
    }

    // ========== DISTANCE FOG (matching basic_shader) ==========
    float fragDistance = length(v_worldPos - cameraPos);
    float min_distance = view_distance * 0.50;
    float max_distance = view_distance * 0.95;
    float fogFactor = 0.0;

    if (fragDistance > min_distance) {
        fogFactor = clamp((fragDistance - min_distance) / (max_distance - min_distance), 0.0, 1.0);
        fogFactor = min(fogFactor, 0.45);
    }
    color = mix(color, distanceColor.rgb, fogFactor);

    // ========== RAIN EFFECTS ==========
    if (rainIntensity > 0.0) {
        // Darken and desaturate
        float rainDarkening = 1.0 - (rainIntensity * 0.35);
        color *= rainDarkening;

        vec3 rainTint = vec3(0.85, 0.88, 1.0);
        color = mix(color, color * rainTint, rainIntensity * 0.4);

        // Wet surface specular
        float wetSpecular = pow(max(dot(N, L), 0.0), 96.0) * rainIntensity * 0.4;
        color += vec3(wetSpecular) * rainTint;
    }

    // ========== TIME OF DAY ==========
    // Use explicit timeOfDay uniform (0=DAY, 1=DUSK, 2=NIGHT) instead of deriving from visibility
    // visibilityMultiplier still controls intensity of effects
    if (timeOfDay > 0) {
        float effectIntensity = 1.0 - visibilityMultiplier;  // How strong the effect should be

        bool isDusk = (timeOfDay == 1);
        bool isNight = (timeOfDay == 2);

        // Darkening - dusk is brighter than night
        float darkening = 1.0;
        if (isDusk) {
            darkening = 1.0 - (effectIntensity * 0.50);
        } else if (isNight) {
            darkening = 1.0 - (effectIntensity * 0.90);
        }
        color *= darkening;

        // Desaturation - dusk keeps more color than night
        vec3 grayscale = vec3(dot(color, vec3(0.299, 0.587, 0.114)));
        float desatAmount = 0.0;
        if (isDusk) {
            desatAmount = effectIntensity * 0.30;
        } else if (isNight) {
            desatAmount = effectIntensity * 0.85;
        }
        color = mix(color, grayscale, desatAmount);

        // Color tinting - warm for dusk, cool for night (mutually exclusive)
        vec3 duskTint = vec3(1.3, 0.85, 0.55);   // Warm orange/golden sunset
        vec3 nightTint = vec3(0.4, 0.5, 0.8);    // Cool blue moonlight

        if (isDusk) {
            color = mix(color, color * duskTint, effectIntensity * 0.7);
        } else if (isNight) {
            color = mix(color, color * nightTint, effectIntensity * 0.6);
        }
    }

    // ========== HARD EDGE FADE ==========
    float fadeStart = view_distance * 0.70;
    float fadeEnd = view_distance * 0.95;
    float hardFade = smoothstep(fadeStart, fadeEnd, fragDistance);

    // At dusk/night, fade to appropriate color instead of daytime distance color
    vec3 fadeTargetColor = distanceColor.rgb;
    if (timeOfDay > 0) {
        vec3 duskFadeColor = vec3(0.25, 0.12, 0.06);  // Warm amber for dusk
        vec3 nightFadeColor = vec3(0.02, 0.03, 0.06); // Dark blue for night
        float effectIntensity = 1.0 - visibilityMultiplier;

        if (timeOfDay == 1) {
            fadeTargetColor = mix(distanceColor.rgb, duskFadeColor, effectIntensity);
        } else if (timeOfDay == 2) {
            fadeTargetColor = mix(distanceColor.rgb, nightFadeColor, effectIntensity);
        }
    }
    color = mix(color, fadeTargetColor, hardFade);

    // ========== TRANSMISSION (simplified) ==========
    // Apply attenuation color tint for transmissive materials
    if (transmissionFactor > 0.0) {
        color *= attenuationColor;
    }

    // Output
    float alpha = (alphaMode == 2) ? baseColor.a : 1.0;

    // For transmission, use (1 - transmissionFactor) as opacity
    // transmissionFactor=1 means fully transparent, =0 means opaque
    if (transmissionFactor > 0.0) {
        alpha = 1.0 - transmissionFactor;
    }

    outputColor = vec4(color, alpha);
}
