#version 330 core

in vec2 texCoord0;
in vec2 texCoord1;
in vec3 surfaceNormal;
in vec3 toLightVector;
in vec3 worldPos;
in float distFromCamera;

out vec4 outputColor;

uniform sampler2D basic_texture;
uniform float view_distance;

// Underwater fog color - dark blue-teal for ocean depths
const vec3 underwaterFogColor = vec3(0.02, 0.08, 0.14);
uniform float waterLevel;

// Time of day
uniform float visibilityMultiplier = 1.0;

void main()
{
    // Sample terrain texture
    vec4 sC = textureLod(basic_texture, fract(texCoord0), 0.0);
    vec3 texColor = vec3(sC.b, sC.g, sC.r);

    // Simple lighting
    vec3 normal = normalize(surfaceNormal);
    vec3 lightDir = normalize(toLightVector);

    float diffuse = max(dot(normal, lightDir), 0.0);
    // Underwater lighting is dimmer
    float brightness = 0.3 + diffuse * 0.3;
    vec3 terrainColor = texColor * brightness;

    // Depth-based darkening
    float depthBelow = max(waterLevel - worldPos.y, 0.0);
    float depthFactor = clamp(depthBelow / 400.0, 0.0, 1.0);  // Full darkening at 400 units deep
    terrainColor *= (1.0 - depthFactor * 0.6);  // Up to 60% darker at max depth

    // === UNDERWATER DISTANCE FOG ===
    // Strong fog effect for ocean floor - visibility limited underwater
    float fogStart = 50.0;        // Start fog very close
    float fogEnd = 400.0;         // Mostly fogged by 25 tiles
    float fogAmount = smoothstep(fogStart, fogEnd, distFromCamera);

    // Stronger fog falloff underwater
    fogAmount = pow(fogAmount, 0.8);

    // Blend with underwater fog color
    vec3 finalColor = mix(terrainColor, underwaterFogColor, fogAmount);

    // Add slight blue tint overall (underwater effect)
    finalColor = mix(finalColor, finalColor * vec3(0.7, 0.85, 1.0), 0.3);

    // === TIME OF DAY (NIGHT) ===
    if (visibilityMultiplier < 1.0) {
        float nightAmount = 1.0 - visibilityMultiplier;

        // Even darker underwater at night
        float darkening = 1.0 - (nightAmount * 0.9);
        finalColor *= darkening;

        // Stronger night fog underwater
        float nightFogStart = 30.0;
        float nightFogEnd = 200.0;
        float nightFogAmount = smoothstep(nightFogStart, nightFogEnd, distFromCamera);
        vec3 nightFogColor = vec3(0.01, 0.02, 0.04);  // Very dark blue-black
        finalColor = mix(finalColor, nightFogColor, nightFogAmount * nightAmount);
    }

    // === HARD EDGE FADE ===
    float fadeStart = view_distance * 0.5;
    float fadeEnd = view_distance * 0.8;
    float hardFade = smoothstep(fadeStart, fadeEnd, distFromCamera);

    // Fade to underwater fog color (not sky)
    vec3 fadeTargetColor = underwaterFogColor;
    if (visibilityMultiplier < 1.0) {
        float nightAmount = 1.0 - visibilityMultiplier;
        fadeTargetColor = mix(underwaterFogColor, vec3(0.01, 0.02, 0.04), nightAmount);
    }
    finalColor = mix(finalColor, fadeTargetColor, hardFade);

    outputColor = vec4(finalColor, 1.0);
}
