#version 330 core
out vec4 FragColor;

in vec2 texCoord0;
in vec3 FragPos;
in vec4 FragPosLightSpace;

uniform sampler2D basic_texture;
uniform sampler2D shadowMap;
uniform vec3 lightDirection = vec3(-0.5, -1.0, -0.5);  // Unified sun direction
uniform bool enableShadows = false;
// Balanced lighting for simple geometry - brighter than terrain for visibility
uniform float ambientStrength = 0.75;  // Bright base for world objects
uniform float diffuseStrength = 0.45;  // Moderate sun strength

// New uniforms for impact marker coloring
uniform bool useCustomColor = false;
uniform vec3 customColor = vec3(1.0, 0.0, 0.0);

// Underwater visibility - uses underwaterStateTexture for per-fragment water level
uniform sampler2D underwaterStateTexture;
uniform vec3 cameraPos = vec3(0.0);
uniform float terrainWidth = 128.0;
uniform float terrainHeight = 128.0;
uniform float tileWidth = 16.0;
uniform float visibilityMultiplier = 1.0;  // Time-of-day visibility (1.0 = day, 0.35 = night)
uniform float view_distance = 8000.0;
uniform vec4 distanceColor = vec4(0.7, 0.8, 0.9, 1.0);  // Fade color (night = dark blue)

float ShadowCalculation(vec4 fragPosLightSpace)
{
    if (!enableShadows) return 0.0;
    
    // Perform perspective divide
    vec3 projCoords = fragPosLightSpace.xyz / fragPosLightSpace.w;
    
    // Transform to [0,1] range
    projCoords = projCoords * 0.5 + 0.5;
    
    // DEBUG: For now, just return a test pattern based on screen position
    // This will help us see if shadow calculation is running at all
    float testShadow = step(0.5, fract(gl_FragCoord.x * 0.01)) * step(0.5, fract(gl_FragCoord.y * 0.01));
    if (testShadow > 0.5) return 1.0;
    
    return 0.0;
}

void main()
{
    // Check if we should use custom color (for impact markers)
    if (useCustomColor) {
        FragColor = vec4(customColor, 1.0);
        return;
    }
    
    // Use texture for normal rendering
    vec4 texColor = texture(basic_texture, texCoord0);
    
    // Apply basic lighting (matching terrain ambient/diffuse levels)
    vec3 lightDir = normalize(-lightDirection);

    // Use terrain-matched lighting values for consistency
    // Terrain uses: ambient 0.55, diffuse 0.55, then applies cloud modulation (0.25-1.1)
    // Average cloud modulation ~0.6, so effective brightness is ~0.55 * 0.6 = 0.33
    float ambient = 0.55;
    float diffuse = max(dot(vec3(0, 0, 1), lightDir), 0.0) * 0.55;

    float lightFactor = ambient + diffuse;

    // Simulate average cloud shadow effect to match terrain brightness
    // Terrain has cloudModulation range 0.25-1.1, average ~0.6
    float cloudModulation = 0.6;
    lightFactor *= cloudModulation;

    // Apply shadow if enabled
    float shadow = 0.0;
    if (enableShadows) {
        shadow = ShadowCalculation(FragPosLightSpace);
    }

    vec3 finalColor = texColor.rgb * lightFactor * (1.0 - shadow * 0.5);

    // ========== ATMOSPHERIC DEPTH EFFECTS ==========
    // Match terrain shader for consistent day/night appearance
    float distance = length(FragPos - cameraPos);

    // Exponential fog (same as terrain - rapid increase then plateaus)
    float min_distance = view_distance * 0.50;
    float max_distance = view_distance * 0.95;

    // Exponential fog formula matching terrain
    float fogDensity = 0.00012;
    float fogFactor = 1.0 - exp(-distance * fogDensity);
    fogFactor = clamp(fogFactor, 0.0, 0.65);  // Match terrain's 0.65 cap

    // Height-based atmospheric scattering (matching terrain)
    float heightFactor = clamp((200.0 - FragPos.y) / 300.0, 0.0, 0.4);
    fogFactor += heightFactor * 0.15;

    // Distance-based desaturation (matching terrain)
    float desaturationFactor = clamp(distance / (view_distance * 0.8), 0.0, 0.5);

    // Aerial perspective shift toward blue haze (matching terrain)
    // At night, use dark night fog color instead of daytime haze
    vec3 aerialColor = vec3(0.6, 0.75, 0.9);
    if (visibilityMultiplier < 1.0) {
        vec3 nightAerialColor = vec3(0.02, 0.03, 0.06);
        float nightAmount = 1.0 - visibilityMultiplier;
        aerialColor = mix(aerialColor, nightAerialColor, nightAmount);
    }
    float aerialBlend = clamp(distance / (view_distance * 1.1), 0.0, 0.4);

    // Apply desaturation
    vec3 desaturatedColor = mix(finalColor, vec3(dot(finalColor, vec3(0.299, 0.587, 0.114))), desaturationFactor);

    // Apply aerial perspective
    vec3 aerialBlended = mix(desaturatedColor, aerialColor * desaturatedColor, aerialBlend);

    // Blend with distance fog - at night, use night fog color instead of daytime distance color
    vec3 fogTargetColor = distanceColor.rgb;
    if (visibilityMultiplier < 1.0) {
        vec3 nightFogColor = vec3(0.02, 0.03, 0.06);  // Dark blue-black night fog
        float nightAmount = 1.0 - visibilityMultiplier;
        fogTargetColor = mix(distanceColor.rgb, nightFogColor, nightAmount);
    }
    finalColor = mix(aerialBlended, fogTargetColor, fogFactor);

    // Contrast reduction at distance (matching terrain)
    if (distance > min_distance) {
        float contrastReduction = clamp((distance - min_distance) / (max_distance - min_distance), 0.0, 0.3);
        // At night, use dark blue fog color instead of gray for contrast reduction
        vec3 contrastTarget = vec3(0.5);
        if (visibilityMultiplier < 1.0) {
            float nightAmount = 1.0 - visibilityMultiplier;
            contrastTarget = mix(vec3(0.5), vec3(0.02, 0.03, 0.06), nightAmount);
        }
        finalColor = mix(finalColor, contrastTarget, contrastReduction * 0.2);
    }

    // ========== UNDERWATER TINTING ==========
    // Calculate quad coordinates from world position
    vec2 quadCoord = vec2(FragPos.x / tileWidth, FragPos.z / tileWidth) / vec2(terrainWidth, terrainHeight);

    // Sample actual water level at this fragment's position
    float localWaterLevel = texture(underwaterStateTexture, quadCoord).r;

    // Only apply tinting if there is water here and fragment is below it
    if (localWaterLevel > 0.0 && FragPos.y < localWaterLevel) {
        // Calculate depth below water surface
        float underwaterDepth = localWaterLevel - FragPos.y;

        // Calculate view angle factor - looking through more water at shallow angles
        vec3 viewDir = normalize(cameraPos - FragPos);
        float viewDot = abs(viewDir.y);  // 1.0 looking straight down, 0.0 horizontal
        float opticalDepth = underwaterDepth / max(viewDot, 0.15);

        // Normalize to 2 tiles (32 units) - extended range for deeper underwater visibility
        float underwaterFade = clamp(opticalDepth / 32.0, 0.0, 1.0);

        // Water tint color - blue-green tint that matches water
        vec3 waterTint = vec3(0.2, 0.35, 0.45);

        // Fade toward water tint based on optical depth
        // Also darken underwater areas (less light penetration)
        float darkenFactor = 1.0 - underwaterFade * 0.5;  // Up to 50% darker at full depth
        finalColor = mix(finalColor * darkenFactor, waterTint, underwaterFade * 0.7);
    }

    // ========== TIME-OF-DAY EFFECTS (DUSK/NIGHT) ==========
    // When visibility is reduced (dusk/night), darken and desaturate colors
    if (visibilityMultiplier < 1.0) {
        // Calculate how much "night" effect to apply (0 = day, 1 = full night)
        float nightAmount = 1.0 - visibilityMultiplier;

        // 1. Darken the scene very heavily - night should be almost black
        // Up to 90% darker at full night
        float darkening = 1.0 - (nightAmount * 0.90);
        finalColor *= darkening;

        // 2. Very heavy desaturation - colors nearly impossible to distinguish at night
        vec3 grayscale = vec3(dot(finalColor, vec3(0.299, 0.587, 0.114)));
        float desatAmount = nightAmount * 0.85;  // Up to 85% desaturation at full night
        finalColor = mix(finalColor, grayscale, desatAmount);

        // 3. Strong dark-blue moonlight tint
        vec3 nightTint = vec3(0.4, 0.5, 0.8);  // Dark blue-ish moonlight
        float tintAmount = nightAmount * 0.6;  // Strong tint
        finalColor = mix(finalColor, finalColor * nightTint, tintAmount);

        // 4. NIGHT DEPTH FOG - visibility very limited at night
        // Start fog almost immediately to create very limited visibility
        // (distance was already calculated above)
        float nightFogStart = 16.0;    // Start fog at ~1 tile (very close)
        float nightFogEnd = 320.0;     // Full fog at ~20 tiles
        float nightFogAmount = smoothstep(nightFogStart, nightFogEnd, distance);
        // Use hardcoded very dark night color - don't rely on distanceColor uniform
        vec3 nightFogColor = vec3(0.02, 0.03, 0.06);  // Very dark blue-black
        finalColor = mix(finalColor, nightFogColor, nightFogAmount * nightAmount);
    }

    // ========== HARD EDGE FADE ==========
    // Ensure geometry fully fades to sky color before reaching the clip plane
    // Uses radial distance for circular fade (distance already calculated above)
    float fadeStart = view_distance * 0.70;  // Start hard fade at 70% of view distance
    float fadeEnd = view_distance * 0.95;    // Fully faded by 95% of view distance
    float hardFade = clamp((distance - fadeStart) / (fadeEnd - fadeStart), 0.0, 1.0);
    // Smooth the fade curve for more gradual transition
    hardFade = smoothstep(0.0, 1.0, hardFade);
    // At night, fade to dark night color instead of daytime distance color
    vec3 fadeTargetColor = distanceColor.rgb;
    if (visibilityMultiplier < 1.0) {
        vec3 nightFadeColor = vec3(0.02, 0.03, 0.06);
        float nightAmount = 1.0 - visibilityMultiplier;
        // Use pow to make the transition more aggressive toward night color
        fadeTargetColor = mix(distanceColor.rgb, nightFadeColor, pow(nightAmount, 0.5));
    }
    finalColor = mix(finalColor, fadeTargetColor, hardFade);

    FragColor = vec4(finalColor, texColor.a);
}
