#version 330 core
out vec4 outputColor;

in vec3 TexCoords;

uniform samplerCube skybox;
uniform vec4 sky_color;
uniform float time; // Pass the elapsed time from your application
uniform float rainIntensity = 0.0; // 0.0 = clear, 1.0 = heavy rain
uniform float visibilityMultiplier = 1.0;  // Time-of-day visibility (1.0 = day, 0.35 = night)
uniform int timeOfDay = 0;  // 0=DAY, 1=DUSK, 2=NIGHT (explicit, not derived from visibility)

// Time-of-day fog colors
const vec3 DUSK_FOG_COLOR = vec3(0.25, 0.12, 0.06);  // Warm amber for dusk/dawn
const vec3 NIGHT_FOG_COLOR = vec3(0.02, 0.03, 0.06); // Dark blue for night

// Simple pseudo-random function
float random(vec2 st) {
    return fract(sin(dot(st.xy, vec2(12.9898, 78.233))) * 43758.5453123);
}

// Smooth random function
float noise(vec2 st) {
    vec2 i = floor(st);
    vec2 f = fract(st);

    float a = random(i);
    float b = random(i + vec2(1.0, 0.0));
    float c = random(i + vec2(0.0, 1.0));
    float d = random(i + vec2(1.0, 1.0));

    vec2 u = f * f * (3.0 - 2.0 * f);

    return mix(a, b, u.x) +
           (c - a) * u.y * (1.0 - u.x) +
           (d - b) * u.x * u.y;
}

void main()
{
    vec3 finalColor = sky_color.rgb;

    // Apply dawn/dusk/night effects
    // Use explicit timeOfDay uniform (0=DAY, 1=DUSK, 2=NIGHT)
    if (timeOfDay > 0) {
        float effectIntensity = 1.0 - visibilityMultiplier;

        bool isDusk = (timeOfDay == 1);
        bool isNight = (timeOfDay == 2);

        // 1. Darkening - dusk sky is brighter than night
        float darkening = 1.0;
        if (isDusk) {
            darkening = 1.0 - (effectIntensity * 0.35);
        } else if (isNight) {
            darkening = 1.0 - (effectIntensity * 0.7);
        }
        finalColor *= darkening;

        // 2. Desaturation - dusk keeps more color
        float luminance = dot(finalColor, vec3(0.299, 0.587, 0.114));
        float desatAmount = 0.0;
        if (isDusk) {
            desatAmount = effectIntensity * 0.2;
        } else if (isNight) {
            desatAmount = effectIntensity * 0.6;
        }
        finalColor = mix(finalColor, vec3(luminance), desatAmount);

        // 3. Color tinting - warm orange/red for dusk sky, cool for night
        if (isDusk) {
            vec3 duskTint = vec3(1.4, 0.7, 0.4);  // Warm orange/red sunset sky
            finalColor = mix(finalColor, finalColor * duskTint, effectIntensity * 0.6);
            // Blend toward dusk fog color
            finalColor = mix(finalColor, DUSK_FOG_COLOR, effectIntensity * 0.5);
        } else if (isNight) {
            // Blend toward night fog color
            finalColor = mix(finalColor, NIGHT_FOG_COLOR, effectIntensity * 0.8);
        }
    }

    // Apply rain atmosphere effects - darken and desaturate sky
    if (rainIntensity > 0.0) {
        // Gray overcast color for rainy sky
        vec3 overcastColor = vec3(0.45, 0.48, 0.52);

        // Darken the sky
        float darkening = 1.0 - (rainIntensity * 0.4);
        finalColor *= darkening;

        // Blend toward gray overcast
        finalColor = mix(finalColor, overcastColor, rainIntensity * 0.6);

        // Desaturate
        float luminance = dot(finalColor, vec3(0.299, 0.587, 0.114));
        finalColor = mix(finalColor, vec3(luminance), rainIntensity * 0.5);
    }

    outputColor = vec4(finalColor, sky_color.a);
}
