#version 330

// Enhanced terrain vertex shader for top-down map rendering
// Includes height information for depth visualization

layout(location = 0) in vec3 position;
layout(location = 1) in vec3 normal;
layout(location = 2) in vec4 texCoords;

out highp vec2 texCoord0;
out highp vec2 texCoord1;
out vec3 surfaceNormal;
out vec3 worldPos;
out float heightValue;
out float edgeFade;  // 0 at edges, 1 in center

uniform mat4 MVP;
uniform mat4 model;
uniform float maxTerrainHeight;  // Maximum terrain height for normalization
uniform float terrainWidth;      // Terrain width in world units
uniform float terrainHeight;     // Terrain depth in world units
uniform float fadeDistance;      // Distance from edge to start fading (world units)

void main()
{
    vec4 worldPosition = model * vec4(position, 1.0);
    worldPos = worldPosition.xyz;

    texCoord0 = vec2(texCoords[0], texCoords[1]);
    texCoord1 = vec2(texCoords[2], texCoords[3]);
    surfaceNormal = mat3(transpose(inverse(model))) * normal;

    // Normalize height for fragment shader (0 = lowest, 1 = highest)
    heightValue = clamp(position.y / max(maxTerrainHeight, 1.0), 0.0, 1.0);

    // Calculate edge fade based on distance from terrain boundaries
    float fadeX = min(worldPos.x, terrainWidth - worldPos.x);
    float fadeZ = min(worldPos.z, terrainHeight - worldPos.z);
    float minDist = min(fadeX, fadeZ);
    edgeFade = clamp(minDist / max(fadeDistance, 1.0), 0.0, 1.0);

    gl_Position = MVP * vec4(position, 1.0);
}
