#version 330 core

in vec2 texCoord0;
smooth in float alpha0;
in float EdgeFactor;
in vec3 surfaceNormal;
in vec3 toLightVector;
in vec3 viewDirection;
in vec2 heightmapCoord;
in vec2 fogMapCoord;  // For area fog sampling
in vec4 clipSpace;
in vec4 reflectionClipSpace;  // World position in reflection camera space
in highp vec2 cloudTexCoord;  // For cloud shadow sampling
in vec3 worldPos;  // World position for fog ray marching
flat in uint isDangerWater;

out vec4 outputColor;

uniform sampler2D basic_texture;
uniform sampler2D heightmapTexture;
uniform sampler2D skyTexture;
uniform sampler2D reflectionTexture;
uniform int hasReflection;
uniform float time;
uniform vec4 skyColor;
uniform float view_distance;

uniform float waterLevel;
uniform float heightmapScale;

// Area fog uniforms
const int MAX_FOG_ZONES = 16;
uniform int numFogZones = 0;
uniform sampler2D fogMapTexture;
uniform vec3 fogColors[MAX_FOG_ZONES];
uniform float fogLimits[MAX_FOG_ZONES];         // FLimit - max fog cap distance
uniform float fogTransparencies[MAX_FOG_ZONES]; // Transp - controls distance falloff rate
uniform float fogMaxOpacities[MAX_FOG_ZONES];
uniform float fogAltitudes[MAX_FOG_ZONES];  // YBegin - fog ceiling height (world units)

// Camera position for fog calculation
uniform vec3 cameraPos;

// Terrain dimensions for fog calculation
uniform float terrainWidth;   // Number of tiles in X
uniform float terrainHeight;  // Number of tiles in Z
uniform float tileWidth;      // Size of each tile in world units

// Time of day uniforms
uniform float visibilityMultiplier = 1.0;  // Time-of-day visibility (1.0 = day, 0.35 = night)
uniform vec4 distanceColor = vec4(0.7, 0.8, 0.9, 1.0);  // Fade color (night = dark blue)
uniform int timeOfDay = 0;  // 0=DAY, 1=DUSK, 2=NIGHT (explicit, not derived from visibility)

// Time-of-day fog colors
const vec3 DUSK_FOG_COLOR = vec3(0.25, 0.12, 0.06);  // Warm amber for dusk/dawn
const vec3 NIGHT_FOG_COLOR = vec3(0.02, 0.03, 0.06); // Dark blue-black for night

// Camera-centric area fog - when player is inside a fog zone, all visible
// fragments are fogged based on distance and vertical position.
const float FOG_ZONE_TRANSITION_RADIUS = 256.0;  // Distance over which fog fades when entering/leaving zones

int sampleFogZone(vec3 worldPos, float terrainWorldWidth, float terrainWorldHeight) {
    vec2 uv = vec2(worldPos.x / terrainWorldWidth, worldPos.z / terrainWorldHeight);
    if (uv.x < 0.0 || uv.x > 1.0 || uv.y < 0.0 || uv.y > 1.0) return -1;
    float fogIndexRaw = texture(fogMapTexture, uv).r * 255.0;
    return int(fogIndexRaw + 0.5) - 1;
}

float estimateFogZoneDepth(vec3 worldPos, int expectedFogIndex, float terrainWorldWidth, float terrainWorldHeight) {
    vec2 uv = vec2(worldPos.x / terrainWorldWidth, worldPos.z / terrainWorldHeight);
    float sameZoneCount = 0.0;
    float totalSamples = 0.0;
    float sampleRadius = FOG_ZONE_TRANSITION_RADIUS;
    vec2 uvRadius = vec2(sampleRadius / terrainWorldWidth, sampleRadius / terrainWorldHeight);

    for (float dist = 0.25; dist <= 1.0; dist += 0.25) {
        for (float angle = 0.0; angle < 6.28; angle += 0.785) {
            vec2 offset = vec2(cos(angle), sin(angle)) * dist;
            vec2 sampleUV = uv + offset * uvRadius;
            if (sampleUV.x >= 0.0 && sampleUV.x <= 1.0 && sampleUV.y >= 0.0 && sampleUV.y <= 1.0) {
                float fogIndexRaw = texture(fogMapTexture, sampleUV).r * 255.0;
                int sampleFogIndex = int(fogIndexRaw + 0.5) - 1;
                totalSamples += 1.0;
                if (sampleFogIndex == expectedFogIndex) {
                    sameZoneCount += 1.0;
                }
            }
        }
    }
    return totalSamples > 0.0 ? sameZoneCount / totalSamples : 1.0;
}

vec3 applyAreaFog(vec3 color, float distFromCamera, vec3 fragWorldPos) {
    if (numFogZones == 0) return color;

    float terrainWorldWidth = terrainWidth * tileWidth;
    float terrainWorldHeight = terrainHeight * tileWidth;

    int cameraFogIndex = sampleFogZone(cameraPos, terrainWorldWidth, terrainWorldHeight);
    if (cameraFogIndex < 0 || cameraFogIndex >= numFogZones) return color;

    float fogCeiling = fogAltitudes[cameraFogIndex];
    float transp = fogTransparencies[cameraFogIndex];  // Already scaled to our world units
    float fogLimit = fogLimits[cameraFogIndex];        // Raw 0-255 value
    vec3 fogColor = fogColors[cameraFogIndex];

    // ========== SIMPLIFIED FOG MODEL ==========
    // Fixed visibility range: 30 tiles (480 world units)
    // Fog accumulates from 0 at player to FLimit/255 at 30 tiles

    float depthA = max(0.0, fogCeiling - cameraPos.y);      // Camera depth in fog
    float depthB = max(0.0, fogCeiling - fragWorldPos.y);   // Fragment depth in fog

    if (depthA <= 0.0 && depthB <= 0.0) return color;

    // Smooth vertical transition for camera crossing ceiling
    float ceilingTransitionHeight = 96.0;
    float cameraVerticalFade = 1.0;
    if (depthA <= 0.0) {
        float aboveAmount = cameraPos.y - fogCeiling;
        cameraVerticalFade = clamp(1.0 - aboveAmount / ceilingTransitionHeight, 0.0, 1.0);
        if (cameraVerticalFade <= 0.0) return color;
    } else if (depthA < ceilingTransitionHeight) {
        cameraVerticalFade = smoothstep(0.0, ceilingTransitionHeight, depthA);
    }

    // Camera zone edge transition with smooth external-to-internal fog handoff
    float cameraZoneDepth = estimateFogZoneDepth(cameraPos, cameraFogIndex, terrainWorldWidth, terrainWorldHeight);
    float zoneTransitionFade = smoothstep(0.0, 0.7, cameraZoneDepth);
    float edgeVisibilityBoost = (1.0 - zoneTransitionFade);

    const float MAX_FOG_DISTANCE = 480.0;  // 30 tiles * 16 units per tile

    // Prevent division by zero
    float safeTransp = max(transp, 1.0);

    // Distance factor: 0 at camera, 1 at 30 tiles, capped at 1
    float distanceFactor = min(distFromCamera / MAX_FOG_DISTANCE, 1.0);

    // Transp modifies accumulation rate (600 = reference clear visibility)
    float transpFactor = safeTransp / 600.0;

    // Depth factor: deeper in fog = more fog
    float depthFactor = min((depthA + depthB) / 200.0, 2.0);

    // Combine factors
    float fogValue = (distanceFactor * depthFactor * 255.0) / transpFactor;

    // Cap at FLimit (0-255 range)
    fogValue = min(fogValue, fogLimit);
    float fogDensity = fogValue / 255.0;

    // Maintain fog visibility at edge to match external view
    float edgeFogDensity = distanceFactor * (fogLimit / 255.0) * 0.85;
    fogDensity = mix(edgeFogDensity, fogDensity, zoneTransitionFade);
    fogDensity *= cameraVerticalFade;
    fogDensity = clamp(fogDensity, 0.0, 1.0);

    // Edge softening
    vec2 fragUV = vec2(fragWorldPos.x / terrainWorldWidth, fragWorldPos.z / terrainWorldHeight);
    if (fragUV.x >= 0.0 && fragUV.x <= 1.0 && fragUV.y >= 0.0 && fragUV.y <= 1.0) {
        int fragFogIndex = sampleFogZone(fragWorldPos, terrainWorldWidth, terrainWorldHeight);

        if (fragFogIndex >= 0 && fragFogIndex != cameraFogIndex) {
            vec2 uvSampleRadius = vec2(128.0 / terrainWorldWidth, 128.0 / terrainWorldHeight);
            float totalWeight = 0.0;
            float sameZoneWeight = 0.0;

            for (float dx = -3.0; dx <= 3.0; dx += 1.0) {
                for (float dy = -3.0; dy <= 3.0; dy += 1.0) {
                    vec2 offset = vec2(dx, dy) / 3.0;
                    float sampleDist = length(offset);
                    if (sampleDist > 1.0) continue;

                    float weight = 1.0 - sampleDist * 0.5;
                    vec2 neighborCoord = clamp(fragUV + offset * uvSampleRadius, vec2(0.0), vec2(1.0));
                    float neighborIndex = texture(fogMapTexture, neighborCoord).r * 255.0;
                    int neighborFogIndex = int(neighborIndex + 0.5) - 1;

                    totalWeight += weight;
                    if (neighborFogIndex == cameraFogIndex) sameZoneWeight += weight;
                }
            }

            float edgeFade = smoothstep(0.0, 0.5, sameZoneWeight / max(totalWeight, 0.001));
            fogDensity *= edgeFade;
        }
    }

    return mix(color, fogColor, fogDensity);
}

void main()
{
    // Calculate water depth in world units
    // Both terrainHeight (from heightmap texture) and waterLevel are already in world units
    float terrainHeight = texture(heightmapTexture, heightmapCoord).r;
    float waterDepthWorld = max(waterLevel - terrainHeight, 0.0);
    // Clamp to reasonable range (several tiles deep max, 1 tile = 16 units)
    waterDepthWorld = clamp(waterDepthWorld, 0.0, 64.0);
    // Normalized depth for legacy effects (0-1 over ~2 tiles)
    float waterDepth = waterDepthWorld / 32.0;

    // Super shallow water: discard only at extreme shallows
    if (waterDepth < 0.0005) {
        discard;
    }

    // Soft shoreline edge - fade alpha based on shallow depth
    // This creates a gradual transition where water meets land
    float shorelineAlpha = smoothstep(0.0005, 0.06, waterDepth);

    // Use radial (spherical) distance from camera for natural circular fog falloff
    float dist = length(worldPos - cameraPos);
    float fogStart = view_distance * 0.50;
    float fogFactor = clamp((dist - fogStart) / (view_distance * 0.95 - fogStart), 0.0, 0.65);

    // Handle danger water (lava)
    if (isDangerWater == 1u) {
        vec2 heatDistortion = vec2(
            sin(time * 2.0 + texCoord0.x * 15.0) * 0.002,
            cos(time * 2.5 + texCoord0.y * 12.0) * 0.002
        );
        vec4 sC = textureLod(basic_texture, fract(texCoord0 + heatDistortion), 0.0);
        vec3 baseColor = vec3(sC.b, sC.g, sC.r);
        baseColor.r = min(baseColor.r * 1.2, 1.0);
        baseColor.b *= 0.9;

        vec3 finalColor = mix(baseColor, skyColor.rgb, fogFactor);
        outputColor = vec4(finalColor, 0.9);
        return;
    }

    // Vectors for lighting calculations
    vec3 baseNormal = normalize(surfaceNormal);
    vec3 viewDir = normalize(viewDirection);
    vec3 lightDir = normalize(toLightVector);

    // === S-SHAPED UV DISTORTION (must be calculated before texture sampling) ===
    // Use world position for consistent ripple scale
    vec2 rippleUV = worldPos.xz * 0.15;

    // Calculate S-curve waves for UV distortion
    float uvWave1 = sin(rippleUV.x + sin(rippleUV.y * 0.7 + time * 1.4) * 1.2 + time * 1.0);
    float uvWave2 = sin(rippleUV.y * 0.9 + sin(rippleUV.x * 0.6 - time * 1.2) * 1.0 - time * 0.9);

    // Create UV offset from the waves - reduced amount for subtler effect
    vec2 uvDistortion = vec2(uvWave1, uvWave2) * 0.004;

    // Reduce distortion at distance for stability
    float uvDistDist = clamp(length(worldPos - cameraPos) / (16.0 * 10.0), 0.0, 1.0);
    uvDistortion *= mix(1.0, 0.1, uvDistDist);

    // Calculate where we are within the texture tile (0-1 range)
    vec2 tilePos = fract(texCoord0);

    // Reduce distortion near tile edges to prevent atlas bleeding
    // Edge margin where we fade out distortion (in UV space)
    float edgeMargin = 0.08;
    float edgeFadeX = smoothstep(0.0, edgeMargin, tilePos.x) * smoothstep(0.0, edgeMargin, 1.0 - tilePos.x);
    float edgeFadeY = smoothstep(0.0, edgeMargin, tilePos.y) * smoothstep(0.0, edgeMargin, 1.0 - tilePos.y);
    float edgeFade = edgeFadeX * edgeFadeY;
    uvDistortion *= edgeFade;

    // Apply distortion to texture coordinates
    vec2 distortedTexCoord = texCoord0 + uvDistortion;

    // Sample water texture with distorted UVs - force mip level 0 to avoid atlas bleeding
    vec4 sC = textureLod(basic_texture, fract(distortedTexCoord), 0.0);
    vec3 waterTexColor = vec3(sC.b, sC.g, sC.r);

    // === ANIMATED RIPPLE NORMAL ===
    // Create multi-layered wave patterns for realistic water surface movement
    // Use world-space coordinates for consistent ripple scale
    vec2 worldUV = heightmapCoord * 200.0;  // Scale for visible ripples

    // Distance factor - reduce ripple intensity at distance for stability
    float rippleDist = clamp(dist / (16.0 * 10.0), 0.0, 1.0);
    float rippleIntensity = mix(1.0, 0.15, rippleDist);

    // Shoreline damping - waves lose energy in shallow water
    // waterDepthWorld is already calculated above (water level - terrain height)
    float shorelineFactor = clamp(waterDepthWorld / 12.0, 0.0, 1.0);  // Full waves at ~1 tile depth
    float shoreDamping = smoothstep(0.0, 0.5, shorelineFactor);  // Smooth transition

    // Layer 1: Large slow waves (ocean swell) - most affected by depth
    float wave1 = sin(worldUV.x * 0.8 + worldUV.y * 0.6 + time * 0.4) * 0.5;
    wave1 += sin(worldUV.x * 0.5 - worldUV.y * 0.8 + time * 0.35) * 0.5;
    wave1 *= shoreDamping;  // Large waves break at shore

    // Layer 2: Medium waves (wind chop) - partially affected
    float wave2 = sin(worldUV.x * 2.5 + time * 0.8) * 0.3;
    wave2 += cos(worldUV.y * 2.2 + time * 0.7) * 0.3;
    wave2 += sin((worldUV.x + worldUV.y) * 1.8 + time * 0.6) * 0.2;
    wave2 *= mix(0.3, 1.0, shoreDamping);  // Some chop remains at shore

    // Layer 3: Small ripples (surface detail) - inverted near shore (resistance effect)
    float wave3 = sin(worldUV.x * 6.0 + worldUV.y * 4.0 + time * 1.5) * 0.15;
    wave3 += cos(worldUV.x * 5.0 - worldUV.y * 7.0 + time * 1.3) * 0.15;
    wave3 += sin(worldUV.x * 8.0 + time * 2.0) * 0.1;
    wave3 += cos(worldUV.y * 9.0 + time * 1.8) * 0.1;

    // Near shore: invert small ripples (water pushing back against land)
    // and add higher frequency shore-specific ripples
    float shoreInversion = mix(-0.6, 1.0, shoreDamping);  // Partial inversion at shore
    wave3 *= shoreInversion;

    // Add shore-specific interference ripples (water reflecting off land)
    float shoreRipple = sin(worldUV.x * 12.0 + time * 2.5) * cos(worldUV.y * 10.0 - time * 2.0);
    shoreRipple *= (1.0 - shoreDamping) * 0.3;  // Only visible near shore
    wave3 += shoreRipple;

    // Combine wave layers with different weights
    float waveX = (wave1 * 0.4 + wave2 * 0.35 + wave3 * 0.25) * rippleIntensity * 0.08;
    float waveZ = (wave1 * 0.35 + wave2 * 0.4 + wave3 * 0.25) * rippleIntensity * 0.08;

    // Perturb the normal based on wave displacement
    vec3 normal = normalize(baseNormal + vec3(waveX, 0.0, waveZ));

    // === FRESNEL EFFECT ===
    // Controls how much reflection vs water color based on viewing angle
    // Toned down to keep focus on water texture with subtle sky shine
    float viewDot = abs(dot(viewDir, normal));
    float fresnel = pow(1.0 - viewDot, 5.0);  // Higher power = reflection only at grazing angles
    fresnel = mix(0.05, 0.45, fresnel);  // Slightly increased for more visible shine

    // === REFLECTION ===
    vec3 reflectionColor;

    // Compute sky-based reflection with rippled normal
    vec3 reflectDir = reflect(-viewDir, normal);

    vec2 skyUV = vec2(
        0.5 + atan(reflectDir.x, reflectDir.z) / (2.0 * 3.14159),
        0.5 - reflectDir.y * 0.5
    );
    vec4 skyReflection = texture(skyTexture, skyUV);
    vec3 skyReflectColor = vec3(skyReflection.b, skyReflection.g, skyReflection.r);

    // Sky-only reflection, toned down to let water texture show through
    reflectionColor = skyReflectColor;

    // Darken near horizon for depth
    float horizonFactor = smoothstep(-0.1, 0.3, reflectDir.y);
    vec3 horizonColor = waterTexColor * 0.4;
    reflectionColor = mix(horizonColor, reflectionColor, horizonFactor);

    // Tint reflection with water color for cohesion
    reflectionColor = mix(reflectionColor, reflectionColor * waterTexColor * 1.5, 0.2);

    // Specular highlights removed for classic Carnivores look

    // === WAVE BRIGHTNESS MODULATION ===
    // Make the water texture itself ripple with light/dark bands
    // Combines all wave layers into a brightness factor
    float waveBrightness = wave1 * 0.3 + wave2 * 0.4 + wave3 * 0.3;
    // Normalize to a subtle brightness range (0.85 to 1.15)
    float textureMod = 1.0 + waveBrightness * 0.15 * rippleIntensity;

    // === S-SHAPED RIPPLE SHADOWS (Classic Carnivores water effect) ===
    // Creates oscillating wavy shadow bands that give the appearance of
    // light refracting through a rippled water surface
    // Reuses uvWave1/uvWave2 calculated earlier for UV distortion

    // Add a third wave layer at different angle for more organic look
    float angle = 0.5;
    vec2 rotatedUV = vec2(
        rippleUV.x * cos(angle) - rippleUV.y * sin(angle),
        rippleUV.x * sin(angle) + rippleUV.y * cos(angle)
    );
    float sWave3 = sin(rotatedUV.x * 0.8 + sin(rotatedUV.y * 0.7 + time * 1.0) * 0.9 + time * 0.7);

    // Combine waves - creates interference pattern of S-curves
    // uvWave1 and uvWave2 were calculated earlier for UV distortion
    float rippleShadow = uvWave1 * 0.5 + uvWave2 * 0.3 + sWave3 * 0.2;

    // Shape into shadow bands - simple sine to shadow conversion
    float shadowBand = rippleShadow * 0.5 + 0.5;  // Convert -1..1 to 0..1

    // Shadow band contrast (0.5 = dark shadow, 1.3 = bright highlight)
    float rippleShadowMod = mix(0.5, 1.3, shadowBand);

    // Reduce effect at distance for stability
    rippleShadowMod = mix(rippleShadowMod, 1.0, rippleDist * 0.5);

    // Reduce in very shallow water (shoreline)
    rippleShadowMod = mix(1.0, rippleShadowMod, shoreDamping);

    // === BASE WATER COLOR ===
    float diffuse = max(dot(normal, lightDir), 0.0);
    float brightness = 0.4 + diffuse * 0.4;
    // Apply both wave brightness modulation and S-shaped ripple shadows
    vec3 baseColor = waterTexColor * brightness * textureMod * rippleShadowMod;

    // Depth-based color tinting
    float depthFactor = clamp(waterDepth / 2.0, 0.0, 1.0);
    vec3 shallowColor = baseColor * 1.3;
    vec3 deepColor = baseColor * 0.6;
    vec3 waterColor = mix(shallowColor, deepColor, depthFactor);

    // === CLOUD SHADOWS ===
    // Sample sky texture to create moving cloud shadows (same as terrain)
    vec4 cloudSample = texture(skyTexture, cloudTexCoord);
    float cloudLuminance = (0.299 * cloudSample.b + 0.587 * cloudSample.g + 0.114 * cloudSample.r);

    // Cloud contrast - bright sky = brighter water, dark clouds = darker water
    float cloudBrightness = clamp((cloudLuminance - 0.2) * 3.0, 0.0, 1.0);
    // Range from 0.4 (under clouds) to 1.05 (clear sky) - water is slightly less affected than terrain
    float cloudModulation = 0.4 + (cloudBrightness * 0.65);

    // Apply cloud shadows to water color
    waterColor *= cloudModulation;

    // === COMBINE REFLECTION AND WATER COLOR ===
    // Reflection should always be visible - use fresnel directly (not reduced)
    vec3 surfaceColor = mix(waterColor, reflectionColor, fresnel);

    // Distance fade for stability
    float distFadeStart = 16.0 * 8.0;
    float distFadeEnd = 16.0 * 15.0;
    float distFade = clamp((dist - distFadeStart) / (distFadeEnd - distFadeStart), 0.0, 1.0);

    vec3 distantColor = mix(waterTexColor * 0.6, reflectionColor, 0.5);
    surfaceColor = mix(surfaceColor, distantColor, distFade * 0.5);

    // Apply distance fog first
    vec3 finalColor = mix(surfaceColor, skyColor.rgb, fogFactor);

    // Apply area fog (localized fog zones) AFTER distance fog
    // Skip area fog at night - night depth fog handles visibility instead
    if (timeOfDay != 2) {
        // This ensures area fog is visible and not washed out by distance fog
        // Use world-space distance for proper Transp-based falloff calculation
        float worldDistance = length(worldPos - cameraPos);
        finalColor = applyAreaFog(finalColor, worldDistance, worldPos);
    }

    // === TRANSPARENCY FOR UNDERWATER VISIBILITY ===
    // Account for viewing angle - looking at shallow angle means light travels
    // through more water to reach the same depth
    // viewDot = 1.0 when looking straight down, approaches 0 at grazing angles
    // Optical depth = waterDepthWorld / viewDot (clamped to avoid division issues)
    float opticalDepth = waterDepthWorld / max(viewDot, 0.15);

    // Normalize to 2 tile depth (32 world units) - extended transparency range for deeper water
    float opticalDepthNorm = clamp(opticalDepth / 32.0, 0.0, 1.0);

    // Transparent water - can see objects through deeper areas
    // Start at 0.2 alpha (quite transparent) and go to 0.95 (nearly opaque at depth)
    float underwaterAlpha = mix(0.2, 0.95, opticalDepthNorm);

    // Deep water (beyond transparency range) should be fully opaque
    // If actual water depth exceeds 48 units (3 tiles), force full opacity
    if (waterDepthWorld > 48.0) {
        underwaterAlpha = 1.0;
    }

    // Fresnel adds significant opacity at grazing angles for shiny water surface
    float finalAlpha = mix(underwaterAlpha, min(underwaterAlpha + 0.35, 0.98), fresnel);

    // Distance fade - water becomes more opaque at distance (can't see underwater details far away)
    finalAlpha = mix(finalAlpha, 0.95, distFade * 0.6);

    // Apply shoreline soft edge - fade out at shallow edges
    finalAlpha *= shorelineAlpha;

    // ========== TIME-OF-DAY EFFECTS (DAWN/DUSK/NIGHT) ==========
    // Uses explicit timeOfDay uniform: 0=DAY, 1=DUSK, 2=NIGHT
    // Dusk/dawn gets warm orange/amber tints, night gets cool blue moonlight
    if (timeOfDay > 0) {
        float effectIntensity = 1.0 - visibilityMultiplier;

        bool isDusk = (timeOfDay == 1);
        bool isNight = (timeOfDay == 2);

        // 1. Darkening - dusk is brighter than night
        float darkening = 1.0;
        if (isDusk) {
            darkening = 1.0 - (effectIntensity * 0.45);
        } else if (isNight) {
            darkening = 1.0 - (effectIntensity * 0.90);
        }
        finalColor *= darkening;

        // 2. Desaturation - dusk keeps more color
        vec3 grayscale = vec3(dot(finalColor, vec3(0.299, 0.587, 0.114)));
        float desatAmount = 0.0;
        if (isDusk) {
            desatAmount = effectIntensity * 0.25;
        } else if (isNight) {
            desatAmount = effectIntensity * 0.85;
        }
        finalColor = mix(finalColor, grayscale, desatAmount);

        // 3. Color tinting - warm amber for dusk, cool blue for night
        if (isDusk) {
            vec3 duskTint = vec3(1.25, 0.80, 0.50);  // Warm amber/orange for sunset water
            finalColor = mix(finalColor, finalColor * duskTint, effectIntensity * 0.65);
        } else if (isNight) {
            vec3 nightTint = vec3(0.4, 0.5, 0.8);    // Cool blue moonlight
            finalColor = mix(finalColor, finalColor * nightTint, effectIntensity * 0.6);
        }

        // 4. Depth fog - dusk has longer visibility than night
        if (isDusk) {
            float duskFogStart = 200.0;
            float duskFogEnd = 800.0;
            float duskFogAmount = smoothstep(duskFogStart, duskFogEnd, dist) * effectIntensity;
            finalColor = mix(finalColor, DUSK_FOG_COLOR, duskFogAmount * 0.8);
        } else if (isNight) {
            float nightFogStart = 150.0;
            float nightFogEnd = 480.0;
            float nightFogAmount = smoothstep(nightFogStart, nightFogEnd, dist) * effectIntensity;
            finalColor = mix(finalColor, NIGHT_FOG_COLOR, nightFogAmount);
        }
    }

    // ========== HARD EDGE FADE ==========
    // Ensure geometry fully fades to sky color before reaching the clip plane
    // Uses radial distance for circular fade (dist already calculated above)
    float fadeStart = view_distance * 0.70;  // Start hard fade at 70% of view distance
    float fadeEnd = view_distance * 0.95;    // Fully faded by 95% of view distance
    float hardFade = clamp((dist - fadeStart) / (fadeEnd - fadeStart), 0.0, 1.0);
    // Smooth the fade curve for more gradual transition
    hardFade = smoothstep(0.0, 1.0, hardFade);
    // At dusk/night, fade to appropriate color instead of daytime sky color
    vec3 fadeTargetColor = skyColor.rgb;
    if (timeOfDay == 1) {
        // Dusk - blend toward warm amber fog
        float fadeIntensity = 1.0 - visibilityMultiplier;
        fadeTargetColor = mix(skyColor.rgb, DUSK_FOG_COLOR, pow(fadeIntensity, 0.5));
    } else if (timeOfDay == 2) {
        // Night - blend toward dark blue fog
        float fadeIntensity = 1.0 - visibilityMultiplier;
        fadeTargetColor = mix(skyColor.rgb, NIGHT_FOG_COLOR, pow(fadeIntensity, 0.5));
    }
    finalColor = mix(finalColor, fadeTargetColor, hardFade);

    outputColor = vec4(finalColor, finalAlpha);
}
