# Carnivores Dinosaur AI Analysis

This document provides a comprehensive analysis of the dinosaur AI systems in the Carnivores game, with a focus on the Velociraptor and comparisons to other dinosaurs.

## Table of Contents

1. [Velociraptor AI Deep Dive](#velociraptor-ai-deep-dive)
2. [AI Comparison Across All Dinosaurs](#ai-comparison-across-all-dinosaurs)
3. [Key Files and Functions](#key-files-and-functions)

---

## Velociraptor AI Deep Dive

The Velociraptor AI is implemented in `Characters.cpp:1252-1498` via the `AnimateVelo()` function. It's a sophisticated state-machine-based AI with smooth physics interpolation.

### Data Structure

From `Hunt.h:242-260`:

```cpp
typedef struct _TCharacter {
  int CType;              // Creature type ID (Velo = 9)
  int StateF;             // State flags (e.g., csONWATER)
  int State;              // Primary behavior state (0, 1, 2, 3)
  int NoWayCnt, NoFindCnt, AfraidTime, tgtime;
  int Phase, FTime;       // Animation phase and frame time
  float vspeed, rspeed;   // Velocity and rotation speed
  float bend;             // Body lean during turns
  float tgx, tgz;         // Target position
  float tgalpha, alpha;   // Target and current facing angles
  float beta, gamma;      // Pitch and roll
  int Slide;              // Slide momentum when turning sharply
  float slidex, slidez;   // Slide direction
} TCharacter;
```

### Animation Phases

From `Characters.cpp:30-37`:

| Phase | Value | Description |
|-------|-------|-------------|
| `VEL_RUN` | 0 | Running animation |
| `VEL_WALK` | 1 | Walking animation |
| `VEL_SLIDE` | 2 | Sliding (momentum during sharp turns) |
| `VEL_SWIM` | 3 | Swimming in water |
| `VEL_JUMP` | 4 | Pounce/jump attack |
| `VEL_DIE` | 5 | Death animation |
| `VEL_EAT` | 6 | Eating/killing player |
| `VEL_SLP` | 7 | Tranquilized/sleeping |

### State Machine

#### Primary States

| State | Behavior |
|-------|----------|
| **0** | Passive/wandering - roams randomly to waypoints |
| **1** | Active/hunting - chasing or fleeing from player |
| **2** | Detection trigger - immediately transitions to State 1 |

#### State Transitions

From `Characters.cpp:1271-1321`:

```
State 2 → State 1:  Immediately on detection (line 1271)
State 1 → State 0:  When AfraidTime expires (lines 1289-1292)
Any → State 0:      When player dies (line 1281)
```

The `AfraidTime` counter controls how long the velo stays aggressive. It's set during detection in `CheckAfraid()` (`Characters.cpp:3090-3184`):

```cpp
cptr->AfraidTime = (int)(1.0 / (kRes+0.1) * 10.f * 1000.f);  // Line 3177
```

### Pathfinding

#### Target Selection

From `Characters.cpp:498-516`:

```cpp
void SetNewTargetPlace(TCharacter *cptr, float R)
{
    // Pick random point within radius R
    p.x = cptr->pos.x + siRand((int)R);
    p.z = cptr->pos.z + siRand((int)R);

    // Validate position is walkable
    if (CheckPlaceCollisionP(p)) goto replace;

    cptr->tgx = p.x;
    cptr->tgz = p.z;
}
```

#### Behavior by State

From `Characters.cpp:1282-1321`:

**State 0 (Passive):**
- Wanders to random points within 8048 units
- Picks new target when within 456 units of current target

**State 1 (Active):**
```cpp
if (pdist > (18+OptAgres*10)*256) {
    // Too far - run AWAY from player (flee behavior)
    cptr->tgx = cptr->pos.x - nv.x;  // Opposite direction
    cptr->tgz = cptr->pos.z - nv.z;
} else {
    // Close enough - chase player directly
    cptr->tgx = PlayerX;
    cptr->tgz = PlayerZ;
}
```

The aggression threshold depends on `OptAgres` (difficulty setting):
- OptAgres=0: Chase within ~4608 units
- OptAgres=1: Chase within ~7168 units
- OptAgres=2: Chase within ~9728 units

#### Sinusoidal Wandering

From `Characters.cpp:1328-1332`:

When chasing at medium distance, adds sine-wave variation for realistic movement:
```cpp
if (cptr->State && pdist>1648) {
    cptr->tgalpha += (float)sin(RealTime/824.f) / 2.f;
}
```

#### Obstacle Avoidance

`LookForAWay()` (`Characters.cpp:363-395`) samples paths at ±15°, ±30°, ±45°... up to ±180°:

```cpp
void LookForAWay(TCharacter *cptr, BOOL wc, BOOL mc)
{
    float dalpha = 15.f;
    for (int i=0; i<12; i++) {
        // Check path at +dalpha degrees
        cptr->tgalpha = alpha + dalpha*pi/180.f;
        curp = CheckPossiblePath(cptr, wc, mc);
        if (!curp) return;  // Clear path found

        // Check path at -dalpha degrees
        cptr->tgalpha = alpha - dalpha*pi/180.f;
        curp = CheckPossiblePath(cptr, wc, mc);
        if (!curp) return;

        dalpha += 15.f;  // Widen search
    }
    cptr->tgalpha = afound;  // Use best path found
}
```

#### Path Checking

`CheckPossiblePath()` (`Characters.cpp:348-360`) raycasts 20 steps forward (64 units each = 1280 total):

```cpp
int CheckPossiblePath(TCharacter *cptr, BOOL wc, BOOL mc)
{
    for (int t=0; t<20; t++) {
        p.x += lookx * 64.f;
        p.z += lookz * 64.f;
        if (CheckPlaceCollision(p, wc, mc)) c++;  // Count obstacles
    }
    return c;
}
```

#### Collision Detection

`CheckPlaceCollision()` (`Characters.cpp:272-313`) checks for:
- **Map boundaries**: Grid 4-508 (out of 512)
- **Water**: `fmWater` flag (0x80)
- **Impassable terrain**: `fmNOWAY` flag (0x20)
- **Steep terrain**: Height difference > 124 units
- **Map objects**: Objects with collision radius > 10

### Attack Sequences

#### Jump Attack

From `Characters.cpp:1299-1302`:

Triggers when:
- Not in water (`!(StateF & csONWATER)`)
- Distance: 900-1324 units (scaled by dino size)
- Facing player within 0.2 radians (~11.5°)

```cpp
if (!(cptr->StateF & csONWATER))
  if (pdist<1324 * cptr->scale && pdist>900 * cptr->scale)
    if (AngleDifference(cptr->alpha, FindVectorAlpha(playerdx, playerdz)) < 0.2f)
      cptr->Phase = VEL_JUMP;
```

#### Kill Attack

From `Characters.cpp:1304-1313`:

Triggers when:
- Distance < 256 units
- Height difference < 256 units
- Not in water

```cpp
if (pdist<256)
  if (fabs(PlayerY - cptr->pos.y - 120) < 256) {
    if (!(cptr->StateF & csONWATER)) {
      cptr->vspeed/= 8.0f;  // Slow down dramatically
      cptr->Phase = VEL_EAT;
    }
    AddDeadBody(cptr, HUNT_EAT);  // Instant kill
  }
```

The attack is an **instant kill** - no damage system, just death via `AddDeadBody()`.

### Movement Physics

#### Rotation System

From `Characters.cpp:1392-1427`:

**Target angle calculation:**
```cpp
float dalpha = fabs(cptr->tgalpha - cptr->alpha);
float drspd = dalpha;
if (drspd > pi) drspd = 2*pi - drspd;  // Shortest rotation path
```

**Rotation speed:**
```cpp
if (drspd > 0.02)
    if (cptr->tgalpha > cptr->alpha)
        currspeed = 0.6f + drspd*1.2f;  // Base + proportional
    else
        currspeed = -0.6f - drspd*1.2f;

if (cptr->AfraidTime) currspeed *= 2.5;  // Faster when hunting
if ((StateF & csONWATER) || Phase==VEL_WALK) currspeed /= 1.4f;  // Slower
```

**Smooth interpolation using `DeltaFunc()` (`mathematics.cpp:26-33`):**
```cpp
void DeltaFunc(float &a, float b, float d)
{
    if (b > a) { a += d; if (a > b) a = b; }
    else       { a -= d; if (a < b) a = b; }
}
```

This moves `a` toward `b` by at most `d` per frame - a simple linear interpolation clamp.

**Body lean (bend):**
```cpp
tgbend = drspd/3;                      // Lean proportional to turn rate
if (tgbend > pi/5) tgbend = pi/5;      // Max ~36°
DeltaFunc(cptr->bend, tgbend, TimeDt / 600-800.f);
```

#### Movement Speed

From `Characters.cpp:1441-1467`:

**Speed by phase:**

| Phase | Speed |
|-------|-------|
| VEL_RUN | 1.2 |
| VEL_JUMP | 1.1 |
| VEL_WALK | 0.428 |
| VEL_SWIM | 0.4 |
| VEL_EAT | 0.0 |

**Speed reduction when turning sharply:**
```cpp
if (drspd > pi/2.f) curspeed *= 2.f - 2.f*drspd / pi;  // 0-100% at 90-180°
```

**Movement application:**
```cpp
DeltaFunc(cptr->vspeed, curspeed, TimeDt / 500.f);  // Smooth acceleration
MoveCharacter(cptr,
    cptr->lookx * cptr->vspeed * TimeDt * cptr->scale,
    cptr->lookz * cptr->vspeed * TimeDt * cptr->scale,
    FALSE, TRUE);
```

#### Slide Mechanic

From `Characters.cpp:1430-1476`:

When turning sharply at high speed, momentum is preserved:

```cpp
if (!cptr->Slide && cptr->vspeed>0.6 && Phase!=VEL_JUMP)
  if (AngleDifference(cptr->tgalpha, cptr->alpha) > pi/2.f) {
    cptr->Slide = (int)(cptr->vspeed * 700.f);  // Store momentum
    cptr->slidex = cptr->lookx;                  // Store direction
    cptr->slidez = cptr->lookz;
    cptr->vspeed = 0;                            // Zero forward speed
  }

// Apply slide each frame
if (cptr->Slide) {
    MoveCharacter(cptr,
        cptr->slidex * cptr->Slide / 600.f * TimeDt * cptr->scale,
        cptr->slidez * cptr->Slide / 600.f * TimeDt * cptr->scale, ...);
    cptr->Slide -= TimeDt;  // Decay
}
```

#### Pitch and Roll

From `Characters.cpp:1479-1493`:

**Terrain adaptation via `ThinkY_Beta_Gamma()`:**
- Samples terrain 48 units ahead/behind for **pitch (beta)**
- Samples terrain 24 units left/right for **roll (gamma)**
- Max angles: ±0.5 pitch, ±0.4 roll

**Roll from rotation:**
```cpp
if (Phase==VEL_WALK) cptr->tggamma += cptr->rspeed / 7.0f;
else                 cptr->tggamma += cptr->rspeed / 5.0f;
DeltaFunc(cptr->gamma, cptr->tggamma, TimeDt / 1624.f);
```

#### Water Handling

From `Characters.cpp:1274-1276` and `1480-1483`:

```cpp
// Detect water depth
if (GetLandUpH(x,z) - GetLandH(x,z) > 140 * scale)
    StateF |= csONWATER;

// In water behavior
if (StateF & csONWATER) {
    pos.y = GetLandUpH(x,z) - 160 * scale;  // Float at surface
    beta /= 2;     // Reduce pitch
    tggamma = 0;   // Zero roll
}
```

### Detection System

The `CheckAfraid()` function (`Characters.cpp:3090-3184`) calculates detection based on:

1. **Look detection** (`kALook`): Based on line-of-sight, player stance, camouflage
2. **Smell detection** (`kASmell`): Based on wind direction, scent cover

Both are divided by species-specific sensitivity coefficients (`LookK`, `SmellK`).

---

## AI Comparison Across All Dinosaurs

### Behavioral Categories

The dinosaurs fall into **four distinct AI archetypes**:

| Category | Dinosaurs | Behavior |
|----------|-----------|----------|
| **Aggressive Predators** | Velociraptor, Raptor | Hunt player, pounce attacks, slide mechanics |
| **Apex Predator** | T-Rex | Hunts player, special detection animations, no sliding |
| **Defensive/Charging** | Triceratops | Flees at distance, charges when close |
| **Pure Herbivores** | Moschops, Gallimimus, Pachycephalosaurus, Stegosaurus, Parasaurolophus | Always flee from player |
| **Ambient** | Dimorphodon | Flying creature, no player interaction |

### State Machine Comparison

| Dinosaur | States Used | AfraidTime Decay |
|----------|-------------|------------------|
| **Velociraptor/Raptor** | 0, 1, 2 | Decrements only when fleeing (pdist > threshold) |
| **T-Rex** | 0, 1, 2, 3, 5 | Never decrements (always hunts once triggered) |
| **Triceratops** | 0, 1, 2 | Refreshed when player < 6000 units, decrements otherwise |
| **Moschops** | 0, 1, 2 | Refreshed when player < 2048, decrements when far |
| **Dimorphodon** | 0 only | No states - ambient flying only |

**Key differences:**

- **T-Rex** uses **State 3** for smell-based detection (vs sight) and **State 5** as a special "just detected" state
- **T-Rex** has special **LookMode** animations (`REX_SEE`, `REX_SMEL`, `REX_SCREAM`) before chasing
- Herbivores have simpler 2-state machines (passive/fleeing)

### Chase vs Flee Behavior

```
                  ┌─────────────────────────────────────────┐
                  │           DISTANCE FROM PLAYER          │
                  ├──────────┬──────────┬──────────┬────────┤
                  │  < 300   │  < thresh│  > thresh│ > 13240│
┌─────────────────┼──────────┼──────────┼──────────┼────────┤
│ Velociraptor    │  ATTACK  │  CHASE   │   FLEE   │  ---   │
│ Raptor          │  ATTACK  │  CHASE   │   FLEE   │  ---   │
│ T-Rex           │  ATTACK  │  CHASE   │  CHASE   │  ---   │
│ Triceratops     │  ATTACK  │  CHARGE  │   FLEE   │  ---   │
│ Moschops        │   ---    │   FLEE   │   FLEE   │ RESPAWN│
│ Pachyceph.      │   ---    │   FLEE   │   FLEE   │  ---   │
│ Stegosaurus     │   ---    │   FLEE   │   FLEE   │  ---   │
│ Parasaur.       │   ---    │   FLEE   │   FLEE   │  ---   │
│ Gallimimus      │   ---    │   FLEE   │   FLEE   │ RESPAWN│
│ Dimorphodon     │   ---    │   ---    │   ---    │  ---   │
└─────────────────┴──────────┴──────────┴──────────┴────────┘
```

**Chase/Flee Threshold Formula:**

| Dinosaur | Threshold (units) | Formula |
|----------|-------------------|---------|
| Velociraptor | 4608-9728 | `(18 + OptAgres*10) * 256` |
| Raptor | 4096-7680 | `(16 + OptAgres*6) * 256` |
| T-Rex | ∞ (always chases) | No threshold check |
| Triceratops | 2560-6400 | `(10 + OptAgres*6) * 256` |

### Attack Mechanics

| Dinosaur | Attack Type | Range | Special Requirements |
|----------|-------------|-------|----------------------|
| **Velociraptor** | Jump + Kill | Jump: 900-1324, Kill: <256 | Facing < 0.2 rad, not in water |
| **Raptor** | Jump + Kill | Jump: 900-1324, Kill: <256 | Facing < 0.2 rad, not in water |
| **T-Rex** | Kill only | <256 | Height diff < 256 |
| **Triceratops** | Trample | <300 | Height diff < 256 |
| **Others** | None | N/A | N/A |

**T-Rex unique behavior:** No jump attack, but has detection sequence animations before chasing:
```cpp
if (cptr->State==2) cptr->Phase = REX_SEE1 + rRand(1);   // Sight detection
               else cptr->Phase = REX_SMEL + rRand(1);   // Smell detection
// Then after animation: REX_SCREAM before running
```

### Movement Speed Comparison

| Dinosaur | Run Speed | Walk Speed | Special |
|----------|-----------|------------|---------|
| **Velociraptor** | 1.2 | 0.428 | Slide: preserves momentum |
| **Raptor** | 1.2 | 0.428 | Slide: preserves momentum |
| **T-Rex** | 2.49 | 0.76 | No slide, largest creature |
| **Triceratops** | 1.2 | 0.30 | Slow acceleration up, fast decel down |
| **Moschops** | 0.6 | 0.3 | Small, slow creature |
| **Gallimimus** | 0.6 | 0.3 | Same as Moschops |
| **Dimorphodon** | 1.5/1.3 | N/A | Flying only (FLY/FLYP phases) |

### Special Mechanics

#### Slide System (Raptors only)

Only **Velociraptor** and **Raptor** have the sliding mechanic:

```cpp
// Characters.cpp:1172-1179 (Raptor), 1431-1437 (Velo)
if (!cptr->Slide && cptr->vspeed>0.6 && Phase!=RAP_JUMP)
  if (AngleDifference(cptr->tgalpha, cptr->alpha) > pi/2.f) {
    cptr->Slide = (int)(cptr->vspeed*700.f);  // Store momentum
    cptr->slidex = cptr->lookx;
    cptr->slidez = cptr->lookz;
    cptr->vspeed = 0;
  }
```

#### T-Rex Detection Animations

```cpp
// Characters.cpp:1602-1631
LookMode = (Phase == REX_SEE || REX_SEE1 || REX_SMEL || REX_SMEL1);
// When LookMode: rotation is disabled, dino faces player direction
// After LookMode ends: REX_SCREAM plays before chase begins
```

#### Dimorphodon Flight

```cpp
// Characters.cpp:2999-3005
if (Phase == DIM_FLY)
  DeltaFunc(pos.y, GetLandH(x,z) + 4048, TimeDt/6.f);  // Rise to cruising altitude
else
  DeltaFunc(pos.y, GetLandH(x,z), TimeDt/16.f);        // Dive down

// Switches between FLY (up) and FLYP (down) based on altitude
if (pos.y > terrain + 2800) Phase = DIM_FLYP;  // Start diving
if (pos.y < terrain + 1800) Phase = DIM_FLY;   // Start climbing
```

#### Respawn Mechanic (Small herbivores)

```cpp
// Characters.cpp:1798-1799 (Moschops), 2898-2899 (Dimorphodon)
if (pdist > 13240)
  if (ReplaceCharacterForward(cptr)) goto TBEGIN;
```

When too far from player, teleports ahead to stay relevant.

### Idle Behavior (Herbivores only)

Herbivores have special idle animations when passive:

| Dinosaur | Idle Chance | Idle Animations |
|----------|-------------|-----------------|
| Moschops | ~6% | MOS_IDLE1, MOS_IDLE2 |
| Triceratops | ~3% | TRI_IDLE1, TRI_IDLE2, TRI_IDLE3 |
| Pachyceph. | ~6% | PAC_IDLE1 |
| Stegosaurus | ~6% | STG_IDLE1 |
| Parasaur. | ~6% | PAR_IDLE1 |
| Gallimimus | ~6% | GAL_IDLE1 |

Raptors and T-Rex never idle - they're always walking or hunting.

### Rotation/Physics Parameters

| Dinosaur | Rotation Accel | Bend Divisor | Terrain Sampling |
|----------|----------------|--------------|------------------|
| Velociraptor | 160ms / 180ms | /3 | 48 forward, 24 side |
| Raptor | 160ms / 180ms | /2 | 48 forward, 24 side |
| T-Rex | 440ms / 620ms | /2 | 348 forward, 324 side |
| Triceratops | 400ms | /3.5 | 128 forward, 64 side |
| Moschops | 260ms | /2 | 64 forward, 32 side |

The T-Rex turns much more slowly (larger values = slower acceleration) but samples terrain much further ahead due to its size.

### Summary: Key Differences from Velociraptor

| Feature | Velociraptor | Other Predators | Herbivores |
|---------|--------------|-----------------|------------|
| **Jump attack** | Yes | T-Rex: No, Raptor: Yes | No |
| **Slide mechanic** | Yes | T-Rex: No, Raptor: Yes | No |
| **Flee behavior** | At distance | T-Rex: Never | Always |
| **Detection anims** | No | T-Rex: Yes | No |
| **Idle animations** | No | T-Rex: No | Yes |
| **Respawn** | No | No | Some |
| **AfraidTime decay** | When fleeing | T-Rex: Never | When far |

---

## Key Files and Functions

### Primary AI Implementation

| File | Lines | Function | Description |
|------|-------|----------|-------------|
| `Characters.cpp` | 995-1240 | `AnimateRaptor()` | Raptor AI |
| `Characters.cpp` | 1252-1498 | `AnimateVelo()` | Velociraptor AI |
| `Characters.cpp` | 1508-1740 | `AnimateTRex()` | T-Rex AI |
| `Characters.cpp` | 1755-1931 | `AnimateMosh()` | Moschops AI |
| `Characters.cpp` | 1943-2133 | `AnimateTric()` | Triceratops AI |
| `Characters.cpp` | 2140-2318 | `AnimatePac()` | Pachycephalosaurus AI |
| `Characters.cpp` | 2324-2502 | `AnimateSteg()` | Stegosaurus AI |
| `Characters.cpp` | 2508-2686 | `AnimatePar()` | Parasaurolophus AI |
| `Characters.cpp` | 2695-2868 | `AnimateGall()` | Gallimimus AI |
| `Characters.cpp` | 2875-3019 | `AnimateDimor()` | Dimorphodon AI |

### Helper Functions

| File | Lines | Function | Description |
|------|-------|----------|-------------|
| `Characters.cpp` | 230-267 | `CheckPlaceCollisionP()` | Collision for placement |
| `Characters.cpp` | 272-313 | `CheckPlaceCollision()` | Collision for movement |
| `Characters.cpp` | 348-360 | `CheckPossiblePath()` | Raycast path checking |
| `Characters.cpp` | 363-395 | `LookForAWay()` | Obstacle avoidance |
| `Characters.cpp` | 454-484 | `MoveCharacter()` | Apply movement with collision |
| `Characters.cpp` | 498-516 | `SetNewTargetPlace()` | Random waypoint selection |
| `Characters.cpp` | 3090-3184 | `CheckAfraid()` | Player detection system |
| `mathematics.cpp` | 26-33 | `DeltaFunc()` | Linear interpolation clamp |

### Key Constants

From `Hunt.h`:

| Constant | Value | Description |
|----------|-------|-------------|
| `fmWater` | 0x80 | Water terrain flag |
| `fmNOWAY` | 0x20 | Impassable terrain flag |
| `csONWATER` | 0x00010000 | Character in water state |
| `HUNT_EAT` | 0 | Death by eating |
| `HUNT_KILL` | 3 | Death by T-Rex |

---

*This AI system is sophisticated for a 1998 game, featuring smooth interpolation, realistic physics, multiple behavior states, and effective pathfinding.*
