# Extended ClassicAIController Plan

This document outlines the plan to extend `CEClassicAIController` to support all dinosaur behavior archetypes from the original Carnivores 2 source code.

## Overview

The current `CEClassicAIController` faithfully recreates the Velociraptor AI (`AnimateVelo`). This plan extends it to support all five behavioral archetypes found in the original game.

---

## Implementation Progress

### Current Status: Phase 2 COMPLETE ✓

**Completed:**
- [x] **Phase 1: Core Infrastructure + PURE_HERBIVORE**
  - Added `BehaviorType` enum with all 5 archetypes
  - Extended `ClassicAnimPhase` enum with new phases (IDLE, DETECT_*, ROAR, FLY_*, FALL)
  - Refactored `AnimateCharacter()` with behavioral switch dispatch
  - Extracted Velociraptor logic into `AnimatePredator()`
  - Implemented `AnimateHerbivore()` with proper state machine
  - Added `m_NewPhase` detection on animation cycle completion (critical for phase selection)
  - Extended `loadConfig()` with new parameters
  - Updated `getPhaseAnimation()` with PHASE_IDLE support
  - Added `selectRandomIdleAnim()` helper

- [x] **Herbivore Config Migration**
  - Converted 6 herbivores in `config.json` from GenericAmbient to ClassicAI:
    - Stegosaurus: walkSpeed=1.2, runSpeed=3.2, panicTriggerDist=64, zigzag enabled
    - Moschops: walkSpeed=1.0, runSpeed=2.0, panicTriggerDist=50.75
    - Gallimimus: walkSpeed=1.1, runSpeed=3.0, panicTriggerDist=50.75
    - Dimetrodon: walkSpeed=1.0, runSpeed=2.0, panicTriggerDist=50.75
    - Pachycephalosaurus: walkSpeed=1.3, runSpeed=5.3, panicTriggerDist=64, zigzag enabled
    - Parasaurolophus: walkSpeed=1.3, runSpeed=5.3, panicTriggerDist=64, zigzag enabled
  - **Speed Scaling Applied**: ~3.3x multiplier applied to match raptor scaling (raptor: walk=1.2, run=2.8)

- [x] **Phase 1 Bug Fixes**
  - Fixed animation thrashing: phase selection now only occurs when `m_NewPhase` is true (animation cycle completed)
  - Added proper first-frame initialization for valid phase state
  - **Fixed animation freezing across AI instances**: Changed `static double lastAnimUpdate` to per-instance member `m_lastAnimUpdateTime` - static variables cause shared state bugs when multiple AI instances exist
  - **Fixed herbivore escape mode oscillation**: `AnimateHerbivore()` now respects `m_isEscaping` flag and uses committed `m_escapeAlpha` direction instead of recalculating every frame, preventing direction flopping at obstacles (water edges, terrain)

- [x] **Phase 2: APEX_PREDATOR** (T-Rex)
  - Implemented `AnimateApexPredator()` with full detection sequence
  - **Detection States**: State 2 (sight detected), State 3 (smell detected), State 5 (damage abort)
  - **Detection Sequence**: Must face player within `detectFaceAngle` -> play detection anim -> ROAR/SCREAM -> chase
  - **Passive Lookout**: ~14% chance per animation cycle to play detection anims while passive (not alerted)
  - **Never Flees**: No afraid time decay, always chases once active
  - **Head Offset**: Player distance calculated from head position (original: 108 units scaled)
  - **Zigzag Only Far**: Only zigzags when chasing at distance > `apexZigzagMinDist` (5648 original units)
  - **RUN Requires Facing**: Only uses RUN animation when facing player within `runFaceAngle` (1.0 radians)
  - **LookMode**: Rotation disabled during detection animations (DETECT_SIGHT, DETECT_SMELL)
  - **Water Threshold**: Different from other creatures (560 vs normal)
  - Updated T-Rex config from GenericAmbient to ClassicAI with APEX_PREDATOR behavior
  - T-Rex animations: Tx_walk, Tx_run1, Tx_swim, Tx_see1/Tx_see2, Tx_sml1/Tx_sml2, Tx_scr1, Tx_kill, Tx_die1
  - **Kill Distance Sync**: Added `getKillDistance()` method for main.cpp player death sync
  - **Hit Zone Debug**: Added B-key toggle for wireframe sphere visualization of AI hit zones
  - **Height Diff Debug**: Added height difference to player in debug window

**Next Steps:**
- [ ] **Phase 3: FLYING_AMBIENT** (Dimorphodon, Pteranodon) - Altitude oscillation, no ground collision
- [ ] **Phase 4: DEFENSIVE_CHARGER** (Triceratops) - Distance-based flee/charge behavior

---

## Lessons Learned from Phase 1 Implementation

### Bug Patterns to Watch For

#### 1. Static Variable Bugs in C++
**Problem**: The original code used `static` variables for time tracking that were copy-pasted during porting. This caused shared state across ALL AI instances.

**Example**:
```cpp
// BAD - shared across all instances
static double lastAnimUpdate = 0;

// GOOD - per-instance member variable
double m_lastAnimUpdateTime = 0.0;  // Declared in .hpp
```

**Symptom**: Animation would freeze for some entities while they continued to move (gliding across terrain without animation).

**Fix**: Convert all per-instance timing/state variables to member variables. Only use `static` for intentionally shared state like debug rate limiting.

#### 2. Escape Mode Directional Commitment
**Problem**: `AnimateHerbivore()` recalculated target direction every frame, overwriting the committed escape direction set by `MoveCharacter()` when escaping obstacles.

**Symptom**: AI would oscillate/flop direction when stuck at water edges or terrain obstacles.

**Fix**: Check `m_isEscaping` flag before recalculating target direction:
```cpp
// In AnimateHerbivore()
if (m_isEscaping) {
    // Keep the committed escape direction, don't recalculate
    m_tgalpha = m_escapeAlpha;
} else {
    // Normal target direction calculation
    m_tgalpha = FindVectorAlpha(targetdx, targetdz);
}
```

#### 3. Time Unit Mismatches
**Problem**: Comparing `current_time` (seconds) with config values (milliseconds) causes timing to be 1000x too fast.

**Example** (from CEAIPlayerRelativeSpawner):
```cpp
// BAD - mixed units
if (current_time - m_last_cull_time >= m_cull_period_ms)  // seconds vs milliseconds

// GOOD - convert to same units
double cull_period_seconds = m_cull_period_ms / 1000.0;
if (current_time - m_last_cull_time >= cull_period_seconds)
```

### Player-Relative Spawner ClassicAI Compatibility

The `CEAIPlayerRelativeSpawner` was designed for `GenericAmbient` controllers that return a `shared_ptr<CERemotePlayerController>`. ClassicAI controllers manage their own geometry and return `nullptr` from spawn calls.

**Required fixes for ClassicAI support**:

1. **Entity Counting**: `getActiveCount()` must count ClassicAI entities by iterating `m_classic_ai_controllers` and matching spawn name prefixes:
```cpp
// Count ClassicAI entities that belong to this spawner
if (m_classic_ai_controllers) {
    for (const auto& ai : *m_classic_ai_controllers) {
        if (ai && !ai->isDead()) {
            const std::string& spawnName = ai->getSpawnName();
            if (!m_spawn_name_base.empty() && spawnName.find(m_spawn_name_base) == 0) {
                count++;
            }
        }
    }
}
```

2. **Spawn Tracking**: Track spawns via `m_spawn_counter` delta instead of return value:
```cpp
int countBefore = m_spawn_counter;
spawnEntity(...);  // May return nullptr for ClassicAI
int spawned = m_spawn_counter - countBefore;
```

3. **Spawn Name Prefix**: Each spawner should have a unique `m_spawn_name_base` (e.g., "dino_herbivore_") to identify its entities.

---

## Lessons Learned from Phase 2 Implementation (APEX_PREDATOR)

### Bug Patterns to Watch For

#### 4. Animation Timing Reset During Loops
**Problem**: Animation timing was being reset on every frame while in PHASE_EAT, causing frame jumping/stuttering during looped eat animations.

**Root Cause**: Two issues combined:
1. The kill block (entering PHASE_EAT) was executing every frame while player was within kill distance, resetting `m_animStartTime = currentTime` repeatedly
2. `setAnimation()` was resetting timing even when the same animation was already playing

**Symptoms**: Visual frame jumping/stuttering when eat animation loops.

**Fix 1**: Guard the kill trigger to only execute once:
```cpp
// Only trigger kill sequence once (when first entering PHASE_EAT)
if (pdist < killDist) {
    if (m_Phase != PHASE_EAT) {  // <-- Guard prevents repeated execution
        m_Phase = PHASE_EAT;
        m_animStartTime = currentTime;
        m_FTime = 0.0f;
        m_currentAnim = "";  // Force setAnimation to apply
        // ... rest of kill logic
    }
}
```

**Fix 2**: In `setAnimation()`, don't reset timing for same animation:
```cpp
void CEClassicAIController::setAnimation(const std::string& name, bool loop) {
    if (name.empty()) return;

    // For looped animations, don't reset timing if same animation continues
    // CEGeometry handles looping internally via fmod on elapsed time
    if (name == m_currentAnim) {
        m_animLooping = loop;  // Just update loop flag
        return;  // Don't reset timing!
    }

    // ... rest of method for new animations
}
```

**Key Insight**: CEGeometry already handles animation looping correctly via `fmod(elapsedTime, totalTime)`. The controller should NOT reset start time for continuing loops.

#### 5. Kill Distance Sync Between AI Controller and Main Loop
**Problem**: T-Rex PHASE_EAT triggered at one distance, but player death in main.cpp used a different hardcoded `CONTACT_DISTANCE = 25.0f`.

**Symptom**: T-Rex would enter eat animation but player wouldn't die (or vice versa).

**Fix**: Add `getKillDistance()` accessor to AI controller and use it in main.cpp:
```cpp
// In CEClassicAIController.hpp
float getKillDistance() const { return m_config.apexKillDist * m_config.scale / 16.0f; }

// In main.cpp - use AI's kill distance instead of hardcoded value
float killDistance = classicAI->getKillDistance();
if (distance < killDistance) {
    // Player killed
}
```

**Key Insight**: ~~Keep kill logic in two places (AI state machine and main loop player death), but ensure they use the SAME distance value.~~ **UPDATED**: Let the AI be the sole decision maker. Main.cpp should ask the AI `hasKilledPlayer()` instead of doing its own distance calculation.

**Better Fix**: AI exposes `hasKilledPlayer()` method that returns true when in PHASE_EAT:
```cpp
// In CEClassicAIController.hpp
bool hasKilledPlayer() const { return m_Phase == PHASE_EAT && !m_isDead; }

// In main.cpp - ask AI if it killed the player (AI decides, not main.cpp)
if (classicAI->hasKilledPlayer()) {
    contactingClassicAI = classicAI.get();
    // Handle player death...
}
```

**Key Insight**: The AI already does the proper distance calculation (with head offset, height checks, etc.) during its `Process()` call. Main.cpp should NOT duplicate this logic - just ask the AI if it made a kill.

#### 6. Hit Zone Debug Visualization
**Pattern**: Added B-key toggle for wireframe sphere rendering of AI hit zones.

**Implementation**:
- `CEPhysicsWorld::drawClassicAIHitZones()` iterates ClassicAI controllers
- For each AI, get hit zones from `CEAIHitZoneManager`
- Render orange wireframe spheres at world positions
- Handles both SPHERE and CAPSULE shapes (capsules render as two spheres at endpoints)

**Files Modified**:
- `CEPhysicsWorld.h/cpp` - Added `drawClassicAIHitZones()`
- `main.cpp` - Call renderer when B key pressed

#### 7. Animation Duration for Phase Timeouts
**Problem**: Hardcoded timeout values (e.g., `3.0` seconds for eat animation) don't match actual animation duration, causing premature phase transitions.

**Symptom**: T-Rex eat animation ended after 3 seconds even though the actual animation was longer.

**Fix**: Use actual animation duration from `m_currentAnimDurationMs`:
```cpp
// BAD - hardcoded timeout
if (animElapsed > 3.0) {
    m_Phase = PHASE_WALK;
}

// GOOD - use actual animation duration
double animDurationSec = m_currentAnimDurationMs / 1000.0;
double eatDuration = std::max(2.0, animDurationSec);  // Minimum 2 seconds
if (animElapsed > eatDuration) {
    m_Phase = PHASE_WALK;
}
```

**Key Insight**: `setAnimation()` already stores the animation duration in `m_currentAnimDurationMs`. Use it for phase timeouts.

#### 8. Detection Spam During Special Phases
**Problem**: `updatePlayerPosition()` was triggering "Player detected!" messages every frame while T-Rex was eating.

**Symptom**: Log spam: "Player detected! Entering active state" repeated hundreds of times during eat sequence.

**Fix**: Skip detection updates during special phases like PHASE_EAT:
```cpp
void CEClassicAIController::updatePlayerPosition(glm::vec3 playerPos, double currentTime) {
    m_playerPos = playerPos;
    m_lastPlayerUpdate = currentTime;

    // Skip detection updates while eating (T-Rex is busy with its kill)
    if (m_Phase == PHASE_EAT) {
        return;
    }

    // ... rest of detection logic
}
```

**Key Insight**: Special phases (EAT, DIE, etc.) should block normal detection/state-change logic to prevent spam and inconsistent state.

### Speed Scaling Guidelines

The original Carnivores values needed ~3.3x scaling to match the raptor baseline:
- Raptor reference: walkSpeed=1.2, runSpeed=2.8
- Scaling factor: ~3.3x from original values

| Original | Scaled |
|----------|--------|
| 0.30 | 1.0 |
| 0.36 | 1.2 |
| 0.40 | 1.3 |
| 0.60 | 2.0 |
| 0.90 | 3.0 |
| 0.96 | 3.2 |
| 1.60 | 5.3 |

---

## Behavioral Archetypes

| Archetype | Examples | Key Behaviors |
|-----------|----------|---------------|
| **Aggressive Predator** | Velociraptor, Raptor | Jump + kill attack, slide mechanic, flee at distance |
| **Apex Predator** | T-Rex | Detection sequence (SEE/SMEL -> SCREAM -> chase), kill only, never flees, plays random lookout anims when passive |
| **Defensive Charger** | Triceratops | Flees at distance, charges when close |
| **Pure Herbivore** | Stegosaurus, Moschops, Pachycephalosaurus, Parasaurolophus, Gallimimus | Always flees, idle animations, no attacks |
| **Flying Ambient** | Dimorphodon, Pteranodon | No ground collision, altitude oscillation, no player interaction |

---

## Original Source Code Analysis

### Stegosaurus (AnimateSteg @ Characters.cpp:3591)

**State Machine:**
- State 0: Passive wandering with idle animations
- State 1: Fleeing from player
- State 2: Detection trigger (immediately transitions to State 1)

**Key Behaviors:**
```cpp
// AfraidTime ALWAYS decays (line 3597)
if (cptr->AfraidTime) cptr->AfraidTime = max(0, cptr->AfraidTime - TimeDt);

// Idle selection (~6.25% chance per animation cycle)
if (rRand(128) > 120) cptr->Phase=STG_IDLE1; else cptr->Phase=STG_WALK;

// Extended idle: 50% chance to stay in idle if already in IDLE2
if (cptr->Phase == STG_IDLE1 || cptr->Phase == STG_IDLE2) {
    if (rRand(128) > 64 && cptr->Phase == STG_IDLE2)
        cptr->Phase = STG_WALK;
    else
        cptr->Phase = STG_IDLE1 + rRand(1);  // Random idle
}

// Panic trigger (< 1024 units from player)
if (pdist < 1024.f) {
    cptr->State = 1;
    cptr->AfraidTime = (6 + rRand(8)) * 1024;  // 6-14 seconds panic
    cptr->Phase = STG_RUN;
}

// Zigzag evasion while fleeing (line 3652-3656)
if (cptr->AfraidTime) {
    cptr->tgalpha += (float)sin(RealTime/1024.f) / 3.f;
}

// ROTATION DISABLED during idle animations (line 3718)
if (cptr->Phase == STG_IDLE1 || cptr->Phase == STG_IDLE2) goto SKIPROT;

// Wander radius: 8048 units
SetNewTargetPlace(cptr, 8048.f);
```

**Speeds:** RUN=0.96, WALK=0.36

---

### T-Rex (AnimateTRex @ Characters.cpp:2264)

**State Machine:**
- State 0: Passive wandering (but plays random SEE/SMEL anims ~14% of the time!)
- State 2: Sight detection triggered (from CheckAfraid)
- State 3: Smell detection triggered (from CheckAfraid)
- State 1: Active chase (after detection sequence completes)
- State 5: Transition state (damage during detection -> immediate chase)

**Key Behaviors:**
```cpp
// NO AfraidTime decay - T-Rex doesn't use decay like herbivores

// Player distance uses HEAD OFFSET (line 2282-2284)
float playerdx = PlayerX - cptr->pos.x - cptr->lookx * 108;
float playerdz = PlayerZ - cptr->pos.z - cptr->lookz * 108;

// LookMode prevents rotation during detection animations
BOOL LookMode = FALSE;
if (cptr->Phase==REX_SEE  || cptr->Phase==REX_SEE1 ||
    cptr->Phase==REX_SMEL || cptr->Phase==REX_SMEL1) LookMode = TRUE;

// Detection trigger (must face player within 0.4 radians first)
if (cptr->State > 1)
    if (AngleDifference(cptr->alpha, palpha) < 0.4f) {
        if (cptr->State==2) cptr->Phase = REX_SEE1 + rRand(1);  // Random sight anim
                       else cptr->Phase = REX_SMEL + rRand(1);  // Random smell anim
        cptr->State = 1;
        cptr->rspeed = 0;  // Stop rotating
    }

// After detection animation completes -> play SCREAM then chase
if (cptr->State)
    if (NewPhase && LookMode) {
        cptr->Phase = REX_SCREAM;
    }

// Damage during detection -> skip to chase (State 5 transition)
if (cptr->State == 5) {
    NewPhase = TRUE;
    cptr->State = 1;
    cptr->Phase = REX_WALK;
    cptr->FTime = 0;
    cptr->tgx = PlayerX;
    cptr->tgz = PlayerZ;
}

// PASSIVE lookout behavior (~14% chance per animation cycle)
if (!cptr->State)
    if (NewPhase)
        if (rRand(128) > 110) {  // ~14% chance
            if (rRand(128) > 64) cptr->Phase = REX_SEE1 + rRand(1);
                            else cptr->Phase = REX_SMEL + rRand(1);
        }

// Rotation disabled during SCREAM, EAT, and LookMode (line 2425-2426)
if (cptr->Phase==REX_SCREAM || cptr->Phase==REX_EAT) goto SKIPROT;
if (LookMode) goto SKIPROT;

// Kill attack at close range (line 2314-2324)
if (pdist < 380)
    if (fabs(PlayerY - cptr->pos.y) < 256) {
        cptr->vspeed /= 8.0f;
        cptr->State = 1;
        cptr->Phase = REX_EAT;
        AddDeadBody(cptr, HUNT_KILL);
    }

// Zigzag only when chasing at distance > 5648 (line 2338-2342)
if (cptr->State && pdist > 5648) {
    cptr->tgalpha += (float)sin(RealTime/824.f) / 6.f;  // Smaller amplitude than herbivores
}

// RUN phase requires facing player within 1.0 radian (line 2390-2392)
if (fabs(cptr->tgalpha - cptr->alpha) < 1.0 ||
    fabs(cptr->tgalpha - cptr->alpha) > 2*pi-1.0)
        cptr->Phase = REX_RUN; else cptr->Phase = REX_WALK;

// Water threshold: 560 units (different from other creatures)
if (GetLandUpH(...) - GetLandH(...) > 560 * cptr->scale)
    cptr->StateF |= csONWATER;
```

**Speeds:** RUN=2.49, WALK=0.76, SWIM=0.70

---

### Dimorphodon (AnimateDimor @ Characters.cpp:4142)

**Flight Phases:**
```cpp
#define DIM_FLY    0   // Ascending - speed 1.5
#define DIM_FLYP   1   // Descending - speed 1.3
#define DIM_FALL   2   // Death fall
#define DIM_DIE    3   // Death on ground
```

**Key Behaviors:**
```cpp
// NO state machine - purely ambient, always State 0
// NO player interaction at all
// NO AfraidTime

// Altitude control with smooth interpolation (line 4265-4268)
if (cptr->Phase == DIM_FLY)
    DeltaFunc(cptr->pos.y, GetLandH(x,z) + 4048, TimeDt / 6.f);   // Rise to 4048 units
else
    DeltaFunc(cptr->pos.y, GetLandH(x,z), TimeDt / 16.f);         // Descend to ground level

// Phase switching based on altitude thresholds (line 4194-4201)
if (cptr->Phase == DIM_FLY)
    if (cptr->pos.y > GetLandH(...) + 2800)
        cptr->Phase = DIM_FLYP;  // Start diving
if (cptr->Phase == DIM_FLYP)
    if (cptr->pos.y < GetLandH(...) + 1800)
        cptr->Phase = DIM_FLY;   // Start climbing

// Minimum altitude floor (line 4271-4272)
if (cptr->pos.y < GetLandH(x,z) + 236)
    cptr->pos.y = GetLandH(x,z) + 256;

// Banking roll during turns - STRONGER than ground creatures (line 4283-4285)
cptr->tggamma = cptr->rspeed / 4.0f;  // vs /5-16 for ground
if (cptr->tggamma > pi / 6.f) cptr->tggamma = pi / 6.f;
if (cptr->tggamma <-pi / 6.f) cptr->tggamma =-pi / 6.f;

// NO MoveCharacter call - direct position update (line 4280-4281)
cptr->pos.x += cptr->lookx * cptr->vspeed * TimeDt;
cptr->pos.z += cptr->lookz * cptr->vspeed * TimeDt;

// Wander radius: 4048 units (smaller than ground creatures)
SetNewTargetPlace(cptr, 4048.f);

// Target reached distance: 1024 units
if (tdist < 1024) SetNewTargetPlace(cptr, 4048.f);

// Audio plays randomly: ~4% chance per phase change (line 4209)
if ((rand() & 1023) > 980) ActivateCharacterFx(cptr);

// Respawn handled by spawner system (original used ReplaceCharacterForward)
```

**Speeds:** FLY=1.5, FLYP=1.3

---

### Triceratops (AnimateTric @ Characters.cpp:3011)

**State Machine:**
- State 0: Passive wandering with idle animations
- State 1: Active (fleeing OR charging based on distance)
- State 2: Detection trigger (immediately transitions to State 1)

**Key Behaviors:**
```cpp
// AfraidTime decays normally (line 3017)
if (cptr->AfraidTime) cptr->AfraidTime = max(0, cptr->AfraidTime - TimeDt);

// BUT: AfraidTime REFRESHES when player is close! (line 3040)
if (pdist < 6000) cptr->AfraidTime = 8000;  // Fixed 8 second refresh

// Player distance uses HORN OFFSET (line 3031-3032)
float playerdx = PlayerX - cptr->pos.x - cptr->lookx * 300 * cptr->scale;
float playerdz = PlayerZ - cptr->pos.z - cptr->lookz * 300 * cptr->scale;

// Distance-based behavior switch (line 3048-3058)
// Threshold: 256*16 + OptAgres/8 = ~4096+ units
if (pdist > 256*16 + OptAgres/8) {
    // FLEE: Run away from player
    nv = normalize(player - pos) * 2048;
    cptr->tgx = cptr->pos.x - nv.x;
    cptr->tgz = cptr->pos.z - nv.z;
} else {
    // CHARGE: Run toward player
    cptr->tgx = PlayerX;
    cptr->tgz = PlayerZ;
}

// TRAMPLING KILL at close range (line 3062-3066) - NOT a charge animation!
if (pdist < 300)
    if (fabs(PlayerY - cptr->pos.y - 160) < 256) {
        cptr->State = 0;  // Returns to passive after kill
        AddDeadBody(cptr, HUNT_EAT);
    }

// Idle selection (~3% chance per animation cycle) (line 3116)
if (rRand(128) > 124) cptr->Phase = TRI_IDLE1; else cptr->Phase = TRI_WALK;

// Extended idle: 50% chance to stay if in IDLE3 (line 3110-3114)
if (cptr->Phase == TRI_IDLE1 || TRI_IDLE2 || TRI_IDLE3) {
    if (rRand(128) > 64 && cptr->Phase == TRI_IDLE3)
        cptr->Phase = TRI_WALK;
    else
        cptr->Phase = TRI_IDLE1 + rRand(2);  // Random from 3 idle anims
}

// ROTATION DISABLED during idle animations (line 3149)
if (cptr->Phase == TRI_IDLE1 || TRI_IDLE2 || TRI_IDLE3) goto SKIPROT;

// Zigzag evasion while fleeing (same as Stegosaurus)
if (cptr->AfraidTime) {
    cptr->tgalpha += (float)sin(RealTime/1024.f) / 3.f;
}
```

**Speeds:** RUN=1.2, WALK=0.30

**IMPORTANT CORRECTION:** Triceratops does NOT have a special "charge animation" - it uses the same RUN animation for both fleeing and charging. The kill is a trampling death when the player is within 300 units of the horn tip.

---

## Extended ClassicAIConfig Parameters

### Design Philosophy

**BehaviorType is the primary driver.** Rather than having many individual boolean flags that could create invalid combinations, the `behaviorType` enum determines all core behavioral characteristics. This matches how the original game worked - each dinosaur type had its own dedicated `Animate*()` function with hardcoded behavior.

The state machine (State 0, 1, 2, 3, 5) uses the same integer values across all types, but their *meaning* is behavior-specific:

| State | AGGRESSIVE_PREDATOR | APEX_PREDATOR | PURE_HERBIVORE | DEFENSIVE_CHARGER | FLYING_AMBIENT |
|-------|---------------------|---------------|----------------|-------------------|----------------|
| 0 | Passive wander | Passive + lookout | Passive + idle | Passive + idle | Flying (only state) |
| 1 | Active chase/flee | Active chase | Fleeing | Fleeing or charging | N/A |
| 2 | Detection trigger | Sight detected | Detection trigger | Detection trigger | N/A |
| 3 | N/A | Smell detected | N/A | N/A | N/A |
| 5 | N/A | Damage abort | N/A | N/A | N/A |

### New Enumerations

```cpp
// Behavioral archetype - PRIMARY driver of AI behavior
// Replaces individual flags like canJumpAttack, alwaysFlees, hasSlide, isFlying
enum class BehaviorType {
    AGGRESSIVE_PREDATOR,  // Velociraptor - jump+kill attack, slide mechanic, flee at distance
    APEX_PREDATOR,        // T-Rex - detection sequence, kill only, never flees, passive lookout
    DEFENSIVE_CHARGER,    // Triceratops - flee at distance, charge attack when close
    PURE_HERBIVORE,       // Stegosaurus - always flee, idle animations, no attacks
    FLYING_AMBIENT        // Dimorphodon - altitude oscillation, no ground collision, no combat
};
```

### New Animation Phases

```cpp
enum ClassicAnimPhase {
    // Existing phases (0-7)
    PHASE_RUN = 0,
    PHASE_WALK = 1,
    PHASE_SLIDE = 2,
    PHASE_SWIM = 3,
    PHASE_JUMP = 4,
    PHASE_DIE = 5,
    PHASE_EAT = 6,
    PHASE_SLEEP = 7,

    // New phases for extended behavior
    PHASE_IDLE = 8,          // Herbivore/passive idle animations (multiple can be configured)
    PHASE_DETECT_SIGHT = 9,  // T-Rex REX_SEE/REX_SEE1 (randomly selected)
    PHASE_DETECT_SMELL = 10, // T-Rex REX_SMEL/REX_SMEL1 (randomly selected)
    PHASE_ROAR = 11,         // T-Rex REX_SCREAM before chase
    PHASE_FLY_UP = 12,       // Dimorphodon ascending
    PHASE_FLY_DOWN = 13,     // Dimorphodon descending
    PHASE_FALL = 14          // Flying creature death fall
};
```

**Note:** Triceratops does NOT need a PHASE_CHARGE - it uses PHASE_RUN for both fleeing and charging.

### New Config Parameters

```cpp
struct ClassicAIConfig {
    // === EXISTING PARAMETERS (unchanged) ===
    std::string name;
    bool debugEnabled = false;
    std::string walkAnim, runAnim, dieAnim, eatAnim, jumpAnim, swimAnim, slideAnim;
    float walkSpeed, runSpeed, jumpSpeed, swimSpeed;
    float scale, heightOffset;
    float wanderRadius, targetReachDist;
    float jumpMinDist, jumpMaxDist, killDist, killHeightDiff;
    float chaseMaxDist, fleeThreshold;
    float zigzagFrequency, zigzagAmplitude, zigzagMinDist;
    float turnRateBase, turnRateProportional, afraidTurnMultiplier;
    float slideSpeedThreshold, slideAngleThreshold;
    float bendDivisor, bendMax;
    float pitchSampleDist, rollSampleDist, maxPitch, maxRoll;
    float viewRange, afraidDuration;
    float maxHealth, hitRadius, hitHeight, hitCenterY;
    bool isDangerous, avoidsWater;

    // === BEHAVIORAL ARCHETYPE (primary behavior driver) ===
    BehaviorType behaviorType = BehaviorType::AGGRESSIVE_PREDATOR;

    // === IDLE ANIMATIONS (PURE_HERBIVORE, DEFENSIVE_CHARGER, APEX_PREDATOR passive) ===
    std::vector<std::string> idleAnims;      // e.g., ["Sr_idle1", "Sr_idle2"] - randomly selected
    float idleOdds = 0.0f;                   // Chance per anim cycle (0.0625 = ~6.25%)

    // === DETECTION SEQUENCE (APEX_PREDATOR only) ===
    std::vector<std::string> sightDetectAnims;   // ["Tx_see", "Tx_see1"] - randomly selected
    std::vector<std::string> smellDetectAnims;   // ["Tx_smel", "Tx_smel1"] - randomly selected
    std::string roarAnim;                        // "Tx_scream" - played after detection
    float detectFaceAngle = 0.4f;                // Must face player within this (radians)
    float passiveLookoutOdds = 0.14f;            // Chance for passive lookout anim (~14%)
    float runFaceAngle = 1.0f;                   // Must face player within this to use RUN (radians)
    float headOffset = 108.0f;                   // Distance offset for head position (original units)
    float apexKillDist = 380.0f;                 // Kill range (original units) - ~23.75 world units
    float apexZigzagMinDist = 5648.0f;           // Only zigzag when chasing beyond this (original units)

    // === DISTANCE-BASED BEHAVIOR (DEFENSIVE_CHARGER only) ===
    // Below this distance: charge toward player; above: flee from player
    float chargeThresholdDist = 256.0f;      // ~4096 original units = 256 world units (16 tiles)
    float trampleKillDist = 18.75f;          // ~300 original units = trampling death range
    float hornOffset = 300.0f;               // Distance offset for horn position (original units, scaled)

    // === PANIC/FEAR (PURE_HERBIVORE, DEFENSIVE_CHARGER) ===
    float panicTriggerDist = 64.0f;          // Distance to trigger panic (~1024 original = 4 tiles)
    float panicDurationMin = 6.0f;           // Min panic duration (seconds)
    float panicDurationMax = 14.0f;          // Max panic duration (seconds)

    // === AFRAID REFRESH (DEFENSIVE_CHARGER only) ===
    float afraidRefreshDist = 375.0f;        // ~6000 original units - refresh to 8 seconds when closer
    float afraidRefreshTime = 8.0f;          // Fixed refresh value (seconds)

    // === FLYING (FLYING_AMBIENT only) ===
    float cruisingAltitude = 253.0f;         // Target altitude above terrain (world units)
    float altitudeUpperThreshold = 175.0f;   // Start diving when above this
    float altitudeLowerThreshold = 112.5f;   // Start climbing when below this
    float ascentRate = 6.0f;                 // DeltaFunc divisor for ascent (smaller = faster)
    float descentRate = 16.0f;               // DeltaFunc divisor for descent
    float minAltitude = 16.0f;               // Minimum altitude floor
    float flyUpSpeed = 1.5f;                 // Speed while ascending
    float flyDownSpeed = 1.3f;               // Speed while descending
    std::string flyUpAnim;                   // Ascending animation
    std::string flyDownAnim;                 // Descending animation
    std::string fallAnim;                    // Death fall animation

    // NOTE: Respawn/despawn behavior is handled by CEAISpawner, not the AI controller
};
```

### Behavior Characteristics by Type

| Characteristic | AGGRESSIVE_PREDATOR | APEX_PREDATOR | DEFENSIVE_CHARGER | PURE_HERBIVORE | FLYING_AMBIENT |
|----------------|---------------------|---------------|-------------------|----------------|----------------|
| Jump Attack | Yes | No | No | No | No |
| Kill Attack | Yes | Yes | Yes (trample) | No | No |
| Slide Mechanic | Yes | No | No | No | No |
| Can Flee | Yes (at distance) | Never | Yes (at distance) | Always | N/A |
| Idle Anims | No | Yes (lookout) | Yes | Yes | No |
| Detection Sequence | No | Yes | No | No | No |
| Ground Collision | Yes | Yes | Yes | Yes | No |
| Afraid Decay | When fleeing | Never | Refresh when close | Always | N/A |
| Zigzag | When chasing | When chasing (far) | When fleeing | When fleeing | No |
| Uses RUN for attack | No | No | Yes (trampling) | N/A | N/A |

---

## Example JSON Configurations

### Stegosaurus (Pure Herbivore)

```json
{
  "controller": "ClassicAI",
  "args": {
    "name": "Stego",
    "behaviorType": "PURE_HERBIVORE",

    "animations": {
      "WALK": "Sr_wlk",
      "RUN": "Sr_run",
      "DIE": "Sr_die"
    },

    "idleAnims": ["Sr_idle1", "Sr_idle2"],
    "idleOdds": 0.0625,

    "panicTriggerDist": 64.0,
    "panicDurationMin": 6.0,
    "panicDurationMax": 14.0,

    "character": {
      "walkSpeed": 0.36,
      "runSpeed": 0.96,
      "maxHealth": 60.0
    }
  }
}
```

### T-Rex (Apex Predator)

```json
{
  "controller": "ClassicAI",
  "args": {
    "name": "TRex",
    "behaviorType": "APEX_PREDATOR",

    "animations": {
      "WALK": "Tx_walk",
      "RUN": "Tx_run",
      "DIE": "Tx_die",
      "EAT": "Tx_eat",
      "SWIM": "Tx_swim"
    },

    "sightDetectAnims": ["Tx_see", "Tx_see1"],
    "smellDetectAnims": ["Tx_smel", "Tx_smel1"],
    "roarAnim": "Tx_scream",
    "detectFaceAngle": 0.4,
    "passiveLookoutOdds": 0.14,
    "runFaceAngle": 1.0,
    "headOffset": 108.0,
    "apexKillDist": 380.0,
    "apexZigzagMinDist": 5648.0,

    "character": {
      "walkSpeed": 0.76,
      "runSpeed": 2.49,
      "swimSpeed": 0.70,
      "maxHealth": 500.0,
      "scale": 2.0
    }
  }
}
```

### Dimorphodon (Flying Ambient)

```json
{
  "controller": "ClassicAI",
  "args": {
    "name": "Dimor",
    "behaviorType": "FLYING_AMBIENT",

    "animations": {
      "DIE": "Dim_die"
    },

    "flyUpAnim": "Dim_fly",
    "flyDownAnim": "Dim_flyp",
    "fallAnim": "Dim_fall",

    "cruisingAltitude": 253.0,
    "altitudeUpperThreshold": 175.0,
    "altitudeLowerThreshold": 112.5,
    "ascentRate": 6.0,
    "descentRate": 16.0,
    "minAltitude": 16.0,
    "flyUpSpeed": 1.5,
    "flyDownSpeed": 1.3,

    "character": {
      "maxHealth": 30.0,
      "scale": 0.8
    }
  }
}
```

### Triceratops (Defensive Charger)

```json
{
  "controller": "ClassicAI",
  "args": {
    "name": "Tric",
    "behaviorType": "DEFENSIVE_CHARGER",

    "animations": {
      "WALK": "Tric_walk",
      "RUN": "Tric_run",
      "DIE": "Tric_die"
    },

    "idleAnims": ["Tric_idle1", "Tric_idle2", "Tric_idle3"],
    "idleOdds": 0.023,

    "chargeThresholdDist": 256.0,
    "trampleKillDist": 18.75,
    "hornOffset": 300.0,
    "afraidRefreshDist": 375.0,
    "afraidRefreshTime": 8.0,

    "character": {
      "walkSpeed": 0.30,
      "runSpeed": 1.2,
      "maxHealth": 200.0,
      "scale": 1.5
    }
  }
}
```

**Note:** Triceratops does NOT have a special charge animation. It uses the same RUN animation for both fleeing and charging. The kill occurs as a "trampling" death when the player is within `trampleKillDist` of the horn tip (calculated with 300-unit horn offset).

---

## Pre-Implementation Refactoring

Before implementing new behavior types, the following refactoring is needed to prepare the codebase:

### 1. Extract Behavior-Specific Methods from AnimateCharacter()

The current `AnimateCharacter()` is ~400 lines of monolithic Velociraptor-specific logic. Refactor to:

```cpp
void AnimateCharacter(double currentTime, double deltaTime) {
    // Common pre-processing (water check, timer updates, distance calc)

    switch (m_config.behaviorType) {
        case BehaviorType::AGGRESSIVE_PREDATOR:
            AnimatePredator(currentTime, deltaTime, TimeDt);
            break;
        case BehaviorType::APEX_PREDATOR:
            AnimateApexPredator(currentTime, deltaTime, TimeDt);
            break;
        case BehaviorType::PURE_HERBIVORE:
            AnimateHerbivore(currentTime, deltaTime, TimeDt);
            break;
        case BehaviorType::DEFENSIVE_CHARGER:
            AnimateDefensiveCharger(currentTime, deltaTime, TimeDt);
            break;
        case BehaviorType::FLYING_AMBIENT:
            AnimateFlying(currentTime, deltaTime, TimeDt);
            break;
    }

    // Common post-processing (terrain adaptation, animation update)
}
```

### 2. Add LookMode Flag

Add member variable and handling for rotation lock:
```cpp
bool m_LookMode = false;  // True = disable rotation (detection anims)
```

Modify rotation logic in `AnimateCharacter()`:
```cpp
// Skip rotation if LookMode is active
if (m_LookMode) goto SKIPROT;  // Or use early return pattern
```

### 3. Extend getPhaseAnimation()

Add cases for new phases:
```cpp
case PHASE_IDLE:
    // Return random selection from idleAnims[]
    return selectRandomIdleAnim();
case PHASE_DETECT_SIGHT:
    return selectRandomDetectAnim(m_config.sightDetectAnims);
case PHASE_DETECT_SMELL:
    return selectRandomDetectAnim(m_config.smellDetectAnims);
case PHASE_ROAR:
    return m_config.roarAnim;
case PHASE_FLY_UP:
    return m_config.flyUpAnim;
case PHASE_FLY_DOWN:
    return m_config.flyDownAnim;
case PHASE_FALL:
    return m_config.fallAnim;
```

### 4. Add Idle Animation Selection Helper

```cpp
std::string selectRandomIdleAnim() const {
    if (m_config.idleAnims.empty()) return m_config.walkAnim;
    int idx = rand() % m_config.idleAnims.size();
    return m_config.idleAnims[idx];
}
```

### 5. Add Flying Movement Method

Flying creatures bypass `MoveCharacter()` and update position directly:
```cpp
void MoveFlying(double deltaTime) {
    float TimeDt = deltaTime * 1000.0f;
    float speed = (m_Phase == PHASE_FLY_UP) ? m_config.flyUpSpeed : m_config.flyDownSpeed;

    // Direct position update (no collision, no terrain height)
    m_pos.x += m_lookx * speed * TimeDt / 1000.0f;
    m_pos.z += m_lookz * speed * TimeDt / 1000.0f;

    // Altitude control via DeltaFunc
    float groundH = GetLandH(m_pos.x, m_pos.z);
    float targetAlt = groundH + m_config.cruisingAltitude;

    if (m_Phase == PHASE_FLY_UP) {
        DeltaFunc(m_pos.y, targetAlt, TimeDt / m_config.ascentRate);
    } else {
        DeltaFunc(m_pos.y, groundH, TimeDt / m_config.descentRate);
    }

    // Minimum altitude floor
    if (m_pos.y < groundH + m_config.minAltitude) {
        m_pos.y = groundH + m_config.minAltitude;
    }
}
```

### 6. Update Debug Window Phase Names

Add new phases to `updateDebugWindow()` switch statement:
```cpp
case PHASE_IDLE: phaseName = "IDLE"; break;
case PHASE_DETECT_SIGHT: phaseName = "DETECT_SIGHT"; break;
case PHASE_DETECT_SMELL: phaseName = "DETECT_SMELL"; break;
case PHASE_ROAR: phaseName = "ROAR"; break;
case PHASE_FLY_UP: phaseName = "FLY_UP"; break;
case PHASE_FLY_DOWN: phaseName = "FLY_DOWN"; break;
case PHASE_FALL: phaseName = "FALL"; break;
```

### 7. Idle Rotation Lock

Both herbivores and defensive chargers disable rotation during idle animations:
```cpp
// In AnimateHerbivore/AnimateDefensiveCharger:
if (m_Phase == PHASE_IDLE) {
    m_rspeed = 0.0f;  // Or skip rotation entirely
    return;  // Skip rotation block
}
```

---

## Implementation Phases

### Phase 1: Core Infrastructure + Herbivore (PURE_HERBIVORE)

**Step 1.1: Header Updates**
- Add `BehaviorType` enum to header (already done)
- Add all new config parameters to `ClassicAIConfig` struct
- Add new member variables: `m_LookMode`, `m_currentIdleAnim`
- Declare new methods: `AnimatePredator()`, `AnimateHerbivore()`, `selectRandomIdleAnim()`

**Step 1.2: Refactor AnimateCharacter()**
- Extract common pre-processing (water check, timer decay, distance calc) to top
- Add behavioral switch statement to dispatch to type-specific methods
- Move existing Velociraptor logic into `AnimatePredator()`
- Extract common post-processing (terrain adaptation, phase animation) to bottom
- Ensure existing Velociraptor behavior is preserved exactly

**Step 1.3: Implement AnimateHerbivore()**
- State 0: Passive wander with idle animations
  - ~6.25% chance per animation cycle to select idle
  - Random selection from `idleAnims[]` array
  - 50% chance to stay in idle if already idle
  - Rotation disabled during idle
- State 1: Fleeing from player
  - AfraidTime ALWAYS decays
  - Zigzag evasion while fleeing
- State 2: Detection trigger -> State 1
- Panic trigger at `panicTriggerDist` (< 64 world units)

**Step 1.4: Update Supporting Code**
- Extend `getPhaseAnimation()` with PHASE_IDLE case
- Add `selectRandomIdleAnim()` helper method
- Update `loadConfig()` to parse: `behaviorType`, `idleAnims`, `idleOdds`, `panicTriggerDist`, `panicDurationMin`, `panicDurationMax`
- Update `updateDebugWindow()` with new phase names

**Step 1.5: Verification**
- Build and verify compilation
- Test existing Velociraptor AI behavior is unchanged
- Test new Stegosaurus-style herbivore behavior

### Phase 2: Apex Predator (APEX_PREDATOR)
- Add detection sequence config parameters
- Add `PHASE_DETECT_SIGHT`, `PHASE_DETECT_SMELL`, `PHASE_ROAR` handling
- Implement `AnimateApexPredator()` method:
  - State 0: Passive wander with random lookout animations
  - State 2: Sight detection (must face player first)
  - State 3: Smell detection
  - State 1: Active chase (never flees)
  - State 5: Damage abort (skip detection, go straight to chase)
- Add `m_LookMode` flag to disable rotation during detection anims
- Afraid time never decays once triggered

### Phase 3: Flying Ambient (FLYING_AMBIENT)
- Add flying config parameters
- Add `PHASE_FLY_UP`, `PHASE_FLY_DOWN`, `PHASE_FALL` handling
- Implement `AnimateFlying()` method:
  - Only State 0 (purely ambient)
  - Altitude oscillation using `DeltaFunc`
  - Phase switches based on altitude thresholds
  - No ground collision checks
  - No player interaction
- Death fall behavior when killed
- Note: Respawn/despawn handled by spawner system

### Phase 4: Defensive Charger (DEFENSIVE_CHARGER) - Optional
- Add distance-based behavior config parameters (`chargeThresholdDist`, `trampleKillDist`)
- Add afraid refresh parameters (`afraidRefreshDist`, `afraidRefreshTime`)
- Implement `AnimateDefensiveCharger()` method:
  - State 0: Passive wander with idle animations (randomly selected from array)
  - State 1: Behavior switches based on distance:
    - Above `chargeThresholdDist`: Flee from player (same as herbivore)
    - Below `chargeThresholdDist`: Charge toward player (using RUN animation)
  - Afraid time refreshes when player within `afraidRefreshDist`
  - Rotation disabled during idle animations
- Trampling kill when player within `trampleKillDist` of horn tip
- Horn offset for distance calculation (300 * scale)

---

## State Machine Reference

### Current (Velociraptor)
```
State 0: Passive wandering
State 1: Active (chasing or fleeing based on distance)
State 2: Detection trigger -> immediately to State 1
```

### Extended (T-Rex)
```
State 0: Passive wandering (with random lookout anims)
State 1: Active chase
State 2: Sight detection triggered
State 3: Smell detection triggered
State 5: Damage abort -> skip to chase
```

### Herbivore
```
State 0: Passive wandering (with idle animations)
State 1: Fleeing from player
State 2: Detection trigger -> immediately to State 1
```

### Flying
```
State 0: Flying (no other states - purely ambient)
Phase alternates: FLY_UP <-> FLY_DOWN based on altitude
```

---

## Files to Modify

1. **CEClassicAIController.hpp**
   - Add new enum types
   - Extend `ClassicAIConfig` struct
   - Add new phase constants
   - Add new member variables for state tracking

2. **CEClassicAIController.cpp**
   - Extend `loadConfig()` to parse new parameters
   - Modify `AnimateCharacter()` with behavioral branching
   - Add new methods: `AnimateIdle()`, `AnimateDetection()`, `AnimateFlight()`
   - Modify movement/rotation to respect `LookMode` and flying

3. **config.json** (runtime)
   - Update spawn configurations to use new parameters

---

## References

- Original Carnivores 2 source: `/Users/tminard/source/carnivores/carnivores_original/Carnivores 2/Characters.cpp`
- Existing AI documentation: `/docs/classic_ai.md`
- Current ClassicAI implementation: `/src/CEClassicAIController.cpp`
