#version 330 core

in vec2 texCoord0;
in vec3 surfaceNormal;
in vec3 toLightVector;
in vec3 viewDirection;
in vec3 worldPos;
in float distFromCamera;

out vec4 outputColor;

uniform sampler2D basic_texture;
uniform sampler2D skyTexture;
uniform float time;
uniform vec4 skyColor;
uniform float view_distance;
uniform float waterLevel;
uniform float tileWidth = 16.0;  // World units per tile

// Time of day
uniform float visibilityMultiplier = 1.0;
uniform int timeOfDay = 0;  // 0=DAY, 1=DUSK, 2=NIGHT (explicit, not derived from visibility)

// Time-of-day fog colors
const vec3 DUSK_FOG_COLOR = vec3(0.25, 0.12, 0.06);  // Warm amber for dusk/dawn
const vec3 NIGHT_FOG_COLOR = vec3(0.02, 0.03, 0.06); // Dark blue-black for night

// Camera position for fog
uniform vec3 cameraPos;

// Underwater fog - for fogging out the water surface when viewed from below
uniform float cameraY = 0.0;  // Camera Y position for underwater detection

// Underwater fog color - dark blue-teal for ocean depths
const vec3 underwaterFogColor = vec3(0.02, 0.08, 0.14);

// Map dimensions for edge blending
uniform float terrainWidth = 512.0;   // Map width in tiles
uniform float terrainHeight = 512.0;  // Map height in tiles

// Atlas UV bounds for the water texture (passed via texCoord0 corners)
// We'll use these to map world-based tiling into atlas space
uniform vec2 atlasUVMin;  // Bottom-left UV of water texture in atlas
uniform vec2 atlasUVMax;  // Top-right UV of water texture in atlas

void main()
{
    // ========== EARLY EXIT: UNDERWATER VIEW ==========
    // When camera is underwater, apply distance and depth based fog to hide coastlines
    if (cameraY < waterLevel) {
        float depthBelowSurface = waterLevel - cameraY;

        // Depth-based darkening: start immediately, fully dark by 32 units (2 tiles)
        float depthDarkening = smoothstep(0.0, 32.0, depthBelowSurface);

        // Distance-based darkening: distant water surface is darker
        // Even when just below surface, distant water should fade to dark
        // Use aggressive range: start at 1 tile, fully dark by 4 tiles
        float distanceDarkening = smoothstep(16.0, 64.0, distFromCamera);

        // Combine - EITHER depth OR distance causes darkening
        float combinedDarkening = max(depthDarkening, distanceDarkening);
        combinedDarkening = clamp(combinedDarkening, 0.0, 1.0);

        // Fog colors - dark blue for underwater, darker for deep
        vec3 deepUnderwaterColor = underwaterFogColor * 0.1;  // 10% of fog color for deep areas

        vec3 finalColor = mix(underwaterFogColor, deepUnderwaterColor, combinedDarkening);
        outputColor = vec4(finalColor, 1.0);
        return;
    }

    // Use world position for tiling - each tile (16 world units) gets one texture repeat
    vec2 worldUV = worldPos.xz / tileWidth;

    // Animate the UVs
    vec2 animUV = worldUV;
    animUV.x += sin(time * 0.3 + worldUV.y * 2.0) * 0.02;
    animUV.y += cos(time * 0.25 + worldUV.x * 2.0) * 0.02;

    // Get fractional part for tiling (0-1 range)
    vec2 tiledUV = fract(animUV);

    // Map tiled UV (0-1) into atlas texture bounds
    // atlasUVMin/Max are uniforms set from C++ via scaleAtlasUV()
    vec2 atlasSize = atlasUVMax - atlasUVMin;
    vec2 finalUV = atlasUVMin + tiledUV * atlasSize;

    // Sample with distance-based mip level for natural blur at distance
    float distMipLevel = clamp(distFromCamera / (16.0 * 6.0), 0.0, 3.0);
    vec4 sC = textureLod(basic_texture, finalUV, distMipLevel);
    vec3 waterTexColor = vec3(sC.b, sC.g, sC.r);

    // Vectors for lighting
    vec3 normal = normalize(surfaceNormal);
    vec3 lightDir = normalize(toLightVector);

    float diffuse = max(dot(normal, lightDir), 0.0);
    float brightness = 0.4 + diffuse * 0.4;  // Match main water ambient (0.4)
    vec3 surfaceColor = waterTexColor * brightness;

    // Ocean water has NO distance fog - it should look uniform to the horizon
    // DoF and minimap handle distance effects instead
    vec3 finalColor = surfaceColor;

    // === TIME OF DAY (DAWN/DUSK/NIGHT) ===
    // Uses explicit timeOfDay uniform: 0=DAY, 1=DUSK, 2=NIGHT
    // Dusk/dawn gets warm orange/amber tints, night gets cool blue moonlight
    if (timeOfDay > 0) {
        float effectIntensity = 1.0 - visibilityMultiplier;

        bool isDusk = (timeOfDay == 1);
        bool isNight = (timeOfDay == 2);

        // 1. Darkening - dusk is brighter than night
        float darkening = 1.0;
        if (isDusk) {
            darkening = 1.0 - (effectIntensity * 0.45);
        } else if (isNight) {
            darkening = 1.0 - (effectIntensity * 0.90);
        }
        finalColor *= darkening;

        // 2. Desaturation - dusk keeps more color
        vec3 grayscale = vec3(dot(finalColor, vec3(0.299, 0.587, 0.114)));
        float desatAmount = 0.0;
        if (isDusk) {
            desatAmount = effectIntensity * 0.25;
        } else if (isNight) {
            desatAmount = effectIntensity * 0.85;
        }
        finalColor = mix(finalColor, grayscale, desatAmount);

        // 3. Color tinting - warm amber for dusk, cool blue for night
        if (isDusk) {
            vec3 duskTint = vec3(1.25, 0.80, 0.50);  // Warm amber/orange for sunset water
            finalColor = mix(finalColor, finalColor * duskTint, effectIntensity * 0.65);
        } else if (isNight) {
            vec3 nightTint = vec3(0.4, 0.5, 0.8);    // Cool blue moonlight
            finalColor = mix(finalColor, finalColor * nightTint, effectIntensity * 0.6);
        }
    }

    // NOTE: Underwater view is handled by early exit at top of main()

    // Ocean water is always fully opaque when viewed from above
    outputColor = vec4(finalColor, 1.0);
}
