#version 330 core
out vec4 outputColor;

in vec2 TexCoords0;
in vec3 WorldPos;
in vec3 CameraRelativePos;  // Position relative to camera for proper fade
in vec2 RadialUV;           // Radial UV for stretch correction

uniform sampler2D basic_texture;
uniform vec4 sky_color;
uniform float rainIntensity = 0.0; // 0.0 = clear, 1.0 = heavy rain
uniform float time = 0.0;          // Time for subtle animation
uniform float visibilityMultiplier = 1.0;  // Time-of-day visibility (1.0 = day, 0.35 = night)
uniform int timeOfDay = 0;  // 0=DAY, 1=DUSK, 2=NIGHT (explicit, not derived from visibility)

// Atmospheric color constants for realistic sky appearance
const vec3 SUNLIGHT_COLOR = vec3(1.0, 0.95, 0.85);      // Warm sunlight
const vec3 SHADOW_TINT = vec3(0.7, 0.75, 0.9);          // Cool blue shadows
const vec3 HORIZON_TINT = vec3(0.85, 0.88, 0.95);       // Slight haze at distance
const vec3 DUSK_FOG_COLOR = vec3(0.25, 0.12, 0.06);     // Warm amber dusk fog (matches terrain)
const vec3 NIGHT_FOG_COLOR = vec3(0.02, 0.03, 0.06);    // Dark blue-black night fog (matches terrain)

void main()
{
    // Calculate circular distance from camera (already relative due to zeroed view translation)
    float distFromCenter = length(CameraRelativePos.xz);

    // Create very gradual circular fade - extended range for more natural blending
    // Start fade earlier and extend the range for a smoother, more gradual transition
    // At night, clouds fade to horizon much sooner to match reduced world visibility
    float effectIntensity = 1.0 - visibilityMultiplier;

    // Base fade values for daytime - start close for lower sky height
    float dayFadeStart = 200.0;
    float dayFadeEnd = 4450.0;

    // Dusk fade values - slightly reduced from daytime
    float duskFadeStart = 150.0;
    float duskFadeEnd = 2500.0;

    // Night fade values - start close but keep same end distance ratio
    float nightFadeStart = 50.0;    // Start fading almost immediately at night
    float nightFadeEnd = 1200.0;    // Complete fade much sooner

    // Interpolate between day/dusk/night fade values based on explicit timeOfDay
    float fadeStart = dayFadeStart;
    float fadeEnd = dayFadeEnd;
    if (timeOfDay == 1) {  // Dusk
        fadeStart = mix(dayFadeStart, duskFadeStart, effectIntensity);
        fadeEnd = mix(dayFadeEnd, duskFadeEnd, effectIntensity);
    } else if (timeOfDay == 2) {  // Night
        fadeStart = mix(dayFadeStart, nightFadeStart, effectIntensity);
        fadeEnd = mix(dayFadeEnd, nightFadeEnd, effectIntensity);
    }
    float fadeAmount = smoothstep(fadeStart, fadeEnd, distFromCenter);

    // ========== STRETCH CORRECTION ==========
    float radialDist = length(RadialUV);  // 0 at center, 1 at edge

    // Compress UV sampling toward edges to counteract stretching
    // Use two-stage compression: gentle in center, aggressive at edges (60%+)
    float baseCompression = 1.0 - pow(radialDist, 1.5) * 0.4;
    // Extra compression for outer 40% of the plane
    float edgeCompression = smoothstep(0.6, 1.0, radialDist) * 0.4;
    float compressionFactor = baseCompression - edgeCompression;

    vec2 correctedUV = TexCoords0;
    if (radialDist > 0.01) {
        vec2 uvCenter = vec2(17.0);  // Half of max_uv (34.0)
        vec2 uvOffset = TexCoords0 - uvCenter;
        correctedUV = uvCenter + uvOffset * compressionFactor;
    }

    vec4 sC = texture(basic_texture, correctedUV);
    vec3 cloudColor = vec3(sC.b, sC.g, sC.r); // BGR to RGB conversion

    // ========== ATMOSPHERIC CLOUD ENHANCEMENT ==========
    // Calculate cloud luminance (brightness) for effect modulation
    float cloudLuminance = dot(cloudColor, vec3(0.299, 0.587, 0.114));

    // 1. Enhanced contrast - make whites whiter, darks more defined
    // Use S-curve for natural contrast enhancement
    float contrastBoost = smoothstep(0.2, 0.8, cloudLuminance);
    cloudColor = mix(cloudColor, cloudColor * (0.8 + contrastBoost * 0.4), 0.5);

    // 2. Warm highlights on bright cloud areas (sunlit tops)
    // Bright areas get warm golden tint simulating sunlight
    float highlightMask = smoothstep(0.5, 0.9, cloudLuminance);
    vec3 sunlitCloud = cloudColor * SUNLIGHT_COLOR;
    cloudColor = mix(cloudColor, sunlitCloud, highlightMask * 0.3);

    // 3. Cool shadows in darker cloud areas (atmospheric scattering)
    // Darker areas get subtle blue tint from sky light
    float shadowMask = 1.0 - smoothstep(0.3, 0.6, cloudLuminance);
    vec3 shadowedCloud = cloudColor * SHADOW_TINT;
    cloudColor = mix(cloudColor, shadowedCloud, shadowMask * 0.25);

    // 4. Soft glow/bloom on bright areas (simulates light scattering in clouds)
    // This makes white clouds appear more luminous and ethereal
    float glowIntensity = pow(cloudLuminance, 2.0) * 0.15;
    cloudColor += vec3(glowIntensity) * SUNLIGHT_COLOR;

    // 5. Subtle depth variation - clouds toward horizon appear hazier
    float horizonHaze = smoothstep(1500.0, 3500.0, distFromCenter);
    cloudColor = mix(cloudColor, cloudColor * HORIZON_TINT, horizonHaze * 0.2);

    // 6. Very subtle brightness variation for "volumetric" feel
    // Simulates light penetrating cloud layers differently
    float depthVariation = sin(correctedUV.x * 0.5 + correctedUV.y * 0.3) * 0.03;
    cloudColor *= (1.0 + depthVariation);

    // ========== TIME-OF-DAY EFFECTS (DUSK/NIGHT) ==========
    // Apply dusk/night effects to clouds before blending with sky
    if (timeOfDay > 0) {
        bool isDusk = (timeOfDay == 1);
        bool isNight = (timeOfDay == 2);

        // Darken clouds
        float darkening = 1.0;
        if (isDusk) {
            darkening = 1.0 - (effectIntensity * 0.45);  // Up to 45% darker at dusk
        } else if (isNight) {
            darkening = 1.0 - (effectIntensity * 0.75);  // Up to 75% darker at night
        }
        cloudColor *= darkening;

        // Desaturate colors
        float cloudLum = dot(cloudColor, vec3(0.299, 0.587, 0.114));
        float desatAmount = 0.0;
        if (isDusk) {
            desatAmount = effectIntensity * 0.3;
        } else if (isNight) {
            desatAmount = effectIntensity * 0.7;
        }
        cloudColor = mix(cloudColor, vec3(cloudLum), desatAmount);

        // Apply appropriate fog tint
        if (isDusk) {
            // Warm amber tint for dusk clouds
            vec3 duskTint = vec3(1.3, 0.85, 0.55);
            cloudColor = mix(cloudColor, cloudColor * duskTint, effectIntensity * 0.5);
            cloudColor = mix(cloudColor, DUSK_FOG_COLOR, effectIntensity * 0.4);
        } else if (isNight) {
            cloudColor = mix(cloudColor, NIGHT_FOG_COLOR, effectIntensity * 0.6);
        }
    }

    // Apply rain-adjusted sky color for blending
    vec3 adjustedSkyColor = sky_color.rgb;

    // Apply dusk/night effects to sky color as well (using explicit timeOfDay)
    if (timeOfDay > 0) {
        bool isDusk = (timeOfDay == 1);
        bool isNight = (timeOfDay == 2);

        // Darken sky color
        float skyDarkening = 1.0;
        if (isDusk) {
            skyDarkening = 1.0 - (effectIntensity * 0.35);
        } else if (isNight) {
            skyDarkening = 1.0 - (effectIntensity * 0.6);
        }
        adjustedSkyColor *= skyDarkening;

        // Desaturate sky color
        float skyLum = dot(adjustedSkyColor, vec3(0.299, 0.587, 0.114));
        float skyDesatAmount = 0.0;
        if (isDusk) {
            skyDesatAmount = effectIntensity * 0.2;
        } else if (isNight) {
            skyDesatAmount = effectIntensity * 0.5;
        }
        adjustedSkyColor = mix(adjustedSkyColor, vec3(skyLum), skyDesatAmount);

        // Apply color tint
        if (isDusk) {
            vec3 duskTint = vec3(1.3, 0.85, 0.55);
            adjustedSkyColor = mix(adjustedSkyColor, adjustedSkyColor * duskTint, effectIntensity * 0.4);
        } else if (isNight) {
            vec3 nightTint = vec3(0.7, 0.8, 1.0);
            adjustedSkyColor = mix(adjustedSkyColor, adjustedSkyColor * nightTint, effectIntensity * 0.3);
        }
    }

    // Apply rain atmosphere effects
    if (rainIntensity > 0.0) {
        vec3 overcastColor = vec3(0.45, 0.48, 0.52);

        // Darken and desaturate clouds
        float darkening = 1.0 - (rainIntensity * 0.4);
        cloudColor *= darkening;
        cloudColor = mix(cloudColor, overcastColor, rainIntensity * 0.5);

        // Desaturate
        float rainLuminance = dot(cloudColor, vec3(0.299, 0.587, 0.114));
        cloudColor = mix(cloudColor, vec3(rainLuminance), rainIntensity * 0.5);

        // Apply to sky color
        adjustedSkyColor *= darkening;
        adjustedSkyColor = mix(adjustedSkyColor, overcastColor, rainIntensity * 0.6);
        float skyLuminance = dot(adjustedSkyColor, vec3(0.299, 0.587, 0.114));
        adjustedSkyColor = mix(adjustedSkyColor, vec3(skyLuminance), rainIntensity * 0.5);
    }

    // Blend cloud color with sky color based on circular distance
    // Use appropriate fog color based on explicit timeOfDay
    vec3 fadeTargetColor = adjustedSkyColor;
    if (timeOfDay == 1) {
        // Dusk - blend toward warm amber fog
        fadeTargetColor = mix(adjustedSkyColor, DUSK_FOG_COLOR, effectIntensity * 0.6);
    } else if (timeOfDay == 2) {
        // Night - blend toward dark blue fog
        fadeTargetColor = mix(adjustedSkyColor, NIGHT_FOG_COLOR, effectIntensity * 0.8);
    }
    vec3 finalColor = mix(cloudColor, fadeTargetColor, fadeAmount);

    // Final clamp to prevent over-bright areas
    finalColor = clamp(finalColor, 0.0, 1.0);

    outputColor = vec4(finalColor, 1.0);
}
