#version 330 core

in vec2 TexCoords;
in vec2 FragPos;

out vec4 FragColor;

uniform vec2 sunPosition;    // Sun position in NDC (-1 to 1)
uniform vec3 sunColor;       // Sun color
uniform float glareIntensity;
uniform float glareSize;
uniform float viewFactor;    // How directly we're looking at sun (0-1)
uniform float aspectRatio;
uniform float time;

// Create a lens flare artifact at a position along the sun-center axis
float lensFlareArtifact(vec2 pos, vec2 artifactPos, float size, float softness)
{
    float dist = length((pos - artifactPos) * vec2(aspectRatio, 1.0));
    return smoothstep(size, size * softness, dist);
}

// Hexagonal shape for aperture-style lens flare
float hexagon(vec2 pos, float size)
{
    pos = abs(pos * vec2(aspectRatio, 1.0));
    float hex = max(pos.x * 0.866025 + pos.y * 0.5, pos.y);
    return smoothstep(size, size * 0.8, hex);
}

// Circular ring artifact
float ring(vec2 pos, vec2 center, float radius, float thickness)
{
    float dist = length((pos - center) * vec2(aspectRatio, 1.0));
    return smoothstep(thickness, 0.0, abs(dist - radius));
}

void main()
{
    // Direction from screen center to sun
    vec2 sunDir = normalize(sunPosition);

    // Base glare - radial glow around sun
    vec2 toSun = FragPos - sunPosition;
    toSun.x *= aspectRatio;  // Correct for aspect ratio
    float distToSun = length(toSun);

    // Large soft glow
    float glow = exp(-distToSun * 2.0 / glareSize);

    // Streaky rays emanating from sun
    float angle = atan(toSun.y, toSun.x);
    float rays = pow(abs(sin(angle * 6.0 + time * 0.1)), 8.0);
    rays *= exp(-distToSun * 3.0 / glareSize);

    // Anamorphic streak (horizontal)
    float anamorphic = exp(-abs(toSun.y) * 15.0 / glareSize) * exp(-abs(toSun.x) * 2.0 / glareSize);

    // Lens flare artifacts - positioned along the line from sun through center
    // Artifacts appear on opposite side of screen from sun
    vec3 flareColor = vec3(0.0);

    // Multiple flare artifacts at different positions
    // Position along sun-center axis, negative values are opposite side
    float artifact1 = lensFlareArtifact(FragPos, sunPosition * 0.6, 0.08, 0.3);
    float artifact2 = lensFlareArtifact(FragPos, sunPosition * 0.3, 0.05, 0.4);
    float artifact3 = lensFlareArtifact(FragPos, sunPosition * -0.2, 0.12, 0.2);
    float artifact4 = lensFlareArtifact(FragPos, sunPosition * -0.5, 0.06, 0.3);
    float artifact5 = lensFlareArtifact(FragPos, sunPosition * -0.8, 0.15, 0.25);

    // Hexagonal artifact
    float hex1 = hexagon(FragPos - sunPosition * 0.4, 0.06);
    float hex2 = hexagon(FragPos - sunPosition * -0.6, 0.08);

    // Ring artifacts
    float ring1 = ring(FragPos, sunPosition * 0.7, 0.05, 0.015);
    float ring2 = ring(FragPos, sunPosition * -0.3, 0.08, 0.01);

    // Color the artifacts with chromatic aberration
    vec3 artifact1Color = vec3(1.0, 0.8, 0.6) * artifact1 * 0.3;
    vec3 artifact2Color = vec3(0.8, 0.9, 1.0) * artifact2 * 0.25;
    vec3 artifact3Color = vec3(0.6, 0.8, 1.0) * artifact3 * 0.2;
    vec3 artifact4Color = vec3(1.0, 0.7, 0.5) * artifact4 * 0.25;
    vec3 artifact5Color = vec3(0.7, 0.85, 1.0) * artifact5 * 0.15;
    vec3 hex1Color = vec3(0.9, 0.95, 1.0) * hex1 * 0.2;
    vec3 hex2Color = vec3(1.0, 0.9, 0.8) * hex2 * 0.15;
    vec3 ring1Color = vec3(0.8, 0.9, 1.0) * ring1 * 0.3;
    vec3 ring2Color = vec3(1.0, 0.95, 0.9) * ring2 * 0.2;

    // Combine all lens flare artifacts
    flareColor = artifact1Color + artifact2Color + artifact3Color +
                 artifact4Color + artifact5Color +
                 hex1Color + hex2Color + ring1Color + ring2Color;

    // Combine all effects
    vec3 color = sunColor * glow * 0.5;                    // Main glow
    color += sunColor * rays * 0.15;                        // Light rays
    color += sunColor * anamorphic * 0.3;                   // Anamorphic streak
    color += flareColor;                                    // Lens artifacts

    // Apply view factor and intensity
    color *= viewFactor * glareIntensity;

    // Calculate alpha from brightness
    float brightness = (color.r + color.g + color.b) / 3.0;
    float alpha = clamp(brightness, 0.0, 1.0);

    // Discard if too dim
    if (alpha < 0.001) {
        discard;
    }

    FragColor = vec4(color, alpha);
}
