#version 330

in highp vec2 texCoord0;
in highp vec2 texCoord1;
in vec3 surfaceNormal;
in vec3 toLightVector;
in vec2 quadCoord;
in vec2 fogMapCoord;  // For area fog sampling
in float wetness;
in highp vec2 out_textCoord_clouds;
in vec3 FragPos;
in vec4 FragPosLightSpace;
in vec4 FragPosDynamicLightSpace;
in vec2 tileLocalCoord;  // For editor mode tile outlines (0-1 within each tile)

out vec4 outputColor;

uniform sampler2D basic_texture;
uniform sampler2D skyTexture;
uniform float view_distance;
uniform vec4 distanceColor;

// Balanced terrain lighting with strong sun contrast
uniform float ambientStrength = 0.45;  // Lower ambient to preserve shadow depth
uniform float diffuseStrength = 0.60;  // Slightly stronger sun to compensate
uniform sampler2D shadowMap;
uniform sampler2D dynamicShadowMap;
uniform vec3 lightDirection = vec3(-0.4, -0.85, -0.4);  // Moderate sun angle (55-60 deg elevation from southwest)
uniform bool enableShadows = false;
uniform bool enableDynamicShadows = false;
uniform float shadowStrength = 0.22;  // How dark shadows appear (0=invisible, 1=black)
uniform float rainIntensity = 0.0;  // 0.0 = no rain, 1.0 = heavy rain
uniform float visibilityMultiplier = 1.0;  // Time-of-day visibility (1.0 = day, 0.35 = night)
uniform int timeOfDay = 0;  // 0=DAY, 1=DUSK, 2=NIGHT (explicit, not derived from visibility)
uniform bool editorMode = false;  // Editor mode: flat shading + tile outlines

// Area fog uniforms
const int MAX_FOG_ZONES = 16;
uniform int numFogZones = 0;
uniform sampler2D fogMapTexture;
uniform vec3 fogColors[MAX_FOG_ZONES];
uniform float fogLimits[MAX_FOG_ZONES];         // FLimit - max fog cap distance
uniform float fogTransparencies[MAX_FOG_ZONES]; // Transp - controls distance falloff rate
uniform float fogMaxOpacities[MAX_FOG_ZONES];
uniform float fogAltitudes[MAX_FOG_ZONES];  // YBegin - fog ceiling height (world units)

// Terrain dimensions for fog ray marching (from vertex shader)
uniform float terrainWidth;   // Number of tiles in X
uniform float terrainHeight;  // Number of tiles in Z
uniform float tileWidth;      // Size of each tile in world units
uniform float heightmapScale = 4.0;  // 4.0 for C2, 2.0 for C1

// Underwater visibility - uses underwaterStateTexture for per-fragment water level
uniform sampler2D underwaterStateTexture;  // Same texture used in vertex shader for wetness
uniform vec3 cameraPos = vec3(0.0);  // Camera position for view angle calculation

// Blob shadows for characters (up to 32 characters)
const int MAX_BLOB_SHADOWS = 32;
uniform int numBlobShadows = 0;
uniform vec3 blobShadowPositions[MAX_BLOB_SHADOWS];  // XYZ world position
uniform float blobShadowRadii[MAX_BLOB_SHADOWS];     // Shadow radius

// Poisson disk samples for organic-looking soft shadows
const vec2 poissonDisk[16] = vec2[](
    vec2(-0.94201624, -0.39906216),
    vec2(0.94558609, -0.76890725),
    vec2(-0.094184101, -0.92938870),
    vec2(0.34495938, 0.29387760),
    vec2(-0.91588581, 0.45771432),
    vec2(-0.81544232, -0.87912464),
    vec2(-0.38277543, 0.27676845),
    vec2(0.97484398, 0.75648379),
    vec2(0.44323325, -0.97511554),
    vec2(0.53742981, -0.47373420),
    vec2(-0.26496911, -0.41893023),
    vec2(0.79197514, 0.19090188),
    vec2(-0.24188840, 0.99706507),
    vec2(-0.81409955, 0.91437590),
    vec2(0.19984126, 0.78641367),
    vec2(0.14383161, -0.14100790)
);

// Calculate blob shadows from character positions
float BlobShadowCalculation(vec3 fragPos)
{
    float totalShadow = 0.0;

    for (int i = 0; i < numBlobShadows && i < MAX_BLOB_SHADOWS; i++) {
        // Calculate horizontal distance from character to fragment
        vec2 delta = fragPos.xz - blobShadowPositions[i].xz;
        float dist = length(delta);
        float radius = blobShadowRadii[i];

        // Skip if outside shadow radius
        if (dist > radius) continue;

        // Smooth circular falloff - darkest at center, fading to edge
        float normalizedDist = dist / radius;
        float blobShadow = 1.0 - smoothstep(0.0, 1.0, normalizedDist);

        // Additional softening for more natural look
        blobShadow *= blobShadow;  // Square for softer falloff

        // Reduce shadow intensity (not fully black)
        blobShadow *= 0.7;

        // Take maximum shadow (darkest wins)
        totalShadow = max(totalShadow, blobShadow);
    }

    return totalShadow;
}

float ShadowCalculation(vec4 fragPosLightSpace, sampler2D shadowMapSampler, bool isDynamic)
{
    // Perform perspective divide
    vec3 projCoords = fragPosLightSpace.xyz / fragPosLightSpace.w;

    // Transform to [0,1] range
    projCoords = projCoords * 0.5 + 0.5;

    // Check if fragment is outside light's view
    if (projCoords.z > 1.0 || projCoords.x < 0.0 || projCoords.x > 1.0 ||
        projCoords.y < 0.0 || projCoords.y > 1.0) {
        return 0.0;
    }

    // Get depth of current fragment from light's perspective
    float currentDepth = projCoords.z;

    // Calculate bias to prevent shadow acne
    // Dynamic shadows need higher bias due to curved character geometry
    float bias = isDynamic ? 0.003 : 0.0005;

    // Use Poisson disk sampling for softer, more organic shadow edges
    float shadow = 0.0;
    vec2 texelSize = 1.0 / textureSize(shadowMapSampler, 0);

    // Spread radius - larger = softer shadows
    // Dynamic shadows need much larger spread for soft, diffuse appearance
    // Static shadows use smaller radius for sharper edges that show tree trunk detail
    float spreadRadius = isDynamic ? 12.0 : 1.2;

    // Sample using Poisson disk pattern (avoids grid artifacts)
    for (int i = 0; i < 16; i++) {
        vec2 offset = poissonDisk[i] * texelSize * spreadRadius;
        float pcfDepth = texture(shadowMapSampler, projCoords.xy + offset).r;
        shadow += currentDepth - bias > pcfDepth ? 1.0 : 0.0;
    }
    shadow /= 16.0;

    // Soften shadow edges with smooth falloff
    // Dynamic shadows get extra softening for more diffuse look
    if (isDynamic) {
        shadow = smoothstep(0.1, 0.7, shadow);  // Softer gradient
    } else {
        shadow = smoothstep(0.0, 1.0, shadow);
    }

    return shadow;
}

// Camera-centric area fog - when player is inside a fog zone, all visible
// fragments are fogged based on distance and vertical position.
//
// This approach:
// 1. Check if camera is in a fog zone and below the fog ceiling
// 2. If so, apply fog to all fragments based on distance from camera
// 3. Fog fades vertically toward the ceiling
// 4. Near-camera clearing creates a "bubble" around the player
//
// External fog viewing (player outside looking in) is handled by fog volume
// bounding box geometry rendered separately.

const float FOG_ZONE_TRANSITION_RADIUS = 320.0;  // Distance over which fog fades when entering/leaving zones
const float FOG_EDGE_TAPER_RADIUS = 192.0;       // Distance for edge tapering effect

// Helper: sample fog zone at a world position
int sampleFogZone(vec3 worldPos, float terrainWorldWidth, float terrainWorldHeight) {
    vec2 uv = vec2(worldPos.x / terrainWorldWidth, worldPos.z / terrainWorldHeight);
    if (uv.x < 0.0 || uv.x > 1.0 || uv.y < 0.0 || uv.y > 1.0) {
        return -1;
    }
    float fogIndexRaw = texture(fogMapTexture, uv).r * 255.0;
    return int(fogIndexRaw + 0.5) - 1;
}

// Helper: estimate how deep inside a fog zone the camera is (0 = at edge, 1 = deep inside)
// This creates a gradual transition when walking into/out of fog zones
// Uses multi-scale sampling to find approximate distance to nearest zone boundary
float estimateFogZoneDepth(vec3 worldPos, int expectedFogIndex, float terrainWorldWidth, float terrainWorldHeight) {
    vec2 uv = vec2(worldPos.x / terrainWorldWidth, worldPos.z / terrainWorldHeight);

    // Find the minimum distance to a non-zone tile by sampling at multiple radii
    float minNonZoneDistance = FOG_ZONE_TRANSITION_RADIUS;

    // Sample at multiple distances for accurate edge detection
    float radii[5] = float[5](32.0, 80.0, 160.0, 240.0, 320.0);

    for (int r = 0; r < 5; r++) {
        float sampleRadius = radii[r];
        vec2 uvRadius = vec2(sampleRadius / terrainWorldWidth, sampleRadius / terrainWorldHeight);

        // Sample in 12 directions for better coverage
        for (int i = 0; i < 12; i++) {
            float angle = float(i) * 0.5236;  // 30 degree increments (pi/6)
            vec2 offset = vec2(cos(angle), sin(angle)) * uvRadius;
            vec2 sampleUV = uv + offset;

            if (sampleUV.x >= 0.0 && sampleUV.x <= 1.0 && sampleUV.y >= 0.0 && sampleUV.y <= 1.0) {
                float fogIndexRaw = texture(fogMapTexture, sampleUV).r * 255.0;
                int sampleFogIndex = int(fogIndexRaw + 0.5) - 1;

                // Found a different zone - this is an edge
                if (sampleFogIndex != expectedFogIndex) {
                    minNonZoneDistance = min(minNonZoneDistance, sampleRadius);
                }
            } else {
                // Edge of terrain also counts as zone boundary
                minNonZoneDistance = min(minNonZoneDistance, sampleRadius);
            }
        }
    }

    // Convert distance to depth factor (0 at edge, 1 deep inside)
    // Use the transition radius for the full range
    return clamp(minNonZoneDistance / FOG_ZONE_TRANSITION_RADIUS, 0.0, 1.0);
}

// Helper: estimate how deep inside a fog zone a FRAGMENT is (for edge tapering)
// Similar to camera depth but optimized for per-fragment calculation
float estimateFragmentZoneDepth(vec3 fragPos, int fogIndex, float terrainWorldWidth, float terrainWorldHeight) {
    vec2 uv = vec2(fragPos.x / terrainWorldWidth, fragPos.z / terrainWorldHeight);

    // Find minimum distance to non-zone tile
    float minNonZoneDistance = FOG_EDGE_TAPER_RADIUS;

    // Sample at 3 radii for reasonable performance
    float radii[3] = float[3](32.0, 96.0, 192.0);

    for (int r = 0; r < 3; r++) {
        float sampleRadius = radii[r];
        vec2 uvRadius = vec2(sampleRadius / terrainWorldWidth, sampleRadius / terrainWorldHeight);

        // Sample in 8 directions
        for (int i = 0; i < 8; i++) {
            float angle = float(i) * 0.785398;
            vec2 offset = vec2(cos(angle), sin(angle)) * uvRadius;
            vec2 sampleUV = uv + offset;

            if (sampleUV.x >= 0.0 && sampleUV.x <= 1.0 && sampleUV.y >= 0.0 && sampleUV.y <= 1.0) {
                float fogIndexRaw = texture(fogMapTexture, sampleUV).r * 255.0;
                int sampleFogIndex = int(fogIndexRaw + 0.5) - 1;

                if (sampleFogIndex != fogIndex) {
                    minNonZoneDistance = min(minNonZoneDistance, sampleRadius);
                }
            } else {
                minNonZoneDistance = min(minNonZoneDistance, sampleRadius);
            }
        }
    }

    return clamp(minNonZoneDistance / FOG_EDGE_TAPER_RADIUS, 0.0, 1.0);
}

// ========== FOG ZONE TRANSITION EFFECTS ==========
// Calculate transition effects when entering/exiting fog zones
// Returns: transitionIntensity (0-1), fogColor for the transition
// This creates a visible "curtain" effect at fog boundaries

struct FogTransitionInfo {
    float intensity;           // How strong the transition effect is (0 = none, 1 = peak)
    vec3 fogColor;            // Color of the fog zone we're transitioning into/through
    float ambientTint;        // How much to tint ambient light toward fog color
    float brightnessBoost;    // Slight brightness at the "wall" of fog
};

FogTransitionInfo calculateFogZoneTransition(vec3 camPos) {
    FogTransitionInfo info;
    info.intensity = 0.0;
    info.fogColor = vec3(0.7);
    info.ambientTint = 0.0;
    info.brightnessBoost = 0.0;

    if (numFogZones == 0) return info;

    float terrainWorldWidth = terrainWidth * tileWidth;
    float terrainWorldHeight = terrainHeight * tileWidth;

    int cameraFogIndex = sampleFogZone(camPos, terrainWorldWidth, terrainWorldHeight);

    // If camera is in a fog zone, check how close to edge we are
    if (cameraFogIndex >= 0 && cameraFogIndex < numFogZones) {
        float cameraZoneDepth = estimateFogZoneDepth(camPos, cameraFogIndex, terrainWorldWidth, terrainWorldHeight);

        // Transition effect is strongest at the edge (zoneDepth near 0)
        // and fades as we move deeper into the zone
        // Peak effect at ~20% into the zone, then fade out
        float edgeProximity = 1.0 - cameraZoneDepth;

        // Create a "bump" effect - peaks when just entering, fades as we go deeper
        // This gives the feeling of "passing through" a fog curtain
        float transitionCurve = smoothstep(0.0, 0.3, edgeProximity) * smoothstep(1.0, 0.5, edgeProximity);

        info.intensity = transitionCurve;
        info.fogColor = fogColors[cameraFogIndex];

        // Ambient tint increases as we enter the fog zone
        // This shifts the overall scene color toward the fog color
        info.ambientTint = smoothstep(0.0, 0.5, 1.0 - cameraZoneDepth) * 0.15;

        // Slight brightness boost at the "wall" to simulate light scattering
        // at the fog boundary (like seeing bright fog wall before entering)
        info.brightnessBoost = transitionCurve * 0.12;
    } else {
        // Camera is OUTSIDE fog zones - check if we're near one
        // Sample nearby to detect approaching fog zones
        float sampleRadius = FOG_ZONE_TRANSITION_RADIUS * 0.5;  // Check half the transition radius
        int nearestFogIndex = -1;
        float closestDist = sampleRadius;

        // Sample in cardinal directions to find nearby fog
        for (float angle = 0.0; angle < 6.28; angle += 0.785) {
            for (float dist = 0.25; dist <= 1.0; dist += 0.25) {
                vec3 samplePos = camPos + vec3(cos(angle), 0.0, sin(angle)) * dist * sampleRadius;
                int sampleIndex = sampleFogZone(samplePos, terrainWorldWidth, terrainWorldHeight);
                if (sampleIndex >= 0 && sampleIndex < numFogZones) {
                    float actualDist = dist * sampleRadius;
                    if (actualDist < closestDist) {
                        closestDist = actualDist;
                        nearestFogIndex = sampleIndex;
                    }
                }
            }
        }

        if (nearestFogIndex >= 0) {
            // We're approaching a fog zone from outside
            // Create subtle anticipation effect
            float approachFactor = 1.0 - (closestDist / sampleRadius);
            info.intensity = approachFactor * 0.3;  // Subtle effect when approaching
            info.fogColor = fogColors[nearestFogIndex];
            info.ambientTint = approachFactor * 0.08;  // Very subtle tint as we approach
            info.brightnessBoost = approachFactor * 0.05;
        }
    }

    return info;
}

// Calculate fog density at a fragment without applying the color blend
// Returns fog density (0-1) and fog color via out parameters
// cloudTexCoord: UV for sampling moving cloud texture for fog wisps
float calculateAreaFogDensity(float distFromCamera, vec3 fragWorldPos, out vec3 outFogColor, vec2 cloudTexCoord) {
    outFogColor = vec3(0.7);  // Default fog color
    if (numFogZones == 0) return 0.0;

    float terrainWorldWidth = terrainWidth * tileWidth;
    float terrainWorldHeight = terrainHeight * tileWidth;

    int cameraFogIndex = sampleFogZone(cameraPos, terrainWorldWidth, terrainWorldHeight);
    if (cameraFogIndex < 0 || cameraFogIndex >= numFogZones) return 0.0;

    float fogCeiling = fogAltitudes[cameraFogIndex];
    float transp = fogTransparencies[cameraFogIndex];
    float fogLimit = fogLimits[cameraFogIndex];
    outFogColor = fogColors[cameraFogIndex];

    float depthA = max(0.0, fogCeiling - cameraPos.y);
    float depthB = max(0.0, fogCeiling - fragWorldPos.y);

    if (depthA <= 0.0 && depthB <= 0.0) return 0.0;

    // Smooth vertical transitions
    float ceilingTransitionHeight = 96.0;
    float cameraVerticalFade = 1.0;
    if (depthA <= 0.0) {
        float aboveAmount = cameraPos.y - fogCeiling;
        cameraVerticalFade = clamp(1.0 - aboveAmount / ceilingTransitionHeight, 0.0, 1.0);
        if (cameraVerticalFade <= 0.0) return 0.0;
    } else if (depthA < ceilingTransitionHeight) {
        cameraVerticalFade = smoothstep(0.0, ceilingTransitionHeight, depthA);
    }

    // Camera zone edge transition (0 = at edge, 1 = deep inside)
    float cameraZoneDepth = estimateFogZoneDepth(cameraPos, cameraFogIndex, terrainWorldWidth, terrainWorldHeight);
    float cameraTransitionFactor = smoothstep(0.0, 0.5, cameraZoneDepth);

    // Base fog calculation
    const float MAX_FOG_DISTANCE = 480.0;
    float safeTransp = max(transp, 1.0);
    float distanceFactor = min(distFromCamera / MAX_FOG_DISTANCE, 1.0);
    float transpFactor = safeTransp / 600.0;
    float depthFactor = min((depthA + depthB) / 200.0, 2.0);
    float fogValue = (distanceFactor * depthFactor * 255.0) / transpFactor;
    fogValue = min(fogValue, fogLimit);
    float fogDensity = fogValue / 255.0;

    // Edge entry transition - blend from thin edge fog to full interior fog
    float edgeOnlyFog = distanceFactor * (fogLimit / 255.0) * 0.6;
    fogDensity = mix(edgeOnlyFog, fogDensity, cameraTransitionFactor);
    fogDensity *= cameraVerticalFade;

    // ========== FOG WISPS / CLOUDINESS ==========
    vec2 wispUV1 = cloudTexCoord * 0.8;
    vec2 wispUV2 = cloudTexCoord * 2.5 + vec2(0.3, 0.7);
    vec2 wispUV3 = cloudTexCoord * 6.0 + vec2(0.5, 0.2);

    float wisp1 = texture(skyTexture, wispUV1).r;
    float wisp2 = texture(skyTexture, wispUV2).r;
    float wisp3 = texture(skyTexture, wispUV3).r;

    float wispNoise = wisp1 * 0.5 + wisp2 * 0.35 + wisp3 * 0.15;
    float wispModulation = 0.4 + wispNoise * 1.2;
    float wispStrength = 0.6 + distanceFactor * 0.4;
    fogDensity *= mix(1.0, wispModulation, wispStrength);

    // ========== FRAGMENT EDGE TAPERING ==========
    int fragFogIndex = sampleFogZone(fragWorldPos, terrainWorldWidth, terrainWorldHeight);

    if (fragFogIndex == cameraFogIndex) {
        // Fragment in same zone - apply edge tapering
        float fragZoneDepth = estimateFragmentZoneDepth(fragWorldPos, cameraFogIndex, terrainWorldWidth, terrainWorldHeight);
        float fragEdgeTaper = smoothstep(0.0, 0.6, fragZoneDepth);
        fogDensity *= fragEdgeTaper;
    } else {
        // Fragment in different/no zone - fade based on proximity to camera's zone
        vec2 fragUV = vec2(fragWorldPos.x / terrainWorldWidth, fragWorldPos.z / terrainWorldHeight);
        if (fragUV.x >= 0.0 && fragUV.x <= 1.0 && fragUV.y >= 0.0 && fragUV.y <= 1.0) {
            vec2 uvSampleRadius = vec2(96.0 / terrainWorldWidth, 96.0 / terrainWorldHeight);
            float totalWeight = 0.0;
            float sameZoneWeight = 0.0;

            for (float dx = -2.0; dx <= 2.0; dx += 1.0) {
                for (float dy = -2.0; dy <= 2.0; dy += 1.0) {
                    float dist = length(vec2(dx, dy));
                    if (dist > 2.5) continue;

                    float weight = 1.0 - dist * 0.3;
                    vec2 neighborCoord = clamp(fragUV + vec2(dx, dy) * uvSampleRadius * 0.5, vec2(0.0), vec2(1.0));
                    float neighborIndex = texture(fogMapTexture, neighborCoord).r * 255.0;
                    int neighborFogIndex = int(neighborIndex + 0.5) - 1;

                    totalWeight += weight;
                    if (neighborFogIndex == cameraFogIndex) {
                        sameZoneWeight += weight;
                    }
                }
            }

            float edgeFade = smoothstep(0.0, 0.5, sameZoneWeight / max(totalWeight, 0.001));
            fogDensity *= edgeFade;
        }
    }

    return clamp(fogDensity, 0.0, 1.0);
}

vec3 applyAreaFog(vec3 color, float distFromCamera, vec3 fragWorldPos) {
    if (numFogZones == 0) return color;

    // Terrain dimensions for converting world coords to fog map UV
    float terrainWorldWidth = terrainWidth * tileWidth;
    float terrainWorldHeight = terrainHeight * tileWidth;

    // Check if CAMERA is in a fog zone
    int cameraFogIndex = sampleFogZone(cameraPos, terrainWorldWidth, terrainWorldHeight);

    // If camera is not in any fog zone, no fog effect (external view handled by fog planes)
    if (cameraFogIndex < 0 || cameraFogIndex >= numFogZones) return color;

    // Get fog parameters for the zone the camera is in
    float fogCeiling = fogAltitudes[cameraFogIndex];
    float transp = fogTransparencies[cameraFogIndex];
    float fogLimit = fogLimits[cameraFogIndex];
    vec3 fogColor = fogColors[cameraFogIndex];

    // ========== DEPTH CALCULATIONS ==========
    // depthA = how deep the camera is below the fog ceiling
    // depthB = how deep the fragment is below the fog ceiling
    float depthA = max(0.0, fogCeiling - cameraPos.y);
    float depthB = max(0.0, fogCeiling - fragWorldPos.y);

    // If both camera and fragment are above the fog ceiling, no fog
    if (depthA <= 0.0 && depthB <= 0.0) return color;

    // ========== SMOOTH VERTICAL TRANSITIONS ==========
    float ceilingTransitionHeight = 96.0;

    float cameraVerticalFade = 1.0;
    if (depthA <= 0.0) {
        float aboveAmount = cameraPos.y - fogCeiling;
        cameraVerticalFade = clamp(1.0 - aboveAmount / ceilingTransitionHeight, 0.0, 1.0);
        if (cameraVerticalFade <= 0.0) return color;
    } else if (depthA < ceilingTransitionHeight) {
        cameraVerticalFade = smoothstep(0.0, ceilingTransitionHeight, depthA);
    }

    // ========== CAMERA ZONE EDGE TRANSITION ==========
    // How deep inside the fog zone is the camera? (0 = at edge, 1 = deep inside)
    float cameraZoneDepth = estimateFogZoneDepth(cameraPos, cameraFogIndex, terrainWorldWidth, terrainWorldHeight);

    // When camera is at edge, use a gradual ramp-up curve for smooth entry
    // smoothstep(0, 0.5, depth) means fog reaches full strength at 50% into zone
    float cameraTransitionFactor = smoothstep(0.0, 0.5, cameraZoneDepth);

    // ========== FOG DENSITY CALCULATION ==========
    const float MAX_FOG_DISTANCE = 480.0;  // 30 tiles

    float safeTransp = max(transp, 1.0);
    float distanceFactor = min(distFromCamera / MAX_FOG_DISTANCE, 1.0);
    float transpFactor = safeTransp / 600.0;
    float depthFactor = min((depthA + depthB) / 200.0, 2.0);

    // Base fog calculation
    float fogValue = (distanceFactor * depthFactor * 255.0) / transpFactor;
    fogValue = min(fogValue, fogLimit);
    float fogDensity = fogValue / 255.0;

    // ========== EDGE ENTRY TRANSITION ==========
    // When entering zone from outside, blend from "thin edge fog" to "full interior fog"
    // Edge fog: light distance-based fog that matches external fog plane appearance
    // Interior fog: full Carnivores fog formula
    float edgeOnlyFog = distanceFactor * (fogLimit / 255.0) * 0.6;
    fogDensity = mix(edgeOnlyFog, fogDensity, cameraTransitionFactor);

    // Apply vertical fade
    fogDensity *= cameraVerticalFade;

    // ========== FRAGMENT EDGE TAPERING ==========
    // Fog should also taper at the edges of the zone for fragments near zone boundaries
    // This makes the fog boundary appear soft rather than hard-edged
    int fragFogIndex = sampleFogZone(fragWorldPos, terrainWorldWidth, terrainWorldHeight);

    if (fragFogIndex == cameraFogIndex) {
        // Fragment is in the same zone as camera - apply edge tapering based on fragment position
        float fragZoneDepth = estimateFragmentZoneDepth(fragWorldPos, cameraFogIndex, terrainWorldWidth, terrainWorldHeight);
        float fragEdgeTaper = smoothstep(0.0, 0.6, fragZoneDepth);
        fogDensity *= fragEdgeTaper;
    } else if (fragFogIndex >= 0) {
        // Fragment is in a different fog zone - fade fog based on proximity to camera's zone
        // Sample around fragment to find how much of camera's zone is nearby
        vec2 fragUV = vec2(fragWorldPos.x / terrainWorldWidth, fragWorldPos.z / terrainWorldHeight);
        vec2 uvSampleRadius = vec2(128.0 / terrainWorldWidth, 128.0 / terrainWorldHeight);
        float totalWeight = 0.0;
        float sameZoneWeight = 0.0;

        for (float dx = -2.0; dx <= 2.0; dx += 1.0) {
            for (float dy = -2.0; dy <= 2.0; dy += 1.0) {
                vec2 offset = vec2(dx, dy) / 2.0;
                float sampleDist = length(offset);
                if (sampleDist > 1.0) continue;

                float weight = 1.0 - sampleDist * 0.4;
                vec2 neighborCoord = clamp(fragUV + offset * uvSampleRadius, vec2(0.0), vec2(1.0));
                float neighborIndex = texture(fogMapTexture, neighborCoord).r * 255.0;
                int neighborFogIndex = int(neighborIndex + 0.5) - 1;

                totalWeight += weight;
                if (neighborFogIndex == cameraFogIndex) {
                    sameZoneWeight += weight;
                }
            }
        }

        float edgeFade = smoothstep(0.0, 0.6, sameZoneWeight / max(totalWeight, 0.001));
        fogDensity *= edgeFade;
    } else {
        // Fragment is outside any fog zone - strong fade
        vec2 fragUV = vec2(fragWorldPos.x / terrainWorldWidth, fragWorldPos.z / terrainWorldHeight);
        if (fragUV.x >= 0.0 && fragUV.x <= 1.0 && fragUV.y >= 0.0 && fragUV.y <= 1.0) {
            vec2 uvSampleRadius = vec2(96.0 / terrainWorldWidth, 96.0 / terrainWorldHeight);
            float sameZoneWeight = 0.0;
            float totalWeight = 0.0;

            for (float dx = -2.0; dx <= 2.0; dx += 1.0) {
                for (float dy = -2.0; dy <= 2.0; dy += 1.0) {
                    float dist = length(vec2(dx, dy));
                    if (dist > 2.5) continue;

                    float weight = 1.0 - dist * 0.3;
                    vec2 neighborCoord = clamp(fragUV + vec2(dx, dy) * uvSampleRadius * 0.5, vec2(0.0), vec2(1.0));
                    float neighborIndex = texture(fogMapTexture, neighborCoord).r * 255.0;
                    int neighborFogIndex = int(neighborIndex + 0.5) - 1;

                    totalWeight += weight;
                    if (neighborFogIndex == cameraFogIndex) {
                        sameZoneWeight += weight;
                    }
                }
            }

            float edgeFade = smoothstep(0.0, 0.5, sameZoneWeight / max(totalWeight, 0.001));
            fogDensity *= edgeFade;
        }
    }

    // Clamp final density
    fogDensity = clamp(fogDensity, 0.0, 1.0);

    return mix(color, fogColor, fogDensity);
}

void main()
{
    vec4 sC;

    // Determine which texture coordinates to use for the current face
    // The atlas has padding to prevent bleeding, but mipmaps can still cause issues at distance
    vec2 uv;
    if (mod(gl_PrimitiveID, 2) == 0) {
        uv = texCoord1;
    } else {
        uv = texCoord0;
    }

    // Calculate the appropriate mipmap level manually to prevent bleeding from adjacent tiles
    // The atlas tiles are 128x128 pixels with 32px padding (total 192x192 per tile in atlas)
    // At lower mipmap levels, the padding becomes insufficient, causing visible seams
    vec2 dx = dFdx(uv * textureSize(basic_texture, 0));
    vec2 dy = dFdy(uv * textureSize(basic_texture, 0));
    float delta_max_sqr = max(dot(dx, dx), dot(dy, dy));
    float mipLevel = 0.5 * log2(delta_max_sqr);

    // Clamp to prevent using the lowest miplevels where padding is too small
    // With 32px padding on 192px tiles, we can safely use up to mip level 4-5
    // (at mip 5: 192px -> 6px, padding becomes <1px which causes bleeding)
    mipLevel = clamp(mipLevel, 0.0, 4.0);

    sC = textureLod(basic_texture, uv, mipLevel);

    // ========== EDITOR MODE - FLAT SHADING + TILE OUTLINES ==========
    if (editorMode) {
        // Convert texture from BGR to RGB format
        vec3 textureColor = vec3(sC.b, sC.g, sC.r);

        // Simple flat shading - just texture color with slight ambient boost for visibility
        vec3 editorColor = textureColor * 1.1;

        // Calculate tile outline using world position
        // Tiles are on a grid with spacing = tileWidth, centered at tileWidth/2 offsets
        // We want to draw lines at tile boundaries (integer multiples of tileWidth)
        vec2 worldPosXZ = FragPos.xz;
        vec2 tilePos = worldPosXZ / tileWidth;  // Position in tile units
        vec2 tileFrac = fract(tilePos);         // Fractional position within tile (0-1)

        // Use screen-space derivatives for consistent line width at any distance
        vec2 dTilePosX = dFdx(tilePos);
        vec2 dTilePosY = dFdy(tilePos);
        float pixelScale = max(length(dTilePosX), length(dTilePosY));

        // Adaptive line width - thicker when far away (in screen space)
        float lineWidth = max(0.03, pixelScale * 1.5);

        // Distance from nearest tile edge (0 = on edge, 0.5 = center of tile)
        float edgeDistX = min(tileFrac.x, 1.0 - tileFrac.x);
        float edgeDistZ = min(tileFrac.y, 1.0 - tileFrac.y);
        float minEdgeDist = min(edgeDistX, edgeDistZ);

        // Create anti-aliased line
        float outlineStrength = 1.0 - smoothstep(0.0, lineWidth, minEdgeDist);

        // Outline color - dark gray for visibility on any terrain
        vec3 outlineColor = vec3(0.1, 0.1, 0.1);

        // Blend outline with terrain color
        editorColor = mix(editorColor, outlineColor, outlineStrength * 0.85);

        // Apply simple distance fade to prevent harsh clipping at view distance
        float distance = length(FragPos - cameraPos);
        float fadeStart = view_distance * 0.85;
        float fadeEnd = view_distance * 0.98;
        float fadeFactor = smoothstep(fadeStart, fadeEnd, distance);
        vec3 skyColor = distanceColor.rgb;
        editorColor = mix(editorColor, skyColor, fadeFactor);

        outputColor = vec4(editorColor, 1.0);
        return;
    }

    // ========== ENHANCED ATMOSPHERIC DEPTH EFFECTS ==========
    // Creates illusion of vast distances through multi-layered fog and aerial perspective

    // Use radial (spherical) distance from camera for natural circular fog falloff
    float distance = length(FragPos - cameraPos);

    // Exponential fog (more realistic than linear - rapid increase then plateaus)
    float min_distance = view_distance * 0.50; // Start fog earlier for gradual buildup
    float max_distance = view_distance * 0.95; // End before clip plane

    // Exponential fog formula: creates natural-looking atmospheric falloff
    float fogDensity = 0.00012; // Adjusted for world-space distance
    float fogFactor = 1.0 - exp(-distance * fogDensity);
    fogFactor = clamp(fogFactor, 0.0, 0.65); // Allow denser fog at extreme distances

    // Height-based atmospheric scattering (valleys get more haze than peaks)
    // Lower terrain accumulates more atmospheric particles, creating depth
    float heightFactor = clamp((200.0 - FragPos.y) / 300.0, 0.0, 0.4);
    fogFactor += heightFactor * 0.15; // Subtle valley haze

    // Distance-based desaturation (distant objects lose color intensity)
    // Simulates light scattering through atmosphere
    float desaturationFactor = clamp(distance / (view_distance * 0.8), 0.0, 0.5);

    // Aerial perspective: distant terrain shifts toward atmospheric blue
    // At night, use dark night fog color instead of daytime haze
    vec3 aerialColor = vec3(0.6, 0.75, 0.9); // Slightly blue-tinted haze
    if (timeOfDay > 0) {
        float effectIntensity = 1.0 - visibilityMultiplier;
        vec3 duskAerialColor = vec3(0.25, 0.12, 0.06);  // Warm amber
        vec3 nightAerialColor = vec3(0.02, 0.03, 0.06); // Dark blue
        if (timeOfDay == 1) {
            aerialColor = mix(aerialColor, duskAerialColor, effectIntensity);
        } else if (timeOfDay == 2) {
            aerialColor = mix(aerialColor, nightAerialColor, effectIntensity);
        }
    }
    float aerialBlend = clamp(distance / (view_distance * 1.1), 0.0, 0.4);

    // Real-time lighting (same as world objects)
    vec3 norm = normalize(surfaceNormal);
    vec3 lightDir = normalize(-lightDirection); // Use the world object light direction
    
    // Convert texture from BGR to RGB format
    vec3 textureColor = vec3(sC.b, sC.g, sC.r);
    
    // Ambient lighting
    vec3 ambient = ambientStrength * textureColor;
    
    // Diffuse lighting
    float diff = max(dot(norm, lightDir), 0.0);
    vec3 diffuse = diffuseStrength * diff * textureColor;

    // Base lighting
    vec3 baseColor = ambient + diffuse;

    // ========== FOG ZONE TRANSITION EFFECTS ==========
    // Calculate transition effects for entering/exiting fog zones
    // This creates a visual "curtain" effect and ambient color shift
    FogTransitionInfo fogTransition = calculateFogZoneTransition(cameraPos);

    // Apply ambient tint toward fog color when near/in fog zone
    // This helps prepare the eye for the fog and makes transition feel natural
    if (fogTransition.ambientTint > 0.0) {
        baseColor = mix(baseColor, baseColor * fogTransition.fogColor * 1.5, fogTransition.ambientTint);
    }

    // ========== PRE-CALCULATE FOG DENSITY FOR CLOUD SHADOW MODULATION ==========
    // Calculate fog density early so we can use it to soften cloud shadows
    // Fog diffuses light, reducing the contrast between sunny and shadowed areas
    float worldDistance = length(FragPos - cameraPos);
    vec3 areaFogColor;
    // Pass cloud texture coords for animated fog wisps effect
    float areaFogDensity = calculateAreaFogDensity(worldDistance, FragPos, areaFogColor, out_textCoord_clouds);

    // Add cloud shadows for dynamic atmosphere
    vec4 cloudColor = texture(skyTexture, out_textCoord_clouds);
    float cloudLuminance = (0.299 * cloudColor.b + 0.587 * cloudColor.g + 0.114 * cloudColor.r);

    // Strong cloud contrast for clearly visible moving shadows
    float cloudBrightness = clamp((cloudLuminance - 0.2) * 3.0, 0.0, 1.0); // Very strong contrast boost

    // Cloud effect: bright sky = 10% brighter (1.10), dark clouds = 75% darker (0.25)
    // This creates sun reflection in clear areas and very dark shadows under clouds
    // BUT: fog diffuses light, reducing this contrast
    // As fog density increases, cloud shadows become less pronounced
    float fogShadowReduction = 1.0 - areaFogDensity * 0.8;  // At full fog, reduce contrast by 80%
    float cloudDarkening = 0.25;
    float cloudRange = 0.85;

    // Reduce the cloud shadow contrast when in fog
    // Move the dark value toward neutral (less dark) and reduce the bright range
    float foggedCloudDarkening = mix(cloudDarkening, 0.7, areaFogDensity);  // Less dark when fogged
    float foggedCloudRange = cloudRange * fogShadowReduction;  // Smaller range when fogged

    float cloudModulation = foggedCloudDarkening + (cloudBrightness * foggedCloudRange);

    vec3 finalColor = baseColor * cloudModulation;

    // Apply shadows if enabled
    float combinedShadow = 0.0;

    if (enableShadows) {
        combinedShadow = ShadowCalculation(FragPosLightSpace, shadowMap, false);
    }

    // Apply blob shadows for characters (replaces dynamic shadow map)
    if (numBlobShadows > 0) {
        float blobShadow = BlobShadowCalculation(FragPos);
        combinedShadow = max(combinedShadow, blobShadow);
    }

    if (enableShadows || numBlobShadows > 0) {
        // Apply shadow to the final color - reduce brightness where shadows are cast
        finalColor = finalColor * (1.0 - combinedShadow * shadowStrength);
    }

    // ========== ATMOSPHERIC BLENDING ==========
    // Apply multiple atmospheric effects to create sense of vast distance

    // 1. Desaturate distant terrain (colors fade toward gray)
    vec3 desaturatedColor = mix(finalColor, vec3(dot(finalColor, vec3(0.299, 0.587, 0.114))), desaturationFactor);

    // 2. Shift toward aerial perspective (blue-tinted atmospheric haze)
    vec3 aerialBlended = mix(desaturatedColor, aerialColor * desaturatedColor, aerialBlend);

    // 3. Blend with horizon fog color for maximum distance
    // At dusk/night, use appropriate fog color instead of daytime distance color
    vec3 fogTargetColor = distanceColor.rgb;
    if (timeOfDay > 0) {
        float effectIntensity = 1.0 - visibilityMultiplier;
        vec3 duskFogColor = vec3(0.25, 0.12, 0.06);  // Warm amber
        vec3 nightFogColor = vec3(0.02, 0.03, 0.06); // Dark blue
        if (timeOfDay == 1) {
            fogTargetColor = mix(distanceColor.rgb, duskFogColor, effectIntensity);
        } else if (timeOfDay == 2) {
            fogTargetColor = mix(distanceColor.rgb, nightFogColor, effectIntensity);
        }
    }
    finalColor = mix(aerialBlended, fogTargetColor, fogFactor);

    // 4. Subtle contrast reduction at distance (atmospheric diffusion)
    // Distant terrain has less contrast, creating softer, hazier appearance
    if (distance > min_distance) {
        float contrastReduction = clamp((distance - min_distance) / (max_distance - min_distance), 0.0, 0.3);
        // At dusk/night, use appropriate fog color instead of gray for contrast reduction
        vec3 contrastTarget = vec3(0.5);
        if (timeOfDay > 0) {
            float effectIntensity = 1.0 - visibilityMultiplier;
            vec3 duskContrastTarget = vec3(0.25, 0.12, 0.06);
            vec3 nightContrastTarget = vec3(0.02, 0.03, 0.06);
            if (timeOfDay == 1) {
                contrastTarget = mix(vec3(0.5), duskContrastTarget, effectIntensity);
            } else if (timeOfDay == 2) {
                contrastTarget = mix(vec3(0.5), nightContrastTarget, effectIntensity);
            }
        }
        finalColor = mix(finalColor, contrastTarget, contrastReduction * 0.2);
    }

    // ========== AREA FOG ==========
    // Apply localized fog zones AFTER distance-based atmospheric effects
    // Skip area fog at night - night depth fog handles visibility instead
    if (timeOfDay != 2) {
        // This ensures area fog is visible and not washed out by distance fog
        // Use the pre-calculated fog density and color from earlier (for cloud shadow modulation)
        finalColor = mix(finalColor, areaFogColor, areaFogDensity);

        // ========== FOG ZONE TRANSITION CURTAIN EFFECT ==========
        // Apply the "curtain" effect when crossing fog zone boundaries
        // This creates a visible brightness/fog pulse as you enter/exit
        if (fogTransition.intensity > 0.0) {
            // Brightness boost at the transition boundary (light scattering at fog wall)
            finalColor *= (1.0 + fogTransition.brightnessBoost);

            // Add a subtle fog overlay during transition (like passing through a thin fog wall)
            // This is distance-independent - affects everything equally during transition
            float curtainFog = fogTransition.intensity * 0.25;  // Up to 25% fog overlay at peak
            finalColor = mix(finalColor, fogTransition.fogColor, curtainFog);
        }
    }

    // Apply shoreline wetness effect
    // Terrain near/under water appears darker and more saturated (wet)
    if (wetness > 0.0) {
        // Darken the terrain to simulate wetness (wet surfaces reflect less diffuse light)
        float darkening = mix(1.0, 0.65, wetness); // Up to 35% darker when fully wet

        // Increase saturation slightly (wet surfaces have richer colors)
        vec3 grayscale = vec3(dot(finalColor, vec3(0.299, 0.587, 0.114)));
        float saturation = mix(1.0, 1.2, wetness * 0.5); // Subtle saturation boost
        vec3 saturatedColor = mix(grayscale, finalColor, saturation);

        // Apply both effects
        finalColor = saturatedColor * darkening;

        // Add subtle specular highlight for water reflectivity on wet terrain
        float wetSpecular = pow(max(dot(norm, lightDir), 0.0), 32.0) * wetness * 0.15;
        finalColor += vec3(wetSpecular);
    }

    // ========== RAIN ATMOSPHERE EFFECTS ==========
    // Darken scene and add wetness when raining
    if (rainIntensity > 0.0) {
        // 1. Darken the scene (overcast sky reduces ambient light)
        float rainDarkening = 1.0 - (rainIntensity * 0.35); // Up to 35% darker
        finalColor *= rainDarkening;

        // 2. Apply blue-gray tint (rain atmosphere)
        vec3 rainTint = vec3(0.85, 0.88, 1.0); // Subtle blue-gray
        finalColor = mix(finalColor, finalColor * rainTint, rainIntensity * 0.4);

        // 3. Add wetness effect to all surfaces (rain makes everything wet)
        float rainWetness = rainIntensity * 0.9; // Strong wetness from rain
        float totalWetness = max(wetness, rainWetness);

        // Wet surface darkening - wet surfaces absorb more light
        float wetDarkening = 1.0 - (totalWetness * 0.25); // Up to 25% darker when fully wet
        finalColor *= wetDarkening;

        // Increase color saturation when wet (wet surfaces have richer colors)
        vec3 grayscale = vec3(dot(finalColor, vec3(0.299, 0.587, 0.114)));
        finalColor = mix(grayscale, finalColor, 1.0 + totalWetness * 0.3); // More saturated

        // Strong specular highlights (shiny wet surfaces reflecting light)
        // Primary sharp specular for direct sun reflection
        float wetSpecular = pow(max(dot(norm, lightDir), 0.0), 128.0) * totalWetness * 0.5;
        // Secondary broader specular for general wet sheen
        float wetSheen = pow(max(dot(norm, lightDir), 0.0), 16.0) * totalWetness * 0.2;
        finalColor += vec3(wetSpecular + wetSheen) * rainTint;

        // 4. Reduced visibility during rain - distance-based fog intensifies
        // Rain makes distant objects harder to see (mist/rain curtain effect)
        float rainVisibilityReduction = rainIntensity * 0.4; // Up to 40% more fog
        float rainFogFactor = fogFactor + (1.0 - fogFactor) * rainVisibilityReduction * (distance / max_distance);
        rainFogFactor = clamp(rainFogFactor, 0.0, 0.75); // Cap at 75% fog
        finalColor = mix(finalColor, distanceColor.rgb * rainTint, rainFogFactor - fogFactor);
    }

    // ========== UNDERWATER TINTING ==========
    // Fade terrain toward water color when submerged
    // Sample actual water level at this fragment's position from the underwater state texture
    float localWaterLevel = texture(underwaterStateTexture, quadCoord).r;

    // Only apply tinting if there is water here (water level > 0) and fragment is below it
    if (localWaterLevel > 0.0 && FragPos.y < localWaterLevel) {
        // Calculate depth below water surface
        float underwaterDepth = localWaterLevel - FragPos.y;

        // Calculate view angle factor - looking through more water at shallow angles
        vec3 viewDir = normalize(cameraPos - FragPos);
        float viewDot = abs(viewDir.y);  // 1.0 looking straight down, 0.0 horizontal
        float opticalDepth = underwaterDepth / max(viewDot, 0.15);

        // Normalize to 2 tiles (32 units) - extended range for deeper underwater visibility
        float underwaterFade = clamp(opticalDepth / 32.0, 0.0, 1.0);

        // Water tint color - blue-green tint that matches water
        vec3 waterTint = vec3(0.2, 0.35, 0.45);

        // Fade toward water tint based on optical depth
        // Also darken underwater areas (less light penetration)
        float darkenFactor = 1.0 - underwaterFade * 0.5;  // Up to 50% darker at full depth
        finalColor = mix(finalColor * darkenFactor, waterTint, underwaterFade * 0.7);
    }

    // ========== TIME-OF-DAY EFFECTS (DAWN/DUSK/NIGHT) ==========
    // Use explicit timeOfDay uniform (0=DAY, 1=DUSK, 2=NIGHT) instead of deriving from visibility
    // visibilityMultiplier still controls intensity of effects
    if (timeOfDay > 0) {
        float effectIntensity = 1.0 - visibilityMultiplier;  // How strong the effect should be

        bool isDusk = (timeOfDay == 1);
        bool isNight = (timeOfDay == 2);

        // 1. Darkening - dusk is brighter than night
        float darkening = 1.0;
        if (isDusk) {
            darkening = 1.0 - (effectIntensity * 0.50);
        } else if (isNight) {
            darkening = 1.0 - (effectIntensity * 0.90);
        }
        finalColor *= darkening;

        // 2. Desaturation - dusk keeps more color than night
        vec3 grayscale = vec3(dot(finalColor, vec3(0.299, 0.587, 0.114)));
        float desatAmount = 0.0;
        if (isDusk) {
            desatAmount = effectIntensity * 0.30;
        } else if (isNight) {
            desatAmount = effectIntensity * 0.85;
        }
        finalColor = mix(finalColor, grayscale, desatAmount);

        // 3. Color tinting - warm for dusk, cool for night (mutually exclusive)
        vec3 duskTint = vec3(1.3, 0.85, 0.55);   // Warm orange/golden sunset
        vec3 nightTint = vec3(0.4, 0.5, 0.8);    // Cool blue moonlight

        if (isDusk) {
            finalColor = mix(finalColor, finalColor * duskTint, effectIntensity * 0.7);
        } else if (isNight) {
            finalColor = mix(finalColor, finalColor * nightTint, effectIntensity * 0.6);
        }

        // 4. Depth fog - dusk has longer visibility than night
        if (isDusk) {
            float duskFogStart = 200.0;
            float duskFogEnd = 800.0;
            float duskFogAmount = smoothstep(duskFogStart, duskFogEnd, distance) * effectIntensity;
            vec3 duskFogColor = vec3(0.25, 0.12, 0.06);  // Dark warm amber
            finalColor = mix(finalColor, duskFogColor, duskFogAmount * 0.8);
        } else if (isNight) {
            float nightFogStart = 150.0;
            float nightFogEnd = 480.0;
            float nightFogAmount = smoothstep(nightFogStart, nightFogEnd, distance) * effectIntensity;
            vec3 nightFogColor = vec3(0.02, 0.03, 0.06); // Very dark blue-black
            finalColor = mix(finalColor, nightFogColor, nightFogAmount);
        }
    }

    // ========== HARD EDGE FADE ==========
    // Ensure geometry fully fades to sky color before reaching the clip plane
    // Uses radial distance for circular fade (distance already calculated above)
    float fadeStart = view_distance * 0.70;  // Start hard fade at 70% of view distance
    float fadeEnd = view_distance * 0.95;    // Fully faded by 95% of view distance
    float hardFade = clamp((distance - fadeStart) / (fadeEnd - fadeStart), 0.0, 1.0);
    // Smooth the fade curve for more gradual transition
    hardFade = smoothstep(0.0, 1.0, hardFade);
    // At dusk/night, fade to appropriate color instead of daytime distance color
    vec3 fadeTargetColor = distanceColor.rgb;
    if (timeOfDay > 0) {
        vec3 duskFadeColor = vec3(0.25, 0.12, 0.06);  // Warm amber for dusk
        vec3 nightFadeColor = vec3(0.02, 0.03, 0.06); // Dark blue for night
        float effectIntensity = 1.0 - visibilityMultiplier;

        if (timeOfDay == 1) {
            fadeTargetColor = mix(distanceColor.rgb, duskFadeColor, effectIntensity);
        } else if (timeOfDay == 2) {
            fadeTargetColor = mix(distanceColor.rgb, nightFadeColor, effectIntensity);
        }
    }
    finalColor = mix(finalColor, fadeTargetColor, hardFade);

    outputColor = vec4(finalColor, 1.0);
}
