#version 330 core
out vec4 FragColor;

in vec2 TexCoord;
in vec3 FragPos;
in vec3 Normal;
in vec4 FragPosLightSpace;
in vec3 toLightVector;
in float faceFlags0;  // Face flags for material properties

uniform sampler2D basic_texture;
uniform sampler2D shadowMap;
uniform vec3 lightDirection = vec3(-0.2, 0.9, -0.3);  // Light from high above, slightly front-left
uniform vec3 viewDirection = vec3(0.0, 0.0, -1.0);      // Player's camera view direction (world space)
// First-person weapon lighting (darkened to reduce washed-out look)
uniform float ambientStrength = 0.0;    // No ambient - rely on diffuse only
uniform float diffuseStrength = 0.50;   // Balanced diffuse for visible light direction
uniform float specularStrength = 0.65;  // Increased specular for more reflective surfaces
uniform float rainIntensity = 0.0;     // 0.0 = dry, 1.0 = heavy rain
uniform float time = 0.0;              // Time for dynamic effects
uniform float visibilityMultiplier = 1.0;  // Time-of-day visibility (1.0 = day, 0.35 = night)
uniform int debugMode = 0;             // 0=off, 1=face flags, 2=normals, 3=specular contribution
uniform bool enable_transparency = true; // Enable sfOpacity transparency (color-key)

// Carnivores Face Flags (from Hunt.h)
// Note: sfPhong (0x0030) and sfEnvMap (0x0050) include the sfMortal bit (0x0010)
const int SF_DOUBLE_SIDE  = 0x0001;  // Render both sides (no backface culling)
const int SF_DARK_BACK    = 0x0002;  // Darken backface instead of culling
const int SF_OPACITY      = 0x0004;  // Binary alpha: black (0) = transparent, else opaque
const int SF_TRANSPARENT  = 0x0008;  // Semi-transparent blending (glass, water)
const int SF_MORTAL       = 0x0010;  // Vital hit area (head, heart) - gameplay only
const int SF_PHONG        = 0x0030;  // Phong specular highlight (C2 only)
const int SF_ENVMAP       = 0x0050;  // Environment map reflection (C2 only)

// Decode face flags from packed float
int getFaceFlags() {
    return int(faceFlags0);
}

bool hasFlag(int flag) {
    return (getFaceFlags() & flag) == flag;
}

// Check if face should have specular highlights
// Since C2 weapons don't use sfPhong/sfEnvMap flags, we detect based on texture color
// Skin tones are excluded from specular, metal/dark surfaces get specular
bool hasSpecularMaterial(vec3 texColor) {
    int flags = getFaceFlags();

    // First check for explicit sfPhong or sfEnvMap flags (if present, always use them)
    bool hasPhong = (flags & 0x0020) != 0;   // Phong-specific bit
    bool hasEnvMap = (flags & 0x0040) != 0;  // EnvMap-specific bit
    if (hasPhong || hasEnvMap) return true;

    // No explicit flags - use texture-based detection for weapons
    // Detect skin tones (warm colors with medium brightness) and exclude them
    float r = texColor.r;
    float g = texColor.g;
    float b = texColor.b;

    // Calculate color properties
    float brightness = (r + g + b) / 3.0;
    float saturation = max(max(r, g), b) - min(min(r, g), b);

    // Skin tone detection: warm colors (more red than blue, medium brightness)
    // Typical skin: R > G > B, brightness 0.3-0.8, not too saturated
    bool isSkinTone = (r > g * 0.85) &&           // Red dominant or close
                      (g > b) &&                   // Green > Blue (warm)
                      (r - b > 0.05) &&            // Red-blue difference (warmth)
                      (brightness > 0.25) &&       // Not too dark
                      (brightness < 0.85) &&       // Not too bright
                      (saturation < 0.6);          // Not oversaturated

    // Metal detection: dark colors, grays, or high contrast
    bool isMetal = (brightness < 0.3) ||          // Dark surfaces (gun metal)
                   (saturation < 0.15) ||         // Desaturated (gray metal)
                   (brightness > 0.85);           // Very bright (chrome)

    // Return true for metal, false for skin
    return isMetal && !isSkinTone;
}

float ShadowCalculation(vec4 fragPosLightSpace, vec3 lightDir, vec3 normal)
{
    // Perform perspective divide
    vec3 projCoords = fragPosLightSpace.xyz / fragPosLightSpace.w;
    
    // Transform to [0,1] range
    projCoords = projCoords * 0.5 + 0.5;
    
    // Check if fragment is outside light's view
    if (projCoords.z > 1.0 || projCoords.x < 0.0 || projCoords.x > 1.0 || 
        projCoords.y < 0.0 || projCoords.y > 1.0) {
        return 0.0; // No shadow outside light frustum
    }
    
    // Get closest depth value from light's perspective
    float closestDepth = texture(shadowMap, projCoords.xy).r;
    
    // Get depth of current fragment from light's perspective
    float currentDepth = projCoords.z;
    
    // Calculate bias based on normal and light direction
    float bias = max(0.01 * (1.0 - dot(normal, lightDir)), 0.005);
    
    // PCF for softer shadows
    float shadow = 0.0;
    vec2 texelSize = 1.0 / textureSize(shadowMap, 0);
    for(int x = -1; x <= 1; ++x)
    {
        for(int y = -1; y <= 1; ++y)
        {
            float pcfDepth = texture(shadowMap, projCoords.xy + vec2(x, y) * texelSize).r;
            shadow += currentDepth - bias > pcfDepth ? 1.0 : 0.0;
        }
    }
    shadow /= 9.0;
    
    return shadow;
}

// Bicubic interpolation weight function (Catmull-Rom spline)
float catmullRomWeight(float x) {
    float ax = abs(x);
    if (ax < 1.0) {
        return (1.5 * ax - 2.5) * ax * ax + 1.0;
    } else if (ax < 2.0) {
        return ((-0.5 * ax + 2.5) * ax - 4.0) * ax + 2.0;
    }
    return 0.0;
}

// High-quality bicubic texture sampling with sharpening for crisp weapon textures
// Weapons are close to camera so we want maximum detail
// Returns vec4 with alpha channel for transparency checks
vec4 bicubicSharpenTexture(sampler2D tex, vec2 uv) {
    vec2 texSize = vec2(textureSize(tex, 0));
    vec2 texelSize = 1.0 / texSize;

    // Convert to texel coordinates
    vec2 texCoord = uv * texSize - 0.5;
    vec2 texelCenter = floor(texCoord);
    vec2 f = texCoord - texelCenter;

    // Sample 4x4 grid for bicubic
    vec4 result = vec4(0.0);
    float totalWeight = 0.0;

    for (int j = -1; j <= 2; j++) {
        for (int i = -1; i <= 2; i++) {
            vec2 samplePos = (texelCenter + vec2(float(i), float(j)) + 0.5) * texelSize;
            float weight = catmullRomWeight(float(i) - f.x) * catmullRomWeight(float(j) - f.y);
            result += texture(tex, samplePos) * weight;
            totalWeight += weight;
        }
    }
    result /= totalWeight;

    // Apply sharpening to enhance weapon detail (weapons are close, need crisp textures)
    float sharpness = 0.35;  // Moderate sharpening for weapons

    // Unsharp mask (RGB only, preserve alpha)
    vec3 center = texture(tex, uv).rgb;
    vec3 blur = vec3(0.0);
    blur += texture(tex, uv + vec2(texelSize.x, 0.0)).rgb;
    blur += texture(tex, uv - vec2(texelSize.x, 0.0)).rgb;
    blur += texture(tex, uv + vec2(0.0, texelSize.y)).rgb;
    blur += texture(tex, uv - vec2(0.0, texelSize.y)).rgb;
    blur *= 0.25;

    vec3 sharpened = result.rgb + (result.rgb - blur) * sharpness;
    return vec4(clamp(sharpened, 0.0, 1.0), result.a);
}

// Soft tone mapping to prevent over-saturation
vec3 toneMapACES(vec3 x) {
    // Simplified ACES filmic tone mapping
    float a = 2.51;
    float b = 0.03;
    float c = 2.43;
    float d = 0.59;
    float e = 0.14;
    return clamp((x * (a * x + b)) / (x * (c * x + d) + e), 0.0, 1.0);
}

void main()
{
    // Use standard texture sampling with subtle blur to soften old textures
    // Small offset to avoid UV seam bleeding while hiding low-res artifacts
    vec2 texelSize = 1.0 / vec2(textureSize(basic_texture, 0));
    float blurAmount = 0.4;  // Subtle blur strength

    vec4 center = texture(basic_texture, TexCoord);
    vec4 blur = texture(basic_texture, TexCoord + vec2(texelSize.x, 0.0) * blurAmount)
              + texture(basic_texture, TexCoord - vec2(texelSize.x, 0.0) * blurAmount)
              + texture(basic_texture, TexCoord + vec2(0.0, texelSize.y) * blurAmount)
              + texture(basic_texture, TexCoord - vec2(0.0, texelSize.y) * blurAmount);

    // Blend: mostly original with a touch of blur
    vec4 rawSample = mix(center, blur * 0.25, 0.3);

    // Convert from BGR to RGB (matching original Carnivores format)
    vec3 textureColor = vec3(rawSample.b, rawSample.g, rawSample.r);
    float textureAlpha = rawSample.a;

    // ========== CARNIVORES ALPHA SYSTEM (sfOpacity) ==========
    // sfOpacity (0x0004): Binary color-key transparency - black pixels are transparent
    // C1 weapon models use this for transparent backgrounds on weapon sprites
    if (hasFlag(SF_OPACITY) && enable_transparency) {
        // Check if pixel is near-black (color-key transparency)
        // Use the raw BGR values before conversion for accurate black detection
        bool isTransparentByAlpha = (textureAlpha < 0.5);
        bool isTransparentByColor = (rawSample.r <= 0.004 && rawSample.g <= 0.004 && rawSample.b <= 0.004);

        if (isTransparentByAlpha || isTransparentByColor) {
            discard;  // Complete transparency - discard fragment
        }
    }

    // Clamp texture color to valid range after sharpening
    textureColor = clamp(textureColor, 0.0, 1.0);

    // Normalize vectors
    vec3 norm = normalize(Normal);
    vec3 lightDir = normalize(-lightDirection);
    // Use player's actual view direction for dynamic reflections
    // This makes specular highlights move as the player looks around
    vec3 viewDir = normalize(-viewDirection);

    // ========== PHONG LIGHTING MODEL ==========
    // Preserves dark colors (black metal) while adding visible highlights
    // Specular is ONLY applied to faces with sfPhong (0x0030) or sfEnvMap (0x0050) flags

    // 1. Ambient - use uniform for per-object control
    vec3 ambient = ambientStrength * textureColor;

    // 2. Diffuse (Lambertian) - heavily compressed range to avoid blowout
    // Use half-lambert style wrap to lift shadows, then aggressively compress the bright end
    float NdotL = max(dot(norm, lightDir), 0.0);
    float wrappedNdotL = NdotL * 0.5 + 0.5;  // Half-lambert: remap 0-1 to 0.5-1.0
    float compressedNdotL = min(wrappedNdotL, 0.7) * 0.5;  // Cap at 70%, scale to 50%
    vec3 diffuse = diffuseStrength * compressedNdotL * textureColor;

    // Base ambient fill to ensure nothing goes too dark
    vec3 baseFill = 0.3 * textureColor;  // Slightly higher constant base lighting
    diffuse += baseFill;

    // 3. Phong Specular - applied to metal surfaces, excluded from skin
    vec3 specular = vec3(0.0);
    float specAngle = 0.0;
    bool isSpecular = hasSpecularMaterial(textureColor);

    if (isSpecular) {
        // This face has the sfPhong or sfEnvMap flag - apply specular
        vec3 reflectDir = reflect(-lightDir, norm);
        specAngle = max(dot(viewDir, reflectDir), 0.0);

        // Primary Phong highlight - tighter for a metallic look
        float phongSpec = pow(specAngle, 32.0);  // Lower exponent = broader, more visible highlight

        // Secondary broader sheen for metallic surfaces
        float metalSheen = pow(specAngle, 8.0) * 0.25;  // Broad soft reflection

        // Dynamic glint - subtle movement of highlight
        float glintPhase = time * 0.4;
        vec3 glintOffset = vec3(sin(glintPhase) * 0.08, cos(glintPhase * 0.8) * 0.06, 0.0);
        vec3 dynamicLightDir = normalize(lightDir + glintOffset);
        vec3 dynamicReflectDir = reflect(-dynamicLightDir, norm);
        float dynamicSpecAngle = max(dot(viewDir, dynamicReflectDir), 0.0);
        float glint = pow(dynamicSpecAngle, 48.0) * 0.9;

        // Combine specular - tinted toward texture color and clamped to prevent blowout
        vec3 specColor = mix(vec3(1.0), textureColor + 0.5, 0.4);  // More tinted toward texture
        float totalSpec = min(phongSpec + metalSheen + glint, 0.6);  // Clamp total specular contribution
        specular = specularStrength * totalSpec * specColor;
    }

    // 4. Rim lighting - subtle edge highlight (only for specular materials)
    vec3 rimLight = vec3(0.0);
    float rim = 0.0;
    if (isSpecular) {
        rim = 1.0 - max(dot(norm, viewDir), 0.0);
        rim = pow(rim, 3.0);
        rimLight = rim * 0.08 * vec3(0.9, 0.95, 1.0);  // Reduced rim
    }

    // Combine all lighting with hard cap to prevent any blowout
    vec3 lighting = ambient + diffuse + specular + rimLight;
    lighting = min(lighting, vec3(0.85));  // Hard cap at 85% brightness

    // ========== RAIN/WET WEAPON EFFECTS ==========
    // Rain makes ALL surfaces wet and reflective (overrides normal material flags)
    if (rainIntensity > 0.0) {
        // 1. Darken weapon (wet surfaces absorb more light)
        float wetDarkening = 1.0 - (rainIntensity * 0.2);
        lighting *= wetDarkening;

        // 2. Calculate wet specular for all faces (rain makes everything shiny)
        vec3 reflectDir = reflect(-lightDir, norm);
        float wetSpecAngle = max(dot(viewDir, reflectDir), 0.0);
        float wetSpec = pow(wetSpecAngle, 64.0) * rainIntensity * 0.8;
        float wetSheen = pow(wetSpecAngle, 16.0) * rainIntensity * 0.3;

        vec3 wetHighlightColor = vec3(0.9, 0.95, 1.0);
        lighting += (wetSpec + wetSheen) * wetHighlightColor;

        // 3. Enhanced rim when wet
        float wetRim = 1.0 - max(dot(norm, viewDir), 0.0);
        wetRim = pow(wetRim, 2.0) * rainIntensity * 0.25;
        lighting += wetRim * wetHighlightColor;

        // 4. Cool tint shift
        vec3 coolTint = vec3(0.92, 0.96, 1.0);
        lighting = mix(lighting, lighting * coolTint, rainIntensity * 0.4);
    }

    // Skip tone mapping - preserve true blacks and original contrast
    // Apply gamma correction to darken midtones (gamma > 1 darkens)
    vec3 finalColor = pow(clamp(lighting, 0.0, 1.0), vec3(1.3));

    // ========== TIME-OF-DAY EFFECTS (DUSK/NIGHT) ==========
    // Weapons held in hand are affected by time of day lighting
    // No fog needed since weapons are close to camera
    if (visibilityMultiplier < 1.0) {
        // visibilityMultiplier: 1.0 = day, 0.6 = dusk, 0.35 = night
        // Dusk range: 0.6 to ~0.5, Night range: below 0.5

        float duskAmount = 0.0;
        float nightAmount = 0.0;

        if (visibilityMultiplier >= 0.5) {
            // DUSK: warm orange tint, slight darkening
            duskAmount = (1.0 - visibilityMultiplier) / 0.5;  // 0 at day, 1 at vis=0.5

            // Slight darkening at dusk (up to 30%)
            float duskDarkening = 1.0 - (duskAmount * 0.30);
            finalColor *= duskDarkening;

            // Warm orange/golden tint for sunset lighting
            vec3 duskTint = vec3(1.0, 0.85, 0.6);  // Warm orange-gold
            float duskTintAmount = duskAmount * 0.35;
            finalColor = mix(finalColor, finalColor * duskTint, duskTintAmount);

            // Slight saturation boost at dusk (golden hour effect)
            float duskSatBoost = duskAmount * 0.15;
            vec3 duskGray = vec3(dot(finalColor, vec3(0.299, 0.587, 0.114)));
            finalColor = mix(finalColor, mix(duskGray, finalColor, 1.0 + duskSatBoost), 1.0);
        } else {
            // NIGHT: cool blue tint, heavy darkening
            nightAmount = (0.5 - visibilityMultiplier) / 0.5;  // 0 at vis=0.5, 1 at vis=0

            // Heavy darkening at night (up to 85%)
            float nightDarkening = 1.0 - (nightAmount * 0.85);
            finalColor *= nightDarkening;

            // Heavy desaturation - colors hard to distinguish at night
            vec3 grayscale = vec3(dot(finalColor, vec3(0.299, 0.587, 0.114)));
            float desatAmount = nightAmount * 0.80;
            finalColor = mix(finalColor, grayscale, desatAmount);

            // Cool blue moonlight tint
            vec3 nightTint = vec3(0.5, 0.6, 0.9);  // Blue-ish moonlight
            float tintAmount = nightAmount * 0.5;
            finalColor = mix(finalColor, finalColor * nightTint, tintAmount);
        }
    }

    // Debug modes
    if (debugMode == 1) {
        // Specular material detection visualization
        // Green = will get specular (metal), Red = no specular (skin/organic)
        vec3 debugColor = isSpecular ? vec3(0.0, 1.0, 0.0) : vec3(1.0, 0.0, 0.0);

        // Blend with original texture to see what's being detected
        debugColor = mix(textureColor, debugColor, 0.7);

        FragColor = vec4(debugColor, 1.0);
        return;
    }
    else if (debugMode == 2) {
        // Normal visualization
        FragColor = vec4(norm * 0.5 + 0.5, 1.0);
        return;
    }
    else if (debugMode == 3) {
        // Specular contribution visualization - shows actual specular intensity
        vec3 debugSpec = vec3(0.0);
        if (isSpecular) {
            debugSpec = vec3(0.0, 0.5, 0.0);  // Base green = has specular material
            debugSpec += specular * 2.0;       // Add boosted actual specular value
        } else {
            debugSpec = vec3(0.3, 0.0, 0.0);  // Dark red = no specular material
        }
        FragColor = vec4(debugSpec, 1.0);
        return;
    }

    FragColor = vec4(finalColor, 1.0);
}

