# AI Controller Refactoring Plan

## Overview

Refactor the monolithic `CEClassicAIController` (6,400+ lines) into a composable architecture based on:

1. **Data Classes** - Pure state structs, no logic
2. **Behaviors** - Stateless logic that reads data and emits Intents
3. **Intents** - Declarative requests ("what I want")
4. **Resolvers** - Apply intents to data, handle conflicts, interpolation

This follows Data-Oriented Design principles with a lightweight ECS-like pattern.

---

## Architecture Diagram

```
┌──────────────────────────────────────────────────────────────────────┐
│                           DATA CLASSES                                │
│  (Pure state - no logic, just values that change over time)          │
│                                                                       │
│  ┌─────────────┐ ┌─────────────┐ ┌─────────────┐ ┌─────────────┐    │
│  │ Transform   │ │ Locomotion  │ │ Animation   │ │ Detection   │    │
│  │ Data        │ │ Data        │ │ Data        │ │ Data        │    │
│  └─────────────┘ └─────────────┘ └─────────────┘ └─────────────┘    │
│  ┌─────────────┐ ┌─────────────┐ ┌─────────────┐                    │
│  │ Combat      │ │ TerrainAdapt│ │ Interpolated│                    │
│  │ Data        │ │ Data        │ │ <T>         │  ← Generic!        │
│  └─────────────┘ └─────────────┘ └─────────────┘                    │
└──────────────────────────────────────────────────────────────────────┘
                              │
                              ▼
┌──────────────────────────────────────────────────────────────────────┐
│                           BEHAVIORS                                   │
│  (Stateless or minimal state - read data, emit Intents)              │
│                                                                       │
│  ┌────────────────┐ ┌────────────────┐ ┌────────────────┐            │
│  │ HazardDeflect  │ │ SlopeDeflect   │ │ PursueTarget   │            │
│  │ Behavior       │ │ Behavior       │ │ Behavior       │            │
│  └────────────────┘ └────────────────┘ └────────────────┘            │
│  ┌────────────────┐ ┌────────────────┐ ┌────────────────┐            │
│  │ WanderBehavior │ │ FleeBehavior   │ │ DetectPlayer   │            │
│  │                │ │                │ │ Behavior       │            │
│  └────────────────┘ └────────────────┘ └────────────────┘            │
└──────────────────────────────────────────────────────────────────────┘
                              │
                              ▼
┌──────────────────────────────────────────────────────────────────────┐
│                            INTENTS                                    │
│  (Declarative requests - "what I want", not "how to do it")          │
│                                                                       │
│  ┌─────────────┐ ┌─────────────┐ ┌─────────────┐ ┌─────────────┐    │
│  │ MoveIntent  │ │ RotateIntent│ │ AnimIntent  │ │ AudioIntent │    │
│  └─────────────┘ └─────────────┘ └─────────────┘ └─────────────┘    │
└──────────────────────────────────────────────────────────────────────┘
                              │
                              ▼
┌──────────────────────────────────────────────────────────────────────┐
│                          INTENT RESOLVERS                             │
│  (Apply intents to data classes, handle conflicts, interpolation)    │
│                                                                       │
│  ┌────────────────────┐ ┌────────────────────┐                       │
│  │ MovementResolver   │ │ RotationResolver   │                       │
│  │ (merges intents,   │ │ (merges intents,   │                       │
│  │  applies DeltaFunc)│ │  handles turn rate)│                       │
│  └────────────────────┘ └────────────────────┘                       │
└──────────────────────────────────────────────────────────────────────┘
```

---

## Phase 1: Foundation (Data Classes & Interpolation)

### 1.1 Create `Interpolated<T>` Template

Generic data class for any value that smoothly transitions over time.

**File:** `src/ai/data/Interpolated.hpp`

```cpp
template<typename T>
struct Interpolated {
    T current;
    T target;
    float rate;  // DeltaFunc divisor (higher = slower)

    Interpolated() : current{}, target{}, rate{1.0f} {}
    Interpolated(T initial, float r = 1.0f)
        : current(initial), target(initial), rate(r) {}
};
```

### 1.2 Create Core Data Classes

**File:** `src/ai/data/AIDataClasses.hpp`

| Data Class | Purpose | Key Fields |
|------------|---------|------------|
| `TransformData` | Position and orientation | position, heading, pitch, roll, forward |
| `LocomotionData` | Movement state | speed (Interpolated), velocity, slideVelocity, turnRate, bodyBend |
| `TerrainAdaptData` | Terrain interaction | slopeAngle, slopeDirection, deflectionBias, hazard proximity |
| `DetectionData` | Player awareness | visibility, smellIntensity, hasLOS, stationaryExposure |
| `AnimationData` | Animation state | currentAnim, fTime, frameA/B, blend, cycleComplete |
| `CombatData` | Health and combat | health, isDead, isFeeding, kill attribution |
| `AIStateData` | High-level AI state | phase, state, afraidTime, targets |

---

## Phase 2: Intent System

### 2.1 Create Intent Types

**File:** `src/ai/intents/AIIntents.hpp`

| Intent | Purpose | Modes |
|--------|---------|-------|
| `MoveIntent` | Request movement | Additive, SetSpeed, SetVelocity |
| `RotateIntent` | Request rotation | FaceDirection, FacePosition, AddRotation |
| `AnimationIntent` | Request animation change | (single winner) |
| `AudioIntent` | Request audio playback | (queued) |
| `TerrainAdaptIntent` | Configure terrain adaptation | (optional) |

### 2.2 Create `FrameIntents` Container

Collects all intents from all behaviors for a single frame.

```cpp
struct FrameIntents {
    std::vector<MoveIntent> movement;
    std::vector<RotateIntent> rotation;
    std::optional<AnimationIntent> animation;
    std::vector<AudioIntent> audio;
    std::optional<TerrainAdaptIntent> terrainAdapt;
};
```

---

## Phase 3: Behavior System

### 3.1 Create Behavior Interface

**File:** `src/ai/behaviors/IBehavior.hpp`

```cpp
class IBehavior {
public:
    virtual ~IBehavior() = default;
    virtual void evaluate(const BehaviorContext& ctx, FrameIntents& out) = 0;
    virtual std::string getName() const = 0;
    virtual void onActivate() {}
    virtual void onDeactivate() {}
};
```

### 3.2 Create `BehaviorContext`

**File:** `src/ai/behaviors/BehaviorContext.hpp`

Read-only view of all data classes plus world query interfaces.

```cpp
struct BehaviorContext {
    // Entity data (const references)
    const TransformData& transform;
    const LocomotionData& locomotion;
    const TerrainAdaptData& terrain;
    const DetectionData& detection;
    const CombatData& combat;
    const AIStateData& state;
    const AnimationData& animation;

    // Timing
    float deltaTime;
    double currentTime;

    // World queries
    const ITerrainQuery* terrainQuery;
    const ISensoryQuery* sensoryQuery;
};
```

### 3.3 Create World Query Interfaces (Primitives)

**File:** `src/ai/primitives/ITerrainQuery.hpp`

```cpp
class ITerrainQuery {
public:
    virtual ~ITerrainQuery() = default;
    virtual float getHeightAt(float x, float z) const = 0;
    virtual float getSlopeAngleAt(float x, float z) const = 0;
    virtual glm::vec3 getSlopeDirectionAt(float x, float z) const = 0;
    virtual bool isWaterAt(float x, float z) const = 0;
    virtual bool isDangerAt(float x, float z) const = 0;
    virtual bool isValidPosition(float x, float z) const = 0;
};
```

**File:** `src/ai/primitives/ISensoryQuery.hpp`

```cpp
class ISensoryQuery {
public:
    virtual ~ISensoryQuery() = default;
    virtual bool hasLineOfSightTo(glm::vec3 from, glm::vec3 to) const = 0;
    virtual float getCoverFactorAt(glm::vec3 pos) const = 0;
};
```

---

## Phase 4: Intent Resolvers

### 4.1 Create `MovementResolver`

**File:** `src/ai/resolvers/MovementResolver.hpp`

- Sorts intents by priority
- Processes SetVelocity (highest priority override)
- Processes SetSpeed (highest priority wins)
- Accumulates Additive intents (weighted by priority)
- Applies DeltaFunc interpolation
- Updates position

### 4.2 Create `RotationResolver`

**File:** `src/ai/resolvers/RotationResolver.hpp`

- Handles FaceDirection/FacePosition/AddRotation modes
- Applies turn rate limits
- Updates heading with DeltaFunc
- Updates forward vector

### 4.3 Create `AnimationResolver`

**File:** `src/ai/resolvers/AnimationResolver.hpp`

- Highest priority animation intent wins
- Handles animation transitions
- Updates fTime accumulation

---

## Phase 5: Behavior Extraction

Extract behaviors from `CEClassicAIController` one at a time.

### Priority Order

| # | Behavior | Source Methods | Complexity | Lines Est. |
|---|----------|---------------|------------|------------|
| 1 | `HazardDeflectionBehavior` | `computeHazardDeflectionDirection()` | Low | ~100 |
| 2 | `SlopeDeflectionBehavior` | `applyTerrainSliding()` (deflection part) | Medium | ~120 |
| 3 | `DownhillSlideBehavior` | `applyTerrainSliding()` (slide part) | Low | ~80 |
| 4 | `TerrainAdaptBehavior` | `ThinkY_Beta_Gamma()` | Medium | ~100 |
| 5 | `WanderBehavior` | `SetNewTargetPlace()` | Low | ~60 |
| 6 | `PlayerDetectionBehavior` | `calculatePlayerVisibility()`, `calculatePlayerSmellIntensity()` | High | ~200 |
| 7 | `PursueBehavior` | Chase logic from `AnimatePredator()` etc. | High | ~150 |
| 8 | `FleeBehavior` | Flee logic from behavior methods | Medium | ~100 |
| 9 | `AttackBehavior` | Jump/kill attack logic | High | ~200 |

---

## Phase 6: Controller Integration

### 6.1 Refactor `CEClassicAIController`

Transform into thin orchestrator:

```cpp
class CEClassicAIController {
public:
    void update(float deltaTime, double currentTime) {
        // 1. Update world queries
        updateTerrainData();
        updateDetectionData(currentTime);

        // 2. Build context
        BehaviorContext ctx = buildContext(deltaTime, currentTime);

        // 3. Collect intents
        FrameIntents intents;
        for (auto& behavior : m_behaviors) {
            behavior->evaluate(ctx, intents);
        }

        // 4. Resolve intents
        m_movementResolver.resolve(intents.movement, m_locomotion, m_transform, deltaTime);
        m_rotationResolver.resolve(intents.rotation, m_locomotion, m_transform, deltaTime);
        m_animationResolver.resolve(intents.animation, m_animation);

        // 5. Queue audio
        for (const auto& audio : intents.audio) {
            queueAudio(audio);
        }
    }

private:
    // Data classes
    TransformData m_transform;
    LocomotionData m_locomotion;
    TerrainAdaptData m_terrainAdapt;
    DetectionData m_detection;
    CombatData m_combat;
    AIStateData m_state;
    AnimationData m_animation;

    // Behaviors
    std::vector<std::unique_ptr<IBehavior>> m_behaviors;

    // Resolvers
    MovementResolver m_movementResolver;
    RotationResolver m_rotationResolver;
    AnimationResolver m_animationResolver;
};
```

---

## File Structure

```
src/ai/
├── data/
│   ├── Interpolated.hpp
│   ├── AIDataClasses.hpp
│   └── AIDataClasses.cpp (if needed for complex defaults)
├── intents/
│   └── AIIntents.hpp
├── behaviors/
│   ├── IBehavior.hpp
│   ├── BehaviorContext.hpp
│   ├── HazardDeflectionBehavior.hpp
│   ├── HazardDeflectionBehavior.cpp
│   ├── SlopeDeflectionBehavior.hpp
│   ├── SlopeDeflectionBehavior.cpp
│   └── ... (more behaviors)
├── primitives/
│   ├── ITerrainQuery.hpp
│   ├── ISensoryQuery.hpp
│   ├── TerrainQueryImpl.hpp
│   └── TerrainQueryImpl.cpp
└── resolvers/
    ├── MovementResolver.hpp
    ├── MovementResolver.cpp
    ├── RotationResolver.hpp
    ├── RotationResolver.cpp
    └── AnimationResolver.hpp
```

---

## Implementation Checklist

### Phase 1: Foundation
- [ ] Create `src/ai/` directory structure
- [ ] Implement `Interpolated<T>` template
- [ ] Implement `TransformData`
- [ ] Implement `LocomotionData`
- [ ] Implement `TerrainAdaptData`
- [ ] Implement `DetectionData`
- [ ] Implement `AnimationData`
- [ ] Implement `CombatData`
- [ ] Implement `AIStateData`

### Phase 2: Intents
- [ ] Implement `MoveIntent`
- [ ] Implement `RotateIntent`
- [ ] Implement `AnimationIntent`
- [ ] Implement `AudioIntent`
- [ ] Implement `FrameIntents`

### Phase 3: Behaviors
- [ ] Implement `IBehavior` interface
- [ ] Implement `BehaviorContext`
- [ ] Implement `ITerrainQuery` interface
- [ ] Implement `ISensoryQuery` interface
- [ ] Implement `TerrainQueryImpl` (wraps C2MapFile)

### Phase 4: Resolvers
- [ ] Implement `MovementResolver`
- [ ] Implement `RotationResolver`
- [ ] Implement `AnimationResolver`

### Phase 5: First Behaviors
- [ ] Implement `HazardDeflectionBehavior`
- [ ] Implement `SlopeDeflectionBehavior`
- [ ] Implement `DownhillSlideBehavior`
- [ ] Test behaviors in isolation
- [ ] Integrate with controller

### Phase 6: Migration
- [ ] Add data classes to `CEClassicAIController`
- [ ] Add behavior list to controller
- [ ] Add resolvers to controller
- [ ] Migrate `applyTerrainSliding()` to use new system
- [ ] Verify behavior matches original
- [ ] Continue extracting remaining behaviors

---

## Design Principles

1. **Data classes are dumb** - No methods beyond simple accessors
2. **Behaviors are pure** - Same input always produces same output
3. **Intents are declarative** - Express what, not how
4. **Resolvers are authoritative** - Single place for conflict resolution
5. **Controller orchestrates** - Minimal logic, maximum delegation
6. **Interpolation is standardized** - `Interpolated<T>` + DeltaFunc everywhere

---

## Testing Strategy

1. **Unit test behaviors** - Mock BehaviorContext, verify intent output
2. **Unit test resolvers** - Provide intents, verify data changes
3. **Integration test** - Run full update loop, compare to original behavior
4. **A/B testing** - Run old and new systems in parallel, compare results

---

## Migration Strategy

1. **Parallel implementation** - New system alongside old
2. **Feature flag** - Switch between old/new per-AI or globally
3. **Gradual migration** - Extract one behavior at a time
4. **Validation** - Compare outputs after each extraction
5. **Cleanup** - Remove old code after validation
