// Copyright 2015 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "ash/metrics/task_switch_time_tracker.h"

#include <string>

#include "ash/test/task_switch_time_tracker_test_api.h"
#include "base/test/histogram_tester.h"
#include "testing/gtest/include/gtest/gtest.h"

namespace ash {

namespace {

// A dummy histogram name.
const std::string kHistogramName = "Dummy.Histogram";

}  // namespace

class TaskSwitchTimeTrackerTest : public testing::Test {
 public:
  TaskSwitchTimeTrackerTest();
  ~TaskSwitchTimeTrackerTest() override;

  // testing::Test:
  void SetUp() override;
  void TearDown() override;

  // Wrapper to the test targets OnTaskSwitch() method.
  void OnTaskSwitch();

  TaskSwitchTimeTracker* time_tracker() {
    return time_tracker_test_api_->time_tracker();
  }

 protected:
  // Used to verify recorded histogram data.
  scoped_ptr<base::HistogramTester> histogram_tester_;

  // A Test API that wraps the test target.
  scoped_ptr<test::TaskSwitchTimeTrackerTestAPI> time_tracker_test_api_;

 private:
  DISALLOW_COPY_AND_ASSIGN(TaskSwitchTimeTrackerTest);
};

TaskSwitchTimeTrackerTest::TaskSwitchTimeTrackerTest() {
}

TaskSwitchTimeTrackerTest::~TaskSwitchTimeTrackerTest() {
}

void TaskSwitchTimeTrackerTest::SetUp() {
  testing::Test::SetUp();

  histogram_tester_.reset(new base::HistogramTester());
  time_tracker_test_api_.reset(
      new test::TaskSwitchTimeTrackerTestAPI(kHistogramName));
  // The TaskSwitchTimeTracker interprets a value of base::TimeTicks() as if the
  // |last_action_time_| has not been set.
  time_tracker_test_api_->Advance(base::TimeDelta::FromMilliseconds(1));
}

void TaskSwitchTimeTrackerTest::TearDown() {
  testing::Test::TearDown();
  time_tracker_test_api_.reset();
  histogram_tester_.reset();
}

void TaskSwitchTimeTrackerTest::OnTaskSwitch() {
  time_tracker()->OnTaskSwitch();
}

// Verifies TaskSwitchTimeTracker::HasLastActionTime() returns false after
// construction.
TEST_F(TaskSwitchTimeTrackerTest,
       HasLastActionTimeShouldBeFalseAfterConstruction) {
  EXPECT_FALSE(time_tracker_test_api_->HasLastActionTime());
}

// Verifies TaskSwitchTimeTracker::HasLastActionTime() returns true after the
// first call to TaskSwitchTimeTracker::OnTaskSwitch() and no histogram data was
// recorded.
TEST_F(TaskSwitchTimeTrackerTest,
       HasLastActionTimeShouldBeTrueAfterOnTaskSwitch) {
  OnTaskSwitch();
  histogram_tester_->ExpectTotalCount(kHistogramName, 0);
}

// Verfies that the histogram data is recorded in the correct buckets.
TEST_F(TaskSwitchTimeTrackerTest, RecordAfterTwoTaskSwitches) {
  OnTaskSwitch();

  time_tracker_test_api_->Advance(base::TimeDelta::FromMilliseconds(2));
  OnTaskSwitch();
  histogram_tester_->ExpectTotalCount(kHistogramName, 1);
  histogram_tester_->ExpectBucketCount(kHistogramName, 0, 1);

  time_tracker_test_api_->Advance(base::TimeDelta::FromSeconds(1));
  OnTaskSwitch();
  histogram_tester_->ExpectTotalCount(kHistogramName, 2);
  histogram_tester_->ExpectBucketCount(kHistogramName, 1, 1);
}

}  // namespace ash
