// Copyright 2015 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/policy/cloud/remote_commands_invalidator.h"

#include <string>

#include "base/logging.h"
#include "components/invalidation/invalidation.h"
#include "components/invalidation/invalidation_service.h"
#include "components/invalidation/invalidation_util.h"
#include "components/invalidation/invalidator_state.h"
#include "components/invalidation/object_id_invalidation_map.h"
#include "components/invalidation/single_object_invalidation_set.h"

namespace policy {

RemoteCommandsInvalidator::RemoteCommandsInvalidator() {
}

RemoteCommandsInvalidator::~RemoteCommandsInvalidator() {
  DCHECK_EQ(SHUT_DOWN, state_);
}

void RemoteCommandsInvalidator::Initialize(
    invalidation::InvalidationService* invalidation_service) {
  DCHECK_EQ(SHUT_DOWN, state_);
  DCHECK(thread_checker_.CalledOnValidThread());

  DCHECK(invalidation_service);
  invalidation_service_ = invalidation_service;

  state_ = STOPPED;
  OnInitialize();
}

void RemoteCommandsInvalidator::Shutdown() {
  DCHECK_NE(SHUT_DOWN, state_);
  DCHECK(thread_checker_.CalledOnValidThread());

  Stop();

  state_ = SHUT_DOWN;
  OnShutdown();
}

void RemoteCommandsInvalidator::Start() {
  DCHECK_EQ(STOPPED, state_);
  DCHECK(thread_checker_.CalledOnValidThread());
  state_ = STARTED;

  OnStart();
}

void RemoteCommandsInvalidator::Stop() {
  DCHECK_NE(SHUT_DOWN, state_);
  DCHECK(thread_checker_.CalledOnValidThread());

  if (state_ == STARTED) {
    Unregister();
    state_ = STOPPED;

    OnStop();
  }
}

void RemoteCommandsInvalidator::OnInvalidatorStateChange(
    syncer::InvalidatorState state) {
  DCHECK_EQ(STARTED, state_);
  DCHECK(thread_checker_.CalledOnValidThread());

  invalidation_service_enabled_ = state == syncer::INVALIDATIONS_ENABLED;
  UpdateInvalidationsEnabled();
}

void RemoteCommandsInvalidator::OnIncomingInvalidation(
    const syncer::ObjectIdInvalidationMap& invalidation_map) {
  DCHECK_EQ(STARTED, state_);
  DCHECK(thread_checker_.CalledOnValidThread());

  if (!invalidation_service_enabled_)
    LOG(WARNING) << "Unexpected invalidation received.";

  const syncer::SingleObjectInvalidationSet& list =
      invalidation_map.ForObject(object_id_);
  if (list.IsEmpty()) {
    NOTREACHED();
    return;
  }

  // Acknowledge all invalidations.
  for (const auto& it : list)
    it.Acknowledge();

  DoRemoteCommandsFetch();
}

std::string RemoteCommandsInvalidator::GetOwnerName() const {
  return "RemoteCommands";
}

void RemoteCommandsInvalidator::ReloadPolicyData(
    const enterprise_management::PolicyData* policy) {
  DCHECK(thread_checker_.CalledOnValidThread());

  if (state_ != STARTED)
    return;

  // Create the ObjectId based on the policy data.
  // If the policy does not specify an the ObjectId, then unregister.
  if (!policy || !policy->has_command_invalidation_source() ||
      !policy->has_command_invalidation_name()) {
    Unregister();
    return;
  }
  const invalidation::ObjectId object_id(policy->command_invalidation_source(),
                                         policy->command_invalidation_name());

  // If the policy object id in the policy data is different from the currently
  // registered object id, update the object registration.
  if (!is_registered_ || !(object_id == object_id_))
    Register(object_id);
}

void RemoteCommandsInvalidator::Register(
    const invalidation::ObjectId& object_id) {
  // Register this handler with the invalidation service if needed.
  if (!is_registered_) {
    OnInvalidatorStateChange(invalidation_service_->GetInvalidatorState());
    invalidation_service_->RegisterInvalidationHandler(this);
    is_registered_ = true;
  }

  object_id_ = object_id;
  UpdateInvalidationsEnabled();

  // Update registration with the invalidation service.
  syncer::ObjectIdSet ids;
  ids.insert(object_id);
  invalidation_service_->UpdateRegisteredInvalidationIds(this, ids);
}

void RemoteCommandsInvalidator::Unregister() {
  if (is_registered_) {
    invalidation_service_->UpdateRegisteredInvalidationIds(
        this, syncer::ObjectIdSet());
    invalidation_service_->UnregisterInvalidationHandler(this);
    is_registered_ = false;
    UpdateInvalidationsEnabled();
  }
}

void RemoteCommandsInvalidator::UpdateInvalidationsEnabled() {
  invalidations_enabled_ = invalidation_service_enabled_ && is_registered_;
}

}  // namespace policy
