// Copyright (c) 2011 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/extensions/extension_uninstall_dialog.h"

#import <Cocoa/Cocoa.h>

#include <string>

#import "base/mac/scoped_nsobject.h"
#import "base/strings/sys_string_conversions.h"
#include "base/strings/utf_string_conversions.h"
#include "chrome/grit/generated_resources.h"
#include "extensions/common/extension.h"
#include "skia/ext/skia_utils_mac.h"
#include "ui/base/l10n/l10n_util_mac.h"
#include "ui/gfx/image/image_skia_util_mac.h"

namespace {

// The Cocoa implementation of ExtensionUninstallDialog. This has a less
// complex life cycle than the Views and GTK implementations because the
// dialog blocks the page from navigating away and destroying the dialog,
// so there's no way for the dialog to outlive its delegate.
class ExtensionUninstallDialogCocoa
    : public extensions::ExtensionUninstallDialog {
 public:
  ExtensionUninstallDialogCocoa(Profile* profile, Delegate* delegate);
  ~ExtensionUninstallDialogCocoa() override;

 private:
  void Show() override;
};

ExtensionUninstallDialogCocoa::ExtensionUninstallDialogCocoa(
    Profile* profile,
    extensions::ExtensionUninstallDialog::Delegate* delegate)
    : extensions::ExtensionUninstallDialog(profile, delegate) {
}

ExtensionUninstallDialogCocoa::~ExtensionUninstallDialogCocoa() {}

void ExtensionUninstallDialogCocoa::Show() {
  NSAlert* alert = [[[NSAlert alloc] init] autorelease];

  NSButton* continueButton = [alert addButtonWithTitle:l10n_util::GetNSString(
      IDS_EXTENSION_PROMPT_UNINSTALL_BUTTON)];
  NSButton* cancelButton = [alert addButtonWithTitle:l10n_util::GetNSString(
      IDS_CANCEL)];
  // Default to accept when triggered via chrome://extensions page.
  if (triggering_extension_) {
    [continueButton setKeyEquivalent:@""];
    [cancelButton setKeyEquivalent:@"\r"];
  }

  [alert setMessageText:base::SysUTF8ToNSString(GetHeadingText())];
  [alert setAlertStyle:NSWarningAlertStyle];
  [alert setIcon:gfx::NSImageFromImageSkia(icon_)];

  base::scoped_nsobject<NSButton> reportAbuseCheckbox;
  if (ShouldShowReportAbuseCheckbox()) {
    reportAbuseCheckbox.reset([[NSButton alloc] initWithFrame:NSZeroRect]);
    [reportAbuseCheckbox setButtonType:NSSwitchButton];
    [reportAbuseCheckbox setTitle:l10n_util::GetNSString(
        IDS_EXTENSION_PROMPT_UNINSTALL_REPORT_ABUSE)];
    [reportAbuseCheckbox sizeToFit];
    [alert setAccessoryView:reportAbuseCheckbox];
  }

  if ([alert runModal] == NSAlertFirstButtonReturn) {
    bool report_abuse_checked =
        reportAbuseCheckbox.get() && [reportAbuseCheckbox state] == NSOnState;
    if (report_abuse_checked)
      HandleReportAbuse();
    OnDialogClosed(report_abuse_checked ?
        CLOSE_ACTION_UNINSTALL_AND_REPORT_ABUSE : CLOSE_ACTION_UNINSTALL);
    delegate_->ExtensionUninstallAccepted();
  } else {
    OnDialogClosed(CLOSE_ACTION_CANCELED);
    delegate_->ExtensionUninstallCanceled();
  }
}

}  // namespace

// static
extensions::ExtensionUninstallDialog*
extensions::ExtensionUninstallDialog::Create(Profile* profile,
                                             gfx::NativeWindow parent,
                                             Delegate* delegate) {
  return new ExtensionUninstallDialogCocoa(profile, delegate);
}
