// Copyright 2015 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

package org.chromium.chrome.browser.webapps;

import android.test.suitebuilder.annotation.MediumTest;

import org.chromium.base.test.util.Feature;
import org.chromium.chrome.browser.ssl.ConnectionSecurityHelperSecurityLevel;
import org.chromium.content_public.common.ScreenOrientationValues;

/**
 * Tests the logic in top controls visibility delegate in WebappActivity.
 */
public class WebappVisibilityTest extends WebappActivityTestBase {
    @MediumTest
    @Feature({"Webapps"})
    public void testShouldShowTopControls() {
        final String webappUrl = "http://originalwebsite.com";
        WebappInfo mockInfo = WebappInfo.create(
                WEBAPP_ID, webappUrl, null, null, ScreenOrientationValues.DEFAULT);
        getActivity().getWebappInfo().copy(mockInfo);

        // Show top controls for out-of-domain URLs.
        assertTrue(getActivity().shouldShowTopControls(
                "http://notoriginalwebsite.com", ConnectionSecurityHelperSecurityLevel.NONE));
        assertTrue(getActivity().shouldShowTopControls(
                "http://otherwebsite.com", ConnectionSecurityHelperSecurityLevel.NONE));

        // Do not show top controls for subdomains and private registries that are secure.
        assertFalse(getActivity().shouldShowTopControls(
                "http://sub.originalwebsite.com", ConnectionSecurityHelperSecurityLevel.NONE));
        assertFalse(getActivity().shouldShowTopControls(
                "http://thing.originalwebsite.com", ConnectionSecurityHelperSecurityLevel.NONE));
        assertFalse(getActivity().shouldShowTopControls(webappUrl,
                ConnectionSecurityHelperSecurityLevel.NONE));
        assertFalse(getActivity().shouldShowTopControls(
                webappUrl + "/things.html", ConnectionSecurityHelperSecurityLevel.NONE));
        assertFalse(getActivity().shouldShowTopControls(
                webappUrl + "/stuff.html", ConnectionSecurityHelperSecurityLevel.NONE));

        // Do not show top controls when URL is not available yet.
        assertFalse(getActivity().shouldShowTopControls("",
                ConnectionSecurityHelperSecurityLevel.NONE));

        // Show top controls for non secure URLs.
        assertTrue(getActivity().shouldShowTopControls(
                "http://sub.originalwebsite.com",
                ConnectionSecurityHelperSecurityLevel.SECURITY_WARNING));
        assertTrue(getActivity().shouldShowTopControls(
                "http://notoriginalwebsite.com",
                ConnectionSecurityHelperSecurityLevel.SECURITY_ERROR));
        assertTrue(getActivity().shouldShowTopControls(
                "http://otherwebsite.com",
                ConnectionSecurityHelperSecurityLevel.SECURITY_ERROR));
        assertTrue(getActivity().shouldShowTopControls(
                "http://thing.originalwebsite.com",
                ConnectionSecurityHelperSecurityLevel.SECURITY_ERROR));
        assertTrue(getActivity().shouldShowTopControls(
                webappUrl, ConnectionSecurityHelperSecurityLevel.SECURITY_WARNING));
        assertTrue(getActivity().shouldShowTopControls(
                webappUrl + "/things.html",
                ConnectionSecurityHelperSecurityLevel.SECURITY_WARNING));
        assertTrue(getActivity().shouldShowTopControls(
                webappUrl + "/stuff.html",
                ConnectionSecurityHelperSecurityLevel.SECURITY_WARNING));
    }
}
