// Copyright 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_PERFORMANCE_MONITOR_PROCESS_METRICS_HISTORY_H_
#define CHROME_BROWSER_PERFORMANCE_MONITOR_PROCESS_METRICS_HISTORY_H_

#include "base/memory/linked_ptr.h"
#include "base/process/process_handle.h"
#include "content/public/browser/child_process_data.h"
#include "content/public/common/process_type.h"

namespace base {
class ProcessMetrics;
}

namespace performance_monitor {
enum ProcessSubtypes {
  kProcessSubtypeUnknown,
  kProcessSubtypePPAPIFlash,
  kProcessSubtypeExtensionPersistent,
  kProcessSubtypeExtensionEvent
};

struct ProcessMetricsMetadata {
  base::ProcessHandle handle;
  int process_type;
  ProcessSubtypes process_subtype;

  ProcessMetricsMetadata()
      : handle(base::kNullProcessHandle),
        process_type(content::PROCESS_TYPE_UNKNOWN),
        process_subtype(kProcessSubtypeUnknown) {}
};

class ProcessMetricsHistory {
 public:
  ProcessMetricsHistory();
  ~ProcessMetricsHistory();

  // Configure this to monitor a specific process.
  void Initialize(const ProcessMetricsMetadata& process_data,
                  int initial_update_sequence);

  // Gather metrics for the process and accumulate with past data.
  void SampleMetrics();

  // Used to mark when this object was last updated, so we can cull
  // dead ones.
  void set_last_update_sequence(int new_update_sequence) {
    last_update_sequence_ = new_update_sequence;
  }

  int last_update_sequence() const { return last_update_sequence_; }

 private:
  void RunPerformanceTriggers();

  // May not be fully populated. e.g. no |id| and no |name| for browser and
  // renderer processes.
  ProcessMetricsMetadata process_data_;
  linked_ptr<base::ProcessMetrics> process_metrics_;
  int last_update_sequence_;

  double cpu_usage_;

  DISALLOW_ASSIGN(ProcessMetricsHistory);
};

}  // namespace performance_monitor

#endif  // CHROME_BROWSER_PERFORMANCE_MONITOR_PROCESS_METRICS_HISTORY_H_
