// Copyright 2015 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#import "chrome/browser/ui/cocoa/single_web_contents_dialog_manager_cocoa.h"

#import "chrome/browser/ui/cocoa/constrained_window/constrained_window_custom_sheet.h"
#import "chrome/browser/ui/cocoa/constrained_window/constrained_window_mac.h"
#import "chrome/browser/ui/cocoa/constrained_window/constrained_window_sheet_controller.h"
#include "components/web_modal/web_contents_modal_dialog_host.h"
#include "components/web_modal/web_contents_modal_dialog_manager.h"
#include "content/public/browser/web_contents.h"

SingleWebContentsDialogManagerCocoa::SingleWebContentsDialogManagerCocoa(
    ConstrainedWindowMac* client,
    id<ConstrainedWindowSheet> sheet,
    web_modal::SingleWebContentsDialogManagerDelegate* delegate)
    : client_(client),
      sheet_([sheet retain]),
      delegate_(delegate),
      host_(nullptr),
      shown_(false) {
  if (client)
    client->set_manager(this);
}

SingleWebContentsDialogManagerCocoa::~SingleWebContentsDialogManagerCocoa() {
}

void SingleWebContentsDialogManagerCocoa::Show() {
  if (shown_)
    return;

  DCHECK(host_);
  NSView* parent_view = host_->GetHostView();
  // Note that simply [parent_view window] for an inactive tab is nil. However,
  // the following should always be non-nil for all WebContents containers.
  NSWindow* parent_window =
      delegate_->GetWebContents()->GetTopLevelNativeWindow();

  shown_ = true;
  [[ConstrainedWindowSheetController controllerForParentWindow:parent_window]
      showSheet:sheet_ forParentView:parent_view];
}

void SingleWebContentsDialogManagerCocoa::Hide() {
}

void SingleWebContentsDialogManagerCocoa::Close() {
  [[ConstrainedWindowSheetController controllerForSheet:sheet_]
      closeSheet:sheet_];
  if (client_) {
    client_->set_manager(nullptr);
    client_->OnDialogClosing();  // |client_| might delete itself here.
    client_ = nullptr;
  }
  delegate_->WillClose(dialog());
}

void SingleWebContentsDialogManagerCocoa::Focus() {
}

void SingleWebContentsDialogManagerCocoa::Pulse() {
  [[ConstrainedWindowSheetController controllerForSheet:sheet_]
      pulseSheet:sheet_];
}

void SingleWebContentsDialogManagerCocoa::HostChanged(
    web_modal::WebContentsModalDialogHost* new_host) {
  // No need to observe the host. For Cocoa, the constrained window controller
  // will reposition the dialog when necessary. The host can also never change.
  // Tabs showing a dialog can not be dragged off a Cocoa browser window.
  // However, closing a tab with a dialog open will set the host back to null.
  DCHECK_NE(!!host_, !!new_host);
  host_ = new_host;
}

gfx::NativeWindow SingleWebContentsDialogManagerCocoa::dialog() {
  return [sheet_ sheetWindow];
}
